/*
 * Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Serial mouse user-level device driver.
 */
#include <stdio.h>
#include <errno.h>
#include <sgtty.h>
#include <fcntl.h>
#include <graphics.h>


/*
 * Configuration for mouse.
 */
#ifndef	MOUSE_TYPE
#define	MOUSE_TYPE	"ms"		/* default mouse type ("ms" or "pc") */
#endif

#ifndef	MOUSE_PORT
#define	MOUSE_PORT	"/dev/tty64"	/* default mouse tty port */
#endif

#define MAX_BYTES	128		/* number of bytes for buffer */


/*
 * States for the mouse.
 */ 
#define	IDLE			0		/* start of byte sequence */
#define	XSET			1		/* setting x delta */
#define	YSET			2		/* setting y delta */
#define	XADD			3		/* adjusting x delta */
#define	YADD			4		/* adjusting y delta */


/*
 * Values in the bytes returned by the mouse for the buttons.
 */
#define	PC_LEFT_BUTTON		4
#define PC_RIGHT_BUTTON		1
#define PC_MIDDLE_BUTTON	2

#define	MS_LEFT_BUTTON		2
#define MS_RIGHT_BUTTON		1


/*
 * Bit fields in the bytes sent by the mouse.
 */
#define TOP_FIVE_BITS		0xf8
#define BOTTOM_THREE_BITS	0x07
#define TOP_BIT			0x80
#define SIXTH_BIT		0x40
#define BOTTOM_TWO_BITS		0x03
#define THIRD_FOURTH_BITS	0x0c
#define BOTTOM_SIX_BITS  	0x3f


static int		(*parse)();	/* parse routine */
static int		fd;		/* file descriptor for mouse */
static int		state;		/* IDLE, XSET, ... */
static GR_BUTTON	buttons;	/* bits 1, 2 and 3 are buttons l, r and m */
static GR_BUTTON	availbuttons;	/* which buttons are available */
static GR_COORD		xd;		/* change in x */
static GR_COORD		yd;		/* change in y */

static int		left;		/* because the button values change */
static int		middle;		/* between mice, the buttons are */
static int		right;		/* redefined */

static char		*bp;		/* buffer pointer */
static int		nbytes;		/* number of bytes left */
static unsigned char	buffer[MAX_BYTES];	/* data bytes read */


/*
 * Parse routines.
 */
static	int	ParsePC();		/* routine to interpret PC mouse */
static	int	ParseMS();		/* routine to interpret MS mouse */


extern	char	*getenv();


/*
 * Open up the mouse device.
 * The mouse can either be a Microsoft serial mouse or a PC serial mouse.
 * Arguments are the type of mouse, and the tty device name the mouse is on.
 * Type is either "pc" or "ms".  If arguments are NULL, then the environment
 * variables are used for the values.  If there are none, then defaults are
 * used.  Returns zero if successful, or negative if unsuccessful.
 */
int
GdOpenMouse(type, port)
	char	*type;			/* type of mouse */
	char	*port;			/* tty port name */
{
	struct sgttyb	sgttyb;		/* terminal modes */

	if ((type == NULL) || (*type == '\0')) {
		type = getenv("MOUSE_TYPE");
		if (type == NULL)
			type = MOUSE_TYPE;
	}

	if ((port == NULL) || (*port == '\0')) {
		port = getenv("MOUSE_PORT");
		if (port == NULL)
			port = MOUSE_PORT;
	}

	if (strcmp(type, "pc") == 0) {			/* PC */
		left = PC_LEFT_BUTTON;
		middle = PC_MIDDLE_BUTTON;
		right = PC_RIGHT_BUTTON;
		parse = ParsePC;
	} else if (strcmp(type, "ms") == 0) {		/* Microsoft */
		left = MS_LEFT_BUTTON;
		right = MS_RIGHT_BUTTON;
		middle = 0;
		parse = ParseMS;
	} else
		return -1;

	fd = open(port, O_NONBLOCK);
	if (fd < 0)
		return -1;

	if (ioctl(fd, TIOCGETP, &sgttyb) == -1)
		return -1;

	sgttyb.sg_flags |= RAW;
	sgttyb.sg_flags &= ~(EVENP | ODDP | ECHO | XTABS | CRMOD);

	if (ioctl(fd, TIOCSETP, &sgttyb) == -1)
		return -1;

	if (ioctl(fd, TIOCFLUSH, 0) < 0)
		return -1;

	availbuttons = left | middle | right;
	state = IDLE;
	nbytes = 0;
	buttons = 0;
	xd = 0;
	yd = 0;

	return 0;
}


/*
 * Return which buttons are available.
 */
void
GdGetButtonInfo(buttons)
	GR_BUTTON	*buttons;		/* implemented buttons */
{
	*buttons = availbuttons;
}


/*
 * Close the mouse device.
 */
void
GdCloseMouse()
{
	if (fd > 0)
		close(fd);
	fd = 0;
}


/*
 * Attempt to read bytes from the mouse and interpret them.
 * Returns -1 on error, 0 if either no bytes were read or not enough
 * was read for a complete state, or 1 if the new state was read.
 * When a new state is read, the current buttons and x and y deltas
 * are returned.  This routine does not block.
 */
int
GdReadMouse(xdptr, ydptr, bptr)
	GR_COORD	*xdptr;			/* returned X delta */
	GR_COORD	*ydptr;			/* returned Y delta */
	GR_BUTTON	*bptr;			/* returned buttons */
{
	*xdptr = 0;
	*ydptr = 0;
	*xdptr = buttons;

	/*
	 * If there are no more bytes left, then read some more,
	 * waiting for them to arrive.  On a signal or a non-blocking
	 * error, return saying there is no new state available yet.
	 */
	if (nbytes <= 0) {
		bp = buffer;
		nbytes = read(fd, bp, MAX_BYTES);
		if (nbytes < 0) {
			if ((errno == EINTR) || (errno == EAGAIN))
				return 0;
			return -1;
		}
	}

	/*
	 * Loop over all the bytes read in the buffer, parsing them.
	 * When a complete state has been read, return the results,
	 * leaving further bytes in the buffer for later calls.
	 */
	while (nbytes-- > 0) {
		if ((*parse)((int) *bp++)) {
			*xdptr = xd;
			*ydptr = yd;
			*bptr = buttons;
			return 1;
		}
	}

	return 0;
}


/*
 * Input routine for PC mouse.
 * Returns nonzero when a new mouse state has been completed.
 */
static int
ParsePC(byte)
	int	byte;			/* current byte read */
{
	int	sign;			/* sign of movement */

	switch (state) {
		case IDLE:
			if ((byte & TOP_FIVE_BITS) == TOP_BIT) {
				buttons = ~byte & BOTTOM_THREE_BITS;
				state = XSET;
			}
			break;

		case XSET:
			sign = 1;
			if (byte > 127) {
				byte = 256 - byte;
				sign = -1;
			}
			xd = byte * sign;
			state = YSET;
			break;

		case YSET:
			sign = 1;
			if (byte > 127) {
				byte = 256 - byte;
				sign = -1;
			}
			yd = -byte * sign;
			state = XADD;
			break;

		case XADD:
			sign = 1;
			if (byte > 127) {
				byte = 256 - byte;
				sign = -1;
			}
			xd += byte;
			state = YADD;
			break;

		case YADD:
			sign = 1;
			if (byte > 127) {
				byte = 256 - byte;
				sign = -1;
			}
			yd -= byte;
			state = IDLE;
			return 1;
	}
	return 0;
}


/*
 * Input routine for Microsoft mouse.
 * Returns nonzero when a new mouse state has been completed.
 */
static int
ParseMS(byte)
	int	byte;			/* current byte from buffer */
{
	switch (state) {
		case IDLE:
			if (byte & SIXTH_BIT) {
				buttons = (byte >> 4) & BOTTOM_TWO_BITS;
				yd = ((byte & THIRD_FOURTH_BITS) << 4);
				xd = ((byte & BOTTOM_TWO_BITS) << 6);
				state = XADD;
			}
			break;

		case XADD:
			xd |= (byte & BOTTOM_SIX_BITS);
			state = YADD;
			break;

		case YADD:
			yd |= (byte & BOTTOM_SIX_BITS);
			state = IDLE;
			if (xd > 127)
				xd -= 256;
			if (yd > 127)
				yd -= 256;
			return 1;
	}
	return 0;
}

/* END CODE */
