/*
 * Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Mouse server routines.
 * Reads data from mouse driver and tracks real position on the screen.
 */
#include <stdio.h>
#include "graph_serv.h"


#define	SCALE	3		/* default scaling factor for acceleration */
#define	THRESH	5		/* default threshhold for acceleration */


static GR_COORD		xpos;		/* current x position of mouse */
static GR_COORD		ypos;		/* current y position of mouse */
static GR_COORD		minx;		/* minimum allowed x position */
static GR_COORD		maxx;		/* maximum allowed x position */
static GR_COORD		miny;		/* minimum allowed y position */
static GR_COORD		maxy;		/* maximum allowed y position */
static GR_SIZE		scale;		/* acceleration scale factor */
static GR_SIZE		thresh;		/* acceleration threshhold */
static GR_BUTTON	buttons;	/* current state of buttons */
static GR_BOOL		changed;	/* mouse state has changed */


extern	int	GdOpenMouse();
extern	int	GdReadMouse();
extern	void	GdGetButtonInfo();
extern	void	GdCloseMouse();


/*
 * Initialize the mouse.
 * This sets its position to (0, 0) with no boundaries and no buttons pressed.
 * Returns nonzero on error.
 */
GsOpenMouse()
{
	if (GdOpenMouse(NULL, NULL) < 0)
		return -1;
	thresh = THRESH;
	scale = SCALE;
	buttons = 0;
	xpos = 0;
	ypos = 0;
	minx = GR_COORD_MIN;
	miny = GR_COORD_MIN;
	maxx = GR_COORD_MAX;
	maxy = GR_COORD_MAX;
	changed = GR_TRUE;
	return 0;
}


/*
 * Return which mouse buttons are implemented.
 */
void
GsGetButtonInfo(buttons)
	GR_BUTTON	*buttons;	/* buttons which are implemented */
{
	GdGetButtonInfo(buttons);
}


/*
 * Terminate the use of the mouse.
 */
void
GsCloseMouse()
{
	GdCloseMouse();
}


/*
 * Restrict the coordinates of the mouse to the specified coordinates.
 */
void
GsRestrictMouse(newminx, newminy, newmaxx, newmaxy)
	GR_COORD	newminx;	/* new minimum x position */
	GR_COORD	newminy;	/* new minimum y position */
	GR_COORD	newmaxx;	/* new maximum x position */
	GR_COORD	newmaxy;	/* new maximum y position */
{
	minx = newminx;
	miny = newminy;
	maxx = newmaxx;
	maxy = newmaxy;
	GsMoveMouse(xpos, ypos);
}


/*
 * Set the acceleration threshhold and scale factors.
 * Acceleration makes the cursor move further for faster movements.
 * Basically, at mouse speeds above the threshold, the excess distance
 * moved is multiplied by the scale factor.  For example, with a threshhold
 * of 5 and a scale of 3, the following gives examples of the original and
 * modified mouse movements:
 *	input:		0	4	5	6	9	20
 *	output:		0	4	5	8	17	50
 */
void
GsSetAccelMouse(newthresh, newscale)
	GR_SIZE		newthresh;	/* new acceleration threshhold */
	GR_SIZE		newscale;	/* new acceleration scale factor */
{
	if (newthresh < 0)
		newthresh = 0;
	if (newscale < 0)
		newscale = 0;
	thresh = newthresh;
	scale = newscale;
}


/*
 * Move the mouse to the specified coordinates.
 * The location is limited by the current mouse coordinate restrictions.
 */
void
GsMoveMouse(newx, newy)
	GR_COORD	newx;		/* desired new x position of mouse */
	GR_COORD	newy;		/* desired new y position of mouse */
{
	if (newx < minx)
		newx = minx;
	if (newx > maxx)
		newx = maxx;
	if (newy < miny)
		newy = miny;
	if (newy > maxy)
		newy = maxy;
	if ((newx != xpos) || (newy != ypos))
		changed = GR_TRUE;
	xpos = newx;
	ypos = newy;
}


/*
 * Read the current location and button states of the mouse.
 * Returns -1 on read error, 0 if nothing new has happened, and 1 if a
 * new position was read.  The current mouse position and buttons are
 * returned even if the position hasn't changed.  This routine does
 * not block.
 */
int
GsReadMouse(xp, yp, bp)
	GR_COORD	*xp;		/* returned x position */
	GR_COORD	*yp;		/* returned y position */
	GR_BUTTON	*bp;		/* returned button state */
{
	GR_COORD	deltax;		/* change in x position */
	GR_COORD	deltay;		/* change in y position */
	GR_BUTTON	newbuttons;	/* new button state */
	int		sign;		/* sign of change */
	int		status;		/* status of reading mouse */

	*xp = xpos;
	*yp = ypos;
	*bp = buttons;

	if (changed) {
		changed = GR_FALSE;
		return 1;
	}

	status = GdReadMouse(&deltax, &deltay, &newbuttons);
	if (status < 0)
		return -1;

	if (status == 0)
		return 0;

	if (buttons != newbuttons) {
		changed = GR_TRUE;
		buttons = newbuttons;
	}

	sign = 1;
	if (deltax < 0) {
		sign = -1;
		deltax = -deltax;
	}
	if (deltax > thresh)
		deltax = thresh + (deltax - thresh) * scale;
	deltax *= sign;

	sign = 1;
	if (deltay < 0) {
		sign = -1;
		deltay = -deltay;
	}
	if (deltay > thresh)
		deltay = thresh + (deltay - thresh) * scale;
	deltay *= sign;

	GsMoveMouse(xpos + deltax, ypos + deltay);

	if (!changed)
		return 0;

	changed = GR_FALSE;
	*xp = xpos;
	*yp = ypos;
	*bp = buttons;
	return 1;
}

/* END CODE */
