/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Low level graphics mode for ega and vga.
 * Since these are "device level" routines, there is no need to check
 * clipping since the higher level routines should have already done this.
 * However, checking for values outside the screen memory is reasonable
 * for safety.
 */

#include "kernel.h"
#include "protect.h"
#include "graph_dev.h"


/* Make this 1 for VGA boards, or 0 for EGA boards.
 * This is also defined in ega.x.
 */
#define	VGA	1


#if	VGA
#define	ROWS	480		/* rows */
#else
#define	ROWS	350		/* rows */
#endif

#define	COLS	640		/* columns */
#define	COLORS	16		/* number of colors */
#define	BLACK	0		/* black */
#define	WHITE	15		/* white */

#define	DONE	0
#define	IN	1
#define	OUT	2

#define	ROM_SCAN_LINES	14	/* number of scan lines in fonts in ROM */
#define	RAM_SCAN_LINES	32	/* number of scan lines in fonts in RAM */
#define	FONT_CHARS	256	/* number of characters in font tables */
#define	CHAR_WIDTH	8	/* number of pixels for character width */
#define	EGA_BASE	0xa0000	/* location of enhanced display memory */



#define	PALREG	0x3c0
#define	SEQREG	0x3c4
#define	SEQVAL	0x3c5
#define	GRREG	0x3ce
#define	GRVAL	0x3cf
#define	ATTRREG	0x3da
#define	CRTCREG	0x3d4
#define	CRTCVAL	0x3d5

#define	GENREG1	0x3c2
#define	GENREG2	0x3cc
#define	GENREG3	0x3ca

#define	DATA_ROTATE	3	/* register number for data rotate */


FORWARD int ega_init();
FORWARD void ega_term();
FORWARD void ega_setmode();
FORWARD void ega_drawrow();
FORWARD void ega_drawcol();
FORWARD void ega_drawtext();
FORWARD void ega_sizetext();
FORWARD void ega_getscreeninfo();
FORWARD void ega_getfontinfo();

extern void ega_drawpoint();
extern void ega_drawline();
extern void ega_setgraphics();
extern void ega_getcharbits();
extern GR_COLOR ega_readpoint();


GR_DEV gr_dev = {
  ega_init, ega_term, ega_setmode, ega_drawpoint, ega_drawline,
  ega_drawrow, ega_drawcol, gen_fillrect, gen_drawellipse,
  gen_fillellipse, ega_drawtext, ega_sizetext, gen_drawarea8,
  gen_drawbitmap, ega_readpoint, gen_setcursor, gen_movecursor,
  gen_checkcursor, gen_fixcursor, gen_copyarea, ega_getcharbits,
  ega_getscreeninfo, ega_getfontinfo, ROWS, COLS, COLORS, BLACK, WHITE
};

typedef struct {
  int action;
  int port1;
  int data1;
  int port2;
  int data2;
} REGIO;


#if	VGA

/* VGA 640x480 16-color graphics (BIOS mode 0x12).
 */
PRIVATE REGIO graphics_on[] = {
  /* Reset attr F/F */
  IN, ATTRREG, 0, 0, 0,

  /* Disable palette */
  OUT, PALREG, 0, 0, 0,

  /* Reset sequencer regs */
  OUT, SEQREG, 0, SEQVAL, 0,
  OUT, SEQREG, 1, SEQVAL, 1,
  OUT, SEQREG, 2, SEQVAL, 0x0f,
  OUT, SEQREG, 3, SEQVAL, 0,
  OUT, SEQREG, 4, SEQVAL, 6,

  /* Misc out reg */
  OUT, GENREG1, 0xe3, 0, 0,

  /* Sequencer enable */
  OUT, SEQREG, 0, SEQVAL, 0x03,

  /* Unprotect crtc regs 0-7 */
  OUT, CRTCREG, 0x11, CRTCVAL, 0,

  /* Crtc */
  OUT, CRTCREG, 0, CRTCVAL, 0x5f,
  OUT, CRTCREG, 1, CRTCVAL, 0x4f,
  OUT, CRTCREG, 2, CRTCVAL, 0x50,
  OUT, CRTCREG, 3, CRTCVAL, 0x82,
  OUT, CRTCREG, 4, CRTCVAL, 0x54,
  OUT, CRTCREG, 5, CRTCVAL, 0x80,
  OUT, CRTCREG, 6, CRTCVAL, 0x0b,
  OUT, CRTCREG, 7, CRTCVAL, 0x3e,
  OUT, CRTCREG, 8, CRTCVAL, 0x00,
  OUT, CRTCREG, 9, CRTCVAL, 0x40,
  OUT, CRTCREG, 10, CRTCVAL, 0x00,
  OUT, CRTCREG, 11, CRTCVAL, 0x00,
  OUT, CRTCREG, 12, CRTCVAL, 0x00,
  OUT, CRTCREG, 13, CRTCVAL, 0x00,
  OUT, CRTCREG, 14, CRTCVAL, 0x00,
  OUT, CRTCREG, 15, CRTCVAL, 0x59,
  OUT, CRTCREG, 16, CRTCVAL, 0xea,
  OUT, CRTCREG, 17, CRTCVAL, 0x8c,
  OUT, CRTCREG, 18, CRTCVAL, 0xdf,
  OUT, CRTCREG, 19, CRTCVAL, 0x28,
  OUT, CRTCREG, 20, CRTCVAL, 0x00,
  OUT, CRTCREG, 21, CRTCVAL, 0xe7,
  OUT, CRTCREG, 22, CRTCVAL, 0x04,
  OUT, CRTCREG, 23, CRTCVAL, 0xe3,
  OUT, CRTCREG, 24, CRTCVAL, 0xff,

  /* Graphics controller */
  OUT, GENREG2, 0x00, 0, 0,
  OUT, GENREG3, 0x01, 0, 0,
  OUT, GRREG, 0, GRVAL, 0x00,
  OUT, GRREG, 1, GRVAL, 0x00,
  OUT, GRREG, 2, GRVAL, 0x00,
  OUT, GRREG, 3, GRVAL, 0x00,
  OUT, GRREG, 4, GRVAL, 0x00,
  OUT, GRREG, 5, GRVAL, 0x00,
  OUT, GRREG, 6, GRVAL, 0x05,
  OUT, GRREG, 7, GRVAL, 0x0f,
  OUT, GRREG, 8, GRVAL, 0xff,

  /* Reset attribute flip/flop */
  IN, ATTRREG, 0, 0, 0,

  /* Palette */
  OUT, PALREG, 0, PALREG, 0x00,
  OUT, PALREG, 1, PALREG, 0x01,
  OUT, PALREG, 2, PALREG, 0x02,
  OUT, PALREG, 3, PALREG, 0x03,
  OUT, PALREG, 4, PALREG, 0x04,
  OUT, PALREG, 5, PALREG, 0x05,
  OUT, PALREG, 6, PALREG, 0x14,
  OUT, PALREG, 7, PALREG, 0x07,
  OUT, PALREG, 8, PALREG, 0x38,
  OUT, PALREG, 9, PALREG, 0x39,
  OUT, PALREG, 10, PALREG, 0x3a,
  OUT, PALREG, 11, PALREG, 0x3b,
  OUT, PALREG, 12, PALREG, 0x3c,
  OUT, PALREG, 13, PALREG, 0x3d,
  OUT, PALREG, 14, PALREG, 0x3e,
  OUT, PALREG, 15, PALREG, 0x3f,
  OUT, PALREG, 16, PALREG, 0x01,
  OUT, PALREG, 17, PALREG, 0x00,
  OUT, PALREG, 18, PALREG, 0x0f,
  OUT, PALREG, 19, PALREG, 0x00,

  /* Enable palette */
  OUT, PALREG, 0x20, 0, 0,

  /* End of table */
  DONE, 0, 0, 0, 0
};


/* VGA 80x25 text (BIOS mode 3).
 */
PRIVATE REGIO graph_off[] = {
  /* Reset attr F/F */
  IN, ATTRREG, 0, 0, 0,

  /* Disable palette */
  OUT, PALREG, 0, 0, 0,

  /* Reset sequencer regs */
  OUT, SEQREG, 0, SEQVAL, 1,
  OUT, SEQREG, 1, SEQVAL, 1,
  OUT, SEQREG, 2, SEQVAL, 3,
  OUT, SEQREG, 3, SEQVAL, 0,
  OUT, SEQREG, 4, SEQVAL, 2,

  /* Misc out reg */
  OUT, GENREG1, 0x63, 0, 0,

  /* Sequencer enable */
  OUT, SEQREG, 0, SEQVAL, 3,

  /* Unprotect crtc regs 0-7 */
  OUT, CRTCREG, 0x11, CRTCVAL, 0,

  /* Crtc */
  OUT, CRTCREG, 0, CRTCVAL, 0x5f,	/* horiz total */
  OUT, CRTCREG, 1, CRTCVAL, 0x4f,	/* horiz end */
  OUT, CRTCREG, 2, CRTCVAL, 0x50,	/* horiz blank */
  OUT, CRTCREG, 3, CRTCVAL, 0x82,	/* end blank */
  OUT, CRTCREG, 4, CRTCVAL, 0x55,	/* horiz retrace */
  OUT, CRTCREG, 5, CRTCVAL, 0x81,	/* end retrace */
  OUT, CRTCREG, 6, CRTCVAL, 0xbf,	/* vert total */
  OUT, CRTCREG, 7, CRTCVAL, 0x1f,	/* overflows */
  OUT, CRTCREG, 8, CRTCVAL, 0x00,	/* row scan */
  OUT, CRTCREG, 9, CRTCVAL, 0x4f,	/* max scan line */
  OUT, CRTCREG, 10, CRTCVAL, 0x00,	/* cursor start */
  OUT, CRTCREG, 11, CRTCVAL, 0x0f,	/* cursor end */
  OUT, CRTCREG, 12, CRTCVAL, 0x0e,	/* start high addr */
  OUT, CRTCREG, 13, CRTCVAL, 0xb0,	/* low addr */
  OUT, CRTCREG, 14, CRTCVAL, 0x16,	/* cursor high */
  OUT, CRTCREG, 15, CRTCVAL, 0x30,	/* cursor low */
  OUT, CRTCREG, 16, CRTCVAL, 0x9c,	/* vert retrace */
  OUT, CRTCREG, 17, CRTCVAL, 0x8e,	/* retrace end */
  OUT, CRTCREG, 18, CRTCVAL, 0x8f,	/* vert end */
  OUT, CRTCREG, 19, CRTCVAL, 0x28,	/* offset */
  OUT, CRTCREG, 20, CRTCVAL, 0x1f,	/* underline */
  OUT, CRTCREG, 21, CRTCVAL, 0x96,	/* vert blank */
  OUT, CRTCREG, 22, CRTCVAL, 0xb9,	/* end blank */
  OUT, CRTCREG, 23, CRTCVAL, 0xa3,	/* crt mode */
  OUT, CRTCREG, 24, CRTCVAL, 0xff,	/* line compare */

  /* Graphics controller */
  OUT, GENREG2, 0x00, 0, 0,
  OUT, GENREG3, 0x01, 0, 0,
  OUT, GRREG, 0, GRVAL, 0x00,
  OUT, GRREG, 1, GRVAL, 0x00,
  OUT, GRREG, 2, GRVAL, 0x00,
  OUT, GRREG, 3, GRVAL, 0x00,
  OUT, GRREG, 4, GRVAL, 0x00,
  OUT, GRREG, 5, GRVAL, 0x10,
  OUT, GRREG, 6, GRVAL, 0x0e,
  OUT, GRREG, 7, GRVAL, 0x00,
  OUT, GRREG, 8, GRVAL, 0xff,

  /* Reset attribute flip/flop */
  IN, ATTRREG, 0, 0, 0,

  /* Palette */
  OUT, PALREG, 0, PALREG, 0x00,
  OUT, PALREG, 1, PALREG, 0x01,
  OUT, PALREG, 2, PALREG, 0x02,
  OUT, PALREG, 3, PALREG, 0x03,
  OUT, PALREG, 4, PALREG, 0x04,
  OUT, PALREG, 5, PALREG, 0x05,
  OUT, PALREG, 6, PALREG, 0x06,
  OUT, PALREG, 7, PALREG, 0x07,
  OUT, PALREG, 8, PALREG, 0x10,
  OUT, PALREG, 9, PALREG, 0x11,
  OUT, PALREG, 10, PALREG, 0x12,
  OUT, PALREG, 11, PALREG, 0x13,
  OUT, PALREG, 12, PALREG, 0x14,
  OUT, PALREG, 13, PALREG, 0x15,
  OUT, PALREG, 14, PALREG, 0x16,
  OUT, PALREG, 15, PALREG, 0x17,
  OUT, PALREG, 16, PALREG, 0x08,
  OUT, PALREG, 17, PALREG, 0x00,
  OUT, PALREG, 18, PALREG, 0x0f,
  OUT, PALREG, 19, PALREG, 0x00,

  /* Enable palette */
  OUT, PALREG, 0x20, 0, 0,

  /* End of table */
  DONE, 0, 0, 0, 0
};

#else

/* EGA 640x350 16-color graphics (BIOS mode 0x10).
 */
PRIVATE REGIO graphics_on[] = {
  /* Reset attr F/F */
  IN, ATTRREG, 0, 0, 0,

  /* Disable palette */
  OUT, PALREG, 0, 0, 0,

  /* Reset sequencer regs */
  OUT, SEQREG, 0, SEQVAL, 0,
  OUT, SEQREG, 1, SEQVAL, 1,
  OUT, SEQREG, 2, SEQVAL, 0x0f,
  OUT, SEQREG, 3, SEQVAL, 0,
  OUT, SEQREG, 4, SEQVAL, 6,

  /* Misc out reg */
  OUT, GENREG1, 0xa7, 0, 0,

  /* Sequencer enable */
  OUT, SEQREG, 0, SEQVAL, 0x03,

  /* Unprotect crtc regs 0-7 */
  OUT, CRTCREG, 0x11, CRTCVAL, 0,

  /* Crtc */
  OUT, CRTCREG, 0, CRTCVAL, 0x5b,
  OUT, CRTCREG, 1, CRTCVAL, 0x4f,
  OUT, CRTCREG, 2, CRTCVAL, 0x53,
  OUT, CRTCREG, 3, CRTCVAL, 0x37,
  OUT, CRTCREG, 4, CRTCVAL, 0x52,
  OUT, CRTCREG, 5, CRTCVAL, 0x00,
  OUT, CRTCREG, 6, CRTCVAL, 0x6c,
  OUT, CRTCREG, 7, CRTCVAL, 0x1f,
  OUT, CRTCREG, 8, CRTCVAL, 0x00,
  OUT, CRTCREG, 9, CRTCVAL, 0x00,
  OUT, CRTCREG, 10, CRTCVAL, 0x00,
  OUT, CRTCREG, 11, CRTCVAL, 0x00,
  OUT, CRTCREG, 12, CRTCVAL, 0x00,
  OUT, CRTCREG, 13, CRTCVAL, 0x00,
  OUT, CRTCREG, 14, CRTCVAL, 0x00,
  OUT, CRTCREG, 15, CRTCVAL, 0x00,
  OUT, CRTCREG, 16, CRTCVAL, 0x5e,
  OUT, CRTCREG, 17, CRTCVAL, 0x2b,
  OUT, CRTCREG, 18, CRTCVAL, 0x5d,
  OUT, CRTCREG, 19, CRTCVAL, 0x28,
  OUT, CRTCREG, 20, CRTCVAL, 0x0f,
  OUT, CRTCREG, 21, CRTCVAL, 0x5f,
  OUT, CRTCREG, 22, CRTCVAL, 0x0a,
  OUT, CRTCREG, 23, CRTCVAL, 0xe3,
  OUT, CRTCREG, 24, CRTCVAL, 0xff,

  /* Graphics controller */
  OUT, GENREG2, 0x00, 0, 0,
  OUT, GENREG3, 0x01, 0, 0,
  OUT, GRREG, 0, GRVAL, 0x00,
  OUT, GRREG, 1, GRVAL, 0x00,
  OUT, GRREG, 2, GRVAL, 0x00,
  OUT, GRREG, 3, GRVAL, 0x00,
  OUT, GRREG, 4, GRVAL, 0x00,
  OUT, GRREG, 5, GRVAL, 0x00,
  OUT, GRREG, 6, GRVAL, 0x05,
  OUT, GRREG, 7, GRVAL, 0x0f,
  OUT, GRREG, 8, GRVAL, 0xff,

  /* Reset attribute flip/flop */
  IN, ATTRREG, 0, 0, 0,

  /* Palette */
  OUT, PALREG, 0, PALREG, 0x00,
  OUT, PALREG, 1, PALREG, 0x01,
  OUT, PALREG, 2, PALREG, 0x02,
  OUT, PALREG, 3, PALREG, 0x03,
  OUT, PALREG, 4, PALREG, 0x04,
  OUT, PALREG, 5, PALREG, 0x05,
  OUT, PALREG, 6, PALREG, 0x06,
  OUT, PALREG, 7, PALREG, 0x07,
  OUT, PALREG, 8, PALREG, 0x38,
  OUT, PALREG, 9, PALREG, 0x39,
  OUT, PALREG, 10, PALREG, 0x3a,
  OUT, PALREG, 11, PALREG, 0x3b,
  OUT, PALREG, 12, PALREG, 0x3c,
  OUT, PALREG, 13, PALREG, 0x3d,
  OUT, PALREG, 14, PALREG, 0x3e,
  OUT, PALREG, 15, PALREG, 0x3f,
  OUT, PALREG, 16, PALREG, 0x01,
  OUT, PALREG, 17, PALREG, 0x00,
  OUT, PALREG, 18, PALREG, 0x0f,
  OUT, PALREG, 19, PALREG, 0x00,

  /* Enable palette */
  OUT, PALREG, 0x20, 0, 0,

  /* End of table */
  DONE, 0, 0, 0, 0
};


/* EGA 80x25 text (BIOS mode 3).
 */
PRIVATE REGIO graph_off[] = {
  /* Reset attr F/F */
  IN, ATTRREG, 0, 0, 0,

  /* Disable palette */
  OUT, PALREG, 0, 0, 0,

  /* Reset sequencer regs */
  OUT, SEQREG, 0, SEQVAL, 1,
  OUT, SEQREG, 1, SEQVAL, 1,
  OUT, SEQREG, 2, SEQVAL, 3,
  OUT, SEQREG, 3, SEQVAL, 0,
  OUT, SEQREG, 4, SEQVAL, 3,

  /* Misc out reg */
  OUT, GENREG1, 0xa7, 0, 0,

  /* Sequencer enable */
  OUT, SEQREG, 0, SEQVAL, 3,

  /* Crtc */
  OUT, CRTCREG, 0, CRTCVAL, 0x5b,	/* horiz total */
  OUT, CRTCREG, 1, CRTCVAL, 0x4f,	/* horiz end */
  OUT, CRTCREG, 2, CRTCVAL, 0x53,	/* horiz blank */
  OUT, CRTCREG, 3, CRTCVAL, 0x37,	/* end blank */
  OUT, CRTCREG, 4, CRTCVAL, 0x51,	/* horiz retrace */
  OUT, CRTCREG, 5, CRTCVAL, 0x5b,	/* end retrace */
  OUT, CRTCREG, 6, CRTCVAL, 0x6c,	/* vert total */
  OUT, CRTCREG, 7, CRTCVAL, 0x1f,	/* overflows */
  OUT, CRTCREG, 8, CRTCVAL, 0x00,	/* row scan */
  OUT, CRTCREG, 9, CRTCVAL, 0x0d,	/* max scan line */
  OUT, CRTCREG, 10, CRTCVAL, 0x00,	/* cursor start */
  OUT, CRTCREG, 11, CRTCVAL, 0x0f,	/* cursor end */
  OUT, CRTCREG, 12, CRTCVAL, 0x00,	/* start high addr */
  OUT, CRTCREG, 13, CRTCVAL, 0x00,	/* low addr */
  OUT, CRTCREG, 14, CRTCVAL, 0x00,	/* cursor high */
  OUT, CRTCREG, 15, CRTCVAL, 0x00,	/* cursor low */
  OUT, CRTCREG, 16, CRTCVAL, 0x5e,	/* vert retrace */
  OUT, CRTCREG, 17, CRTCVAL, 0x2b,	/* retrace end */
  OUT, CRTCREG, 18, CRTCVAL, 0x5d,	/* vert end */
  OUT, CRTCREG, 19, CRTCVAL, 0x28,	/* offset */
  OUT, CRTCREG, 20, CRTCVAL, 0x0f,	/* underline */
  OUT, CRTCREG, 21, CRTCVAL, 0x5e,	/* vert blank */
  OUT, CRTCREG, 22, CRTCVAL, 0x0a,	/* end blank */
  OUT, CRTCREG, 23, CRTCVAL, 0xa3,	/* crt mode */
  OUT, CRTCREG, 24, CRTCVAL, 0xff,	/* line compare */

  /* Graphics controller */
  OUT, GENREG2, 0x00, 0, 0,
  OUT, GENREG3, 0x01, 0, 0,
  OUT, GRREG, 0, GRVAL, 0x00,
  OUT, GRREG, 1, GRVAL, 0x00,
  OUT, GRREG, 2, GRVAL, 0x00,
  OUT, GRREG, 3, GRVAL, 0x00,
  OUT, GRREG, 4, GRVAL, 0x00,
  OUT, GRREG, 5, GRVAL, 0x10,
  OUT, GRREG, 6, GRVAL, 0x0e,
  OUT, GRREG, 7, GRVAL, 0x00,
  OUT, GRREG, 8, GRVAL, 0xff,

  /* Reset attribute flip/flop */
  IN, ATTRREG, 0, 0, 0,

  /* Palette */
  OUT, PALREG, 0, PALREG, 0x00,
  OUT, PALREG, 1, PALREG, 0x01,
  OUT, PALREG, 2, PALREG, 0x02,
  OUT, PALREG, 3, PALREG, 0x03,
  OUT, PALREG, 4, PALREG, 0x04,
  OUT, PALREG, 5, PALREG, 0x05,
  OUT, PALREG, 6, PALREG, 0x14,
  OUT, PALREG, 7, PALREG, 0x07,
  OUT, PALREG, 8, PALREG, 0x38,
  OUT, PALREG, 9, PALREG, 0x39,
  OUT, PALREG, 10, PALREG, 0x3a,
  OUT, PALREG, 11, PALREG, 0x3b,
  OUT, PALREG, 12, PALREG, 0x3c,
  OUT, PALREG, 13, PALREG, 0x3d,
  OUT, PALREG, 14, PALREG, 0x3e,
  OUT, PALREG, 15, PALREG, 0x3f,
  OUT, PALREG, 16, PALREG, 0x08,
  OUT, PALREG, 17, PALREG, 0x00,
  OUT, PALREG, 18, PALREG, 0x0f,
  OUT, PALREG, 19, PALREG, 0x00,

  /* Enable palette */
  OUT, PALREG, 0x20, 0, 0,

  /* End of table */
  DONE, 0, 0, 0, 0
};

#endif


/* Values for the data rotate register to implement drawing modes. */
PRIVATE unsigned char mode_table[GR_MAX_MODE + 1] = {
  0x00, 0x18, 0x10, 0x08
};


/*===========================================================================*
 *				writeregs				     *
 *===========================================================================*/
PRIVATE void writeregs(rp)
register REGIO *rp;
{
/* Set the graphics registers as indicated by the given table */
  for (; rp->action != DONE; rp++) {
	switch (rp->action) {
	    case IN:	in_byte(rp->port1);	break;
	    case OUT:
		out_byte(rp->port1, rp->data1);
		if (rp->port2) out_byte(rp->port2, rp->data2);
		break;
	}
  }
}


/*===========================================================================*
 *				out_word				     *
 *===========================================================================*/
PRIVATE void out_word(p, d)
unsigned int p;
unsigned int d;
{
/* Output a word to an I/O port. */
  out_byte(p, d & 0xff);
  out_byte(p + 1, (d >> 8) & 0xff);
}


/*===========================================================================*
 *				ega_init				     *
 *===========================================================================*/
PRIVATE int ega_init(rows, cols, colors)
GR_SIZE rows;
GR_SIZE cols;
long colors;
{
/* Setup for drawing to the ega.
 * Returns nonzero if the parameters are illegal.
 */
  if ((rows && (rows != ROWS)) || (cols && (cols != COLS))
      || (colors && (colors != COLORS)))
	return -1;
  writeregs(graphics_on);
  for (rows = 0; rows < ROWS; rows++) ega_drawrow(0, COLS - 1, rows, BLACK);
  return 0;
}


/*===========================================================================*
 *				ega_term				     *
 *===========================================================================*/
PRIVATE void ega_term()
{
/* Terminate graphics mode for the EGA.
 * This resets back to normal text mode.
 */
  long srcoffset;
  long destoffset;
  int data;
  int ch;
  int row;

  ega_setmode(GR_MODE_SET);
  for (row = 0; row < ROWS; row++) ega_drawrow(0, COLS - 1, row, BLACK);

  /* Copy character table from ROM back into bit plane 2 before turning
   * off graphics.
   */
  out_word(SEQREG, 0x0100);	/* syn reset */
  out_word(SEQREG, 0x0402);	/* cpu writes only to map 2 */
  out_word(SEQREG, 0x0704);	/* sequential addressing */
  out_word(SEQREG, 0x0300);	/* clear synchronous reset */

  out_word(GRREG, 0x0204);	/* select map 2 for CPU reads */
  out_word(GRREG, 0x0005);	/* disable odd-even addressing */

  srcoffset = rom_char_addr;
  destoffset = EGA_BASE;
  for (ch = 0; ch < FONT_CHARS; ch++) {
	for (row = 0; row < ROM_SCAN_LINES; row++) {
		data = get_byte(FLAT_DS_SELECTOR, srcoffset++);
		put_byte(FLAT_DS_SELECTOR, destoffset++, data);
	}
	destoffset += (RAM_SCAN_LINES - ROM_SCAN_LINES);
  }

  /* Finally set the registers back for text mode. */
  writeregs(graph_off);
}


/*===========================================================================*
 *				ega_setmode				     *
 *===========================================================================*/
PRIVATE void ega_setmode(mode)
GR_MODE mode;			/* drawing mode */
{
/* Set the drawing mode.
 * This is either SET, OR, AND, or XOR.
 */
  if (mode > GR_MAX_MODE) return;
  out_byte(GRREG, DATA_ROTATE);
  out_byte(GRVAL, mode_table[mode]);
}


/*===========================================================================*
 *				ega_drawrow				     *
 *===========================================================================*/
PRIVATE void ega_drawrow(x1, x2, y, color)
GR_COORD x1;
GR_COORD x2;
GR_COORD y;
GR_COLOR color;
{
  ega_drawline(x1, y, x2, y, color);
}


/*===========================================================================*
 *				ega_drawcol				     *
 *===========================================================================*/
PRIVATE void ega_drawcol(x, y1, y2, color)
GR_COORD x;
GR_COORD y1;
GR_COORD y2;
GR_COLOR color;
{
  ega_drawline(x, y1, x, y2, color);
}


/*===========================================================================*
 *				ega_drawtext				     *
 *===========================================================================*/
PRIVATE void ega_drawtext(x, y, str, cc, fg)
GR_COORD x, y;			/* bottom left location */
GR_CHAR *str;			/* character string */
GR_SIZE cc;			/* length of string */
GR_COLOR fg;			/* foreground color */
{
  GR_SIZE width;		/* width of character */
  GR_SIZE height;		/* height of character */
  GR_BITMAP bitmap[ROM_SCAN_LINES];	/* bitmap for character */

  y -= (ROM_SCAN_LINES - 1);
  while (cc-- > 0) {
	ega_getcharbits(*str++, bitmap, &width, &height);
	gen_drawbitmap(x, y, width, height, bitmap, fg);
	x += width;
  }
}


/*===========================================================================*
 *				ega_sizetext				     *
 *===========================================================================*/
PRIVATE void ega_sizetext(buf, cc, retx, rety)
GR_CHAR *buf;
GR_SIZE cc;
GR_SIZE *retx;
GR_SIZE *rety;
{
  *retx = CHAR_WIDTH * cc;
  *rety = ROM_SCAN_LINES;
}


/*===========================================================================*
 *				ega_getscreeninfo			     *
 *===========================================================================*/
PRIVATE void ega_getscreeninfo(ip)
GR_SCREEN_INFO *ip;		/* where to return result */
{
/* Return the screen configuration information for the ega. */
  ip->rows = gr_dev.rows;
  ip->cols = gr_dev.cols;
#if VGA
  ip->xdpcm = 27;		/* assumes screen width of 24 cm */
  ip->ydpcm = 27;		/* assumes screen height of 18 cm */
#else
  ip->xdpcm = 27;		/* assumes screen width of 24 cm */
  ip->ydpcm = 19;		/* assumes screen height of 18 cm */
#endif
  ip->maxcolor = gr_dev.colors - 1;
  ip->black = gr_dev.black;
  ip->white = gr_dev.white;
  ip->fonts = 1;		/* only 1 builtin font */
}


/*===========================================================================*
 *				ega_getfontinfo				     *
 *===========================================================================*/
PRIVATE void ega_getfontinfo(font, fip)
GR_FONT font;			/* font number */
GR_FONT_INFO *fip;		/* where to return result */
{
/* Return the font information about a font in the ega.
 * Currently we only support one font.
 */
  int i;

  fip->font = font;
  if (font != 0) {
	fip->height = 1;
	fip->maxwidth = 1;
	fip->baseline = 0;
	fip->fixed = GR_TRUE;
	for (i = 0; i < 256; i++) fip->widths[i] = 1;
  }
  fip->height = ROM_SCAN_LINES;
  fip->maxwidth = CHAR_WIDTH;
  fip->baseline = 3;
  fip->fixed = GR_TRUE;
  for (i = 0; i < 256; i++) fip->widths[i] = CHAR_WIDTH;
}

/* END CODE */
