/* Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * mini-X graphics - message interface between server and device driver
 */

#ifndef	GRAPH_MSG_H
#define	GRAPH_MSG_H

#include <graphics.h>


/* Flag definitions. */
typedef unsigned short GR_FLAGS;	/* flag values */

#define	GR_FLAG_FILLAREA	((GR_FLAGS) 0x01)	/* fill areas */


/* Messages for drawing primitives. */
typedef struct {
  GR_FUNC func;			/* function code */
  GR_COUNT count;		/* count of items */
  GR_FLAGS flags;		/* flags to modify function */
  short pad;			/* padding */
} GR_MSG_HEADER;


typedef struct {
  GR_MSG_HEADER head;
  long magic;			/* magic number (and version) */
  GR_SIZE rows;			/* number of rows (or 0) */
  GR_SIZE cols;			/* number of columns (or 0) */
  long colors;			/* number of colors (or 0) */
} GR_MSG_INIT;


typedef struct {
  GR_MSG_HEADER head;
} GR_MSG_TERM;


typedef struct {
  GR_MSG_HEADER head;
  GR_POINT points[1];
} GR_MSG_POINTS;


typedef struct {
  GR_MSG_HEADER head;
  GR_LINE lines[1];
} GR_MSG_LINES;


typedef struct {
  GR_MSG_HEADER head;
  GR_RECT rects[1];
} GR_MSG_RECTS;


typedef struct {
  GR_MSG_HEADER head;
  GR_ELLIPSE ellips[1];
} GR_MSG_ELLIPS;


typedef struct {
  GR_MSG_HEADER head;
  GR_POINT points[1];
} GR_MSG_POLY;


typedef struct {
  GR_MSG_HEADER head;
  GR_COLOR foreground;
} GR_MSG_SETFOREGROUND;


typedef struct {
  GR_MSG_HEADER head;
  GR_COLOR background;
} GR_MSG_SETBACKGROUND;


typedef struct {
  GR_MSG_HEADER head;
  GR_BOOL flag;
} GR_MSG_SETUSEBACKGROUND;


typedef struct {
  GR_MSG_HEADER head;
  GR_RECT cliprects[1];
} GR_MSG_SETCLIPRECTS;


typedef struct {
  GR_MSG_HEADER head;
  GR_MODE mode;
} GR_MSG_SETMODE;


typedef struct {
  GR_MSG_HEADER head;
  GR_COORD x;
  GR_COORD y;
  GR_SIZE width;
  GR_SIZE height;
  GR_COLOR8 colors[1];
} GR_MSG_AREA8;


typedef struct {
  GR_MSG_HEADER head;
  GR_COORD x;
  GR_COORD y;
  GR_SIZE width;
  GR_SIZE height;
  GR_BITMAP bitmaps[1];
} GR_MSG_BITMAP;


typedef struct {
  GR_MSG_HEADER head;
  GR_COORD x;
  GR_COORD y;
  GR_CHAR chars[1];
} GR_MSG_TEXT;


typedef struct {
  GR_MSG_HEADER head;
  GR_COORD sx;
  GR_COORD sy;
  GR_COORD dx;
  GR_COORD dy;
  GR_SIZE width;
  GR_SIZE height;
} GR_MSG_COPYAREA;


typedef struct {
  GR_MSG_HEADER head;
  GR_SIZE width;
  GR_SIZE height;
  GR_COLOR foreground;
  GR_COLOR background;
  GR_BITMAP bitmaps[1];
} GR_MSG_SETCURSOR;


typedef struct {
  GR_MSG_HEADER head;
  GR_COORD x;
  GR_COORD y;
} GR_MSG_MOVECURSOR;


typedef struct {
  GR_MSG_HEADER head;
  GR_COORD x;
  GR_COORD y;
  GR_SIZE width;
  GR_SIZE height;
} GR_MSG_READAREA8;


typedef struct {
  GR_MSG_HEADER head;
} GR_MSG_GETSCREENINFO;


typedef struct {
  GR_MSG_HEADER head;
  GR_FONT font;
} GR_MSG_GETFONTINFO;


/* Rounding function to make sure that each message in a buffer begins
 * on a word boundary.  Some machines will trap if this is not done.
 */
#define	GR_ROUND(len)	(((len) + 3) & ~3)


/* Sizes of some messages, according to the number of items in the message. */
#define	GR_SIZEOF_MSG_POINTS(count) GR_ROUND(sizeof(GR_MSG_POINTS) + \
  (((long) (count)) - 1) * sizeof(GR_POINT))

#define	GR_SIZEOF_MSG_LINES(count) GR_ROUND(sizeof(GR_MSG_LINES) + \
  (((long) (count)) - 1) * sizeof(GR_LINE))

#define	GR_SIZEOF_MSG_RECTS(count) GR_ROUND(sizeof(GR_MSG_RECTS) + \
  (((long) (count)) - 1) * sizeof(GR_RECT))

#define	GR_SIZEOF_MSG_ELLIPS(count) GR_ROUND(sizeof(GR_MSG_ELLIPS) + \
  (((long) (count)) - 1) * sizeof(GR_ELLIPSE))

#define	GR_SIZEOF_MSG_POLY(count) GR_ROUND(sizeof(GR_MSG_POLY) + \
  (((long) (count)) - 1) * sizeof(GR_POINT))

#define	GR_SIZEOF_MSG_AREA8(count) GR_ROUND(sizeof(GR_MSG_AREA8) + \
  (((long) (count)) - 1) * sizeof(GR_COLOR8))

#define	GR_SIZEOF_MSG_BITMAP(count) GR_ROUND(sizeof(GR_MSG_BITMAP) + \
  (((long) (count)) - 1) * sizeof(GR_BITMAP))

#define	GR_SIZEOF_MSG_TEXT(count) GR_ROUND(sizeof(GR_MSG_TEXT) + \
  (((long) (count)) - 1) * sizeof(GR_CHAR))

#define	GR_SIZEOF_MSG_SETCLIPRECTS(count) GR_ROUND(sizeof(GR_MSG_SETCLIPRECTS) + \
  (((long) (count)) - 1) * sizeof(GR_RECT))

#define	GR_SIZEOF_MSG_SETCURSOR(count) GR_ROUND(sizeof(GR_MSG_SETCURSOR) + \
  (((long) (count)) - 1) * sizeof(GR_BITMAP))


/* Macro to convert from a number of adjacent bits to the number of
 * bitmap words needed to hold that many bits.
 */
#define	BITMAP_WORDS(bits) (((bits) + (sizeof(GR_BITMAP) * 8) - 1) \
  / (sizeof(GR_BITMAP) * 8))


/* Maximum values */
#define	GR_MAX_SIZEOF_MSG	2000
#define	GR_MAX_COUNT		1000


/* Magic value to prevent accidental use of the graphics device, and also to
 * verify that the device driver and graphics library structures are in sync.
 * Incrementing this value allows for new versions of the structures to be used.
 * It also allows the word and byte order of messages to be determined if
 * that ever becomes necessary (networks).
 */
#define	GR_INIT_MAGIC		0x99880001L


/* Function codes */
#define	GR_FUNC_NOP		0
#define	GR_FUNC_INIT		1
#define	GR_FUNC_SETFOREGROUND	2
#define	GR_FUNC_SETCLIPRECTS	3
#define	GR_FUNC_DRAWPOINTS	4
#define	GR_FUNC_DRAWLINES	5
#define	GR_FUNC_DRAWRECTS	6
#define	GR_FUNC_DRAWPOLY	7
#define	GR_FUNC_DRAWELLIPS	8
#define	GR_FUNC_DRAWTEXT	9
#define	GR_FUNC_DRAWAREA8	10
#define	GR_FUNC_DRAWBITMAP	11
#define	GR_FUNC_COPYAREA	12
#define	GR_FUNC_SETCURSOR	13
#define	GR_FUNC_MOVECURSOR	14
#define	GR_FUNC_SETBACKGROUND	15
#define	GR_FUNC_SETMODE		16
#define	GR_FUNC_TERM		17
#define	GR_FUNC_READAREA8	18
#define	GR_FUNC_GETSCREENINFO	19
#define	GR_FUNC_SETUSEBACKGROUND 20
#define	GR_FUNC_GETFONTINFO	21

#endif

/* END CODE */
