/*
This code is copyright (C) 1998 Robert O'Callahan.
This code is free for non-commercial use. Modification in all forms is permitted.
This license continues to apply to any modified versions. This license text must be
reproduced and distributed with any modified versions.
As a matter of courtesy I (Robert O'Callahan) would like to be informed of
any potentially useful modifications.
*/

#ifndef __SSH_H
#define __SSH_H

#include "zlib.h"

/* Some of this code has been adapted from Ian Goldberg's Pilot SSH */

typedef enum {
    SSH_MSG_NONE, SSH_MSG_DISCONNECT, SSH_SMSG_PUBLIC_KEY,
    SSH_CMSG_SESSION_KEY, SSH_CMSG_USER, SSH_CMSG_AUTH_RHOSTS,
    SSH_CMSG_AUTH_RSA, SSH_SMSG_AUTH_RSA_CHALLENGE,
    SSH_CMSG_AUTH_RSA_RESPONSE, SSH_CMSG_AUTH_PASSWORD,
    SSH_CMSG_REQUEST_PTY, SSH_CMSG_WINDOW_SIZE, SSH_CMSG_EXEC_SHELL,
    SSH_CMSG_EXEC_CMD, SSH_SMSG_SUCCESS, SSH_SMSG_FAILURE,
    SSH_CMSG_STDIN_DATA, SSH_SMSG_STDOUT_DATA, SSH_SMSG_STDERR_DATA,
    SSH_CMSG_EOF, SSH_SMSG_EXITSTATUS,
    SSH_MSG_CHANNEL_OPEN_CONFIRMATION, SSH_MSG_CHANNEL_OPEN_FAILURE,
    SSH_MSG_CHANNEL_DATA, SSH_MSG_CHANNEL_INPUT_EOF,
    SSH_MSG_CHANNEL_OUTPUT_CLOSED, SSH_MSG_OBSOLETED0,
    SSH_SMSG_X11_OPEN, SSH_CMSG_PORT_FORWARD_REQUEST, SSH_MSG_PORT_OPEN,
    SSH_CMSG_AGENT_REQUEST_FORWARDING, SSH_SMSG_AGENT_OPEN,
    SSH_MSG_IGNORE, SSH_CMSG_EXIT_CONFIRMATION,
    SSH_CMSG_X11_REQUEST_FORWARDING, SSH_CMSG_AUTH_RHOSTS_RSA,
    SSH_MSG_DEBUG, SSH_CMSG_REQUEST_COMPRESSION,
    SSH_CMSG_MAX_PACKET_SIZE, SSH_CMSG_AUTH_TIS,
    SSH_SMSG_AUTH_TIS_CHALLENGE, SSH_CMSG_AUTH_TIS_RESPONSE,
    SSH_CMSG_AUTH_KERBEROS, SSH_SMSG_AUTH_KERBEROS_RESPONSE
} SSHMessage;

typedef enum {
    SSH_CIPHER_NONE, SSH_CIPHER_IDEA, SSH_CIPHER_DES, SSH_CIPHER_3DES,
    SSH_CIPHER_TSS, SSH_CIPHER_RC4, SSH_CIPHER_BLOWFISH
} SSHCipher;

#define SSH_CIPHER_MAX SSH_CIPHER_BLOWFISH

typedef enum {
    SSH_AUTH_NONE, SSH_AUTH_RHOSTS, SSH_AUTH_RSA, SSH_AUTH_PASSWORD,
    SSH_AUTH_RHOSTS_RSA
} SSHAuthMethod;

#define SSH_PROTOFLAG_SCREEN_NUMBER 1
#define SSH_PROTOFLAG_HOST_IN_FWD_OPEN 2

#define SSH_MAX_SEND_PACKET_SIZE   250000

typedef struct {
  char FAR * hostname;

  int server_protocol_flags;

  /* This buffer is used to hold the outgoing data, and encrypted in-place
     here if necessary. */
  unsigned char FAR * outbuf;
  long outbuflen;
  /* This buffer is used by the SSH protocol processing to store uncompressed
     packet data for compression. User data is never streamed through here;
     it is compressed directly from the user's buffer. */
  unsigned char FAR * precompress_outbuf;
  long precompress_outbuflen;
  /* this is the length of the packet data, including the type header */
  long outgoing_packet_len;

  /* This buffer is used by the SSH protocol processing to store decompressed
     packet data. User data is never streamed through here; it is decompressed
     directly to the user's buffer. */
  unsigned char FAR * postdecompress_inbuf;
  long postdecompress_inbuflen;

  unsigned char FAR * payload;
  long payload_grabbed;
  long payloadlen;
  long payload_datastart;
  long payload_datalen;

  z_stream compress_stream;
  z_stream decompress_stream;
  BOOL compressing;
  int compression_level;

/* This function is changed according to the current state of the protocol */
/* SSH_handle_packet requires NO PAYLOAD on entry */
  void (* packet_handler)(struct _TInstVar FAR * pvar, char FAR * data, int len, int padding);
  int status_flags;
  int num_FWD_server_acks_expected;

  int win_cols;
  int win_rows;
} SSHState;

#define STATUS_DONT_SEND_USER_NAME            0x01
#define STATUS_EXPECTING_COMPRESSION_RESPONSE 0x02
#define STATUS_DONT_SEND_CREDENTIALS          0x04
#define STATUS_HOST_OK                        0x08
#define STATUS_INTERACTIVE                    0x10

void SSH_init(struct _TInstVar FAR * pvar);
void SSH_notify_disconnecting(struct _TInstVar FAR * pvar, char FAR * reason);
void SSH_handle_server_ID(struct _TInstVar FAR * pvar, char FAR * ID);
void SSH_notify_win_size(struct _TInstVar FAR * pvar, int cols, int rows);
void SSH_notify_user_name(struct _TInstVar FAR * pvar);
void SSH_notify_cred(struct _TInstVar FAR * pvar);
void SSH_notify_host_OK(struct _TInstVar FAR * pvar);
void SSH_send(struct _TInstVar FAR * pvar, unsigned char const FAR * buf, int buflen);
/* SSH_extract_payload returns number of bytes extracted */
int SSH_extract_payload(struct _TInstVar FAR * pvar, unsigned char FAR * dest, int len);
void SSH_get_compression_info(struct _TInstVar FAR * pvar, char FAR * dest, int len);
void SSH_end(struct _TInstVar FAR * pvar);

/* len must be <= SSH_MAX_SEND_PACKET_SIZE */
void SSH_channel_send(struct _TInstVar FAR * pvar, uint32 remote_channel_num,
                      unsigned char FAR * buf, int len);
void SSH_fail_channel_open(struct _TInstVar FAR * pvar, uint32 remote_channel_num);
void SSH_confirm_channel_open(struct _TInstVar FAR * pvar, uint32 remote_channel_num, uint32 local_channel_num);
void SSH_channel_output_eof(struct _TInstVar FAR * pvar, uint32 remote_channel_num);
void SSH_channel_input_eof(struct _TInstVar FAR * pvar, uint32 remote_channel_num);
void SSH_request_forwarding(struct _TInstVar FAR * pvar, int from_server_port,
  char FAR * to_local_host, int to_local_port);
void SSH_request_X11_forwarding(struct _TInstVar FAR * pvar,
  char FAR * auth_protocol, char FAR * auth_data, int screen_num);
void SSH_open_channel(struct _TInstVar FAR * pvar, uint32 local_channel_num,
  char FAR * to_remote_host, int to_remote_port, char FAR * originator);

#define SSH_is_any_payload(pvar) ((pvar)->ssh_state.payload_datalen > 0)
#define SSH_get_host_name(pvar) ((pvar)->ssh_state.hostname)
#define SSH_handle_packet(pvar, data, len, padding) ((pvar)->ssh_state.packet_handler((pvar), (data), (len), (padding)))
#define SSH_get_compression_level(pvar) ((pvar)->ssh_state.compressing ? (pvar)->ts_SSH_CompressionLevel : 0)

#endif
