/*
	The Memory manager. Does all the work needed for maintaining
	a memory of lists and a symbol table.
*/

#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <float.h>
#include <setjmp.h>
#include "concon.h"
#include "global.h"
#include "symbol.h"
#include "manager.fdf"
#include "parser.fdf"
#include "format.fdf"
#include "states.fdf"
#include "service.fdf"
#include "manager.h"


char    *lexemesP;       /* Pointer to lexemes bank */
List    *availableP = NULL, /* The available list */
		**symTablePP,    /* Pointer to symbol table */
		*listPoolP;          /* Pointer to first list bank */
State   *stateSpaceP;
State   *stateP = NULL;  /* Current state. */

/* The sizes */
long    firstMemSize,    symTableSize, 
		lexSize,         listSize, 
		stateSpaceSize,  statesFree,
		listFree;

int freeStates; /* Fullfills the same role as availableP does for the
				   list space, but now for the state space. */
char *lexTailP = NULL;    /* Pointer to the beginning of 
								empty part of lexemes bank */
clock_t beginTime;
extern clock_t conConTimer;

Descr *inDescrP   = NULL, *outDescrP   = NULL,
	  *trueDescrP = NULL, *falseDescrP = NULL;

/* from service.c */
extern int numDescrs;

/* 
	Before routines in this modules are used, call proMem_init first!
	Use proMem_exit to clean up the memory again. 
*/
void proMem_init( long symTab, long lex, long lst, long states ) 
				/* symTab, lexemes, lists and states */
	{
	int  i;
    conConTimer = clock();
    symTablePP  = (List **)malloc( (symTableSize=symTab)*sizeof(List *) );
    for (i=0;i<symTab;i++) symTablePP[i] = NULL;
    
	lexTailP = lexemesP = malloc( lexSize = lex );
    listPoolP = malloc( listSize=lst );
	
	stateSpaceP = malloc( (stateSpaceSize = states)*sizeof( State ));
    if (symTablePP == NULL || lexemesP == NULL || listPoolP == NULL)
		{
		proFor_print("\nERROR - Not enough memory...");
		longjmp( env, 1);
		} 

/*  Clear the linked list space. */
    listFree=(int)(lst/sizeof(List));
    for (i=0;i<(listFree);i++) 
		{
		listPoolP[i].nextP = &listPoolP[i+1];
        listPoolP[i].u.b = 0;
        listPoolP[i].data.dataP = NULL;
        }
	listPoolP[listFree-1].nextP = NULL;
    availableP = listPoolP;
    proState_clearStateSpace();
    }
	
void proMem_exit( void )
	{
	free( symTablePP );
	free( lexemesP );
	free( listPoolP );
	free( stateSpaceP );
	}


/*
	Registering a list of hard coded commands.
*/
void proMem_register( Commands *cP )
   {
   List *lP;
   Commands *currP = cP;
   
   stateP->token = KEYWORD;
   while (currP->actionFuncP != nofunc)
       {
       strcpy(lexTailP, currP->lexeme );
       lP = proMem_insert();
       lP->u.p.locked = TRUE;
       lP->nextP = proMem_getAvail( );
       lP = lP->nextP;
       lP->u.p.type=ACTIONFUNC;
       lP->data.actionFuncP = currP->actionFuncP;
       currP++;
       }
   }

/*
	Declaring some constants.
*/

void proMem_constants( void )
	{
	List new, *lP;
	
    cc_clearLElm( &new );

    cc_makeAtom( &new, "TRUE" );
    ((List *)new.data.dataP)->u.p.locked = TRUE;

     cc_makeAtom( &new, "FALSE" );
    ((List *)new.data.dataP)->u.p.locked = TRUE;

    cc_makeAtom( &new, "FLOAT-EPSILON" );
    ((List *)new.data.dataP)->u.p.locked = TRUE;
    strcpy(lexTailP, "FLOAT-EPSILON" );
    lP = new.data.dataP;
    lP->u.p.locked = TRUE;
    lP->nextP = proMem_getAvail( );
	cc_makeReal( lP->nextP, FLT_EPSILON );
	
    cc_makeAtom( &new, "FLOAT-MAX" );
    ((List *)new.data.dataP)->u.p.locked = TRUE;
    lP = new.data.dataP;
    lP->u.p.locked = TRUE;
    lP->nextP = proMem_getAvail( );
	cc_makeReal( lP->nextP, FLT_MAX );
	}


/*
	Extended services.
*/
void proMem_memFree( void )
	{
	char buf[256];
    sprintf( buf, "\nMemory free:\n%d bytes for lexemes.\n%ld list elements free\n%ld states free on stack.\n%d descriptors currently used.\n",
            LEX_MEM_LEFT, listFree, statesFree, numDescrs );
	proFor_print( buf );
	}

Descr *proMem_getSymbolTable( void )
	{
	char  lexTypes[][12] = 
		{
		{"ALPHA"},
		{"LITERAL"},
		{"KEYWORD"},
		};

	char bindStr[2][10] = 
		{
		{"UNBOUND"},
		{"BOUND"},
		};
	Descr *descrP, *symDescrP;
	List  *traverserP, new;
	int   i, bound;
	
	cc_newDescr( &descrP, NULL );
	cc_newDescr( &symDescrP, NULL );

	for (i=0;i<symTableSize;i++)
		{
		traverserP= symTablePP[i];
	
		while (traverserP != NULL)
			{
			cc_clearDescr( symDescrP );
			cc_clearLElm( &new );

/* Bound/unbound ? */
			bound = 0;
			if (((List *)traverserP->data.dataP)
				->nextP != NULL)
				bound = 1;	 
			cc_makeAtom( &new, bindStr[bound] );
			cc_copyAndInsert( symDescrP, &new, FALSE );

/* The symbol itself. */
            new.u.p.type = ((List *)
                       traverserP->data.dataP)->u.p.type;
			new.data.dataP = traverserP->data.dataP;
			cc_copyAndInsert( symDescrP, &new, FALSE );

/* What's the type ? */
			cc_makeAtom( &new, 
				lexTypes[((List *)traverserP->data.dataP)
                ->u.p.type] );
			cc_copyAndInsert( symDescrP, &new, FALSE );

/* Now insert into the symbol list. */
			cc_makeSubList( &new, symDescrP->nextP );
			cc_copyAndInsert( descrP, &new, FALSE );						
			
/* Next symbol. */
			traverserP = traverserP->nextP;
			}
		}
	cc_freeDescr( &symDescrP );
	cc_clearLElm( &new );
	cc_makeSubList( &new, descrP->nextP );
	cc_newDescr( &symDescrP, NULL );	
	cc_copyAndInsert( symDescrP, &new, FALSE );
	cc_freeDescr( &descrP );
	return symDescrP;
	}

/*
	Symbol table entry done here.
*/
int  proMem_hashPjw( char *s )
	{
	char *p;
	unsigned long h=0, g;
	
	for (p=s;*p!='\0';p++)
		{
		h=(h<<4)+(*p);
		g = h&0xf0000000L;
		if (g)
			{
			h=h^(g>>24);
			h=h^g;
			}
		}
	return (int)(h%symTableSize);
	}


/* list functions */
List *proMem_getAvail( void )
	{
	List *newP;
	
    if ( (newP = availableP) == NULL)
        {
        proMem_garbageCollect();
        if ( (newP = availableP) == NULL)
            {
            proFor_print("\nERROR - Not Enough Memory...");
            longjmp( env, 1 );
            }
        }
    availableP = availableP->nextP;
	listFree--;
	cc_clearLElm( newP );
	return newP;
	}	

List *proMem_getAvailAndCopy( List *valP )
	{
	List *newP;
	
    if ( (newP = availableP) == NULL)
        {
        proMem_garbageCollect();
        if ( (newP = availableP) == NULL)
            {
            proFor_print("\nERROR - Not Enough Memory...");
            longjmp( env, 1 );
            }
        }
    availableP = availableP->nextP;
	listFree--;
    memcpy( newP, valP, sizeof(List) );
	return newP;
    }   

/* symbol table functions */
List *proMem_insert( void )
	{
	List *newP = NULL, *datP = NULL;
	List new, dat;
	int bin;
	
	cc_clearLElm( &new );
	cc_clearLElm( &dat );
	bin  = proMem_hashPjw( lexTailP );
	new.nextP = symTablePP[bin];
    new.u.p.type = SYMBOL;
	new.data.dataP = NULL;

	dat.nextP = NULL;
    dat.u.p.type = stateP->token;
	dat.data.dataP = lexTailP;

    newP = proMem_getAvailAndCopy( &new );
	symTablePP[bin] = newP;
    datP = proMem_getAvailAndCopy( &dat );
	newP->data.dataP=datP;

	lexTailP += strlen( lexTailP ) + 1;
	if (LEX_MEM_LEFT <= 4*readSize)
		{
		proMem_garbageCollect();
		if (LEX_MEM_LEFT <= 4*readSize)
			{
			proFor_print("\nERROR - Not enough memory in lexeme buffer.\nAborting...");
			getchar();
			exit(1);
			} 
		}
	return datP;
	} 	
	
List *proMem_lookup( char *s )
	{
	int bin;
	List *symbolsP;

	bin  = proMem_hashPjw( s );
	symbolsP = symTablePP[bin];
	
	while (symbolsP != NULL)
		if (!strcmp( s, ((List *)
		   (symbolsP->data.dataP))->data.dataP ))
			return ((List *)(symbolsP->data.dataP));
		else 
			symbolsP=(List *)symbolsP->nextP;		
	return NULL;
	}	




/*
	processList applies func to every list element, and closeFunc
	when the end of a list is encountered. Used for displaying,
	marking and copying lists.
*/
List *proMem_processList( List *lP,  
							Inherit (*func)(List *lP), 
								void (*closeFunc)( void ) )
	{
	List *retListP = lP;
	Inherit cut = FALSE;

	if (lP == NULL)
		return lP;
	
	while (lP != NULL && !(cut))
		{
		cut = func( lP );
        if (lP->u.p.type == SUBLIST && !(cut))
			{
			proMem_processList( lP->data.dataP, 
					func, closeFunc );
            closeFunc( );
			}
		lP = lP->nextP;
		}	
	return retListP;
	}



/* The functions that output a list in ascii form */
List *proMem_displayList( List *lP )
	{
	List *retListP = lP;
	proFor_newLine();
	proMem_processList( lP, displayable, 
						proMem_closeBracket );
	return retListP;
	}
	
Inherit displayable( List *lP )
	{
	char numbers[80];
    if (lP->u.p.quoted )
        proFor_out("\'");
    switch (lP->u.p.type)
		{
		case INTEGER:
			sprintf( numbers, "%ld", lP->data.integer );
			proFor_out( numbers );
			break;
		case REAL:
			sprintf( numbers, "%.8g",  lP->data.real ); 
			proFor_out( numbers );
			break;
		case ALPHA:
		case LITERAL:
		case KEYWORD:
			proFor_out(	(char *)((List *)lP->data.dataP)->
						 data.dataP );
			break;
		case SUBLIST:
			proFor_out( "(" );
			break;
		default:
			sprintf( numbers, "[Unprintable Token %d]", 
                lP->u.p.type );
			proFor_out( numbers );
			break;
		}
	return FALSE;
	}

void proMem_closeBracket( void )
	{
	proFor_out( ")" );
	}	

/*
	List copying.
*/	
List *proMem_copyList( Descr *descrP, List *lP, Inherit full )
	{
	List *traverserP = (List *)descrP;
	Descr *subDescrP;
	
	while (lP != NULL)
		{
        traverserP->nextP = proMem_getAvailAndCopy( lP );
		traverserP = traverserP->nextP;
        if (full  && lP->u.p.type == SUBLIST)
			{
			cc_newDescr( &subDescrP, NULL );
			traverserP->data.dataP = 
				proMem_copyList( subDescrP, lP->data.dataP,
					full );
			cc_freeDescr( &subDescrP );
			}
		lP = lP->nextP;
		}	
	return descrP->nextP;
	}

/* garbage collection */
Inherit mark( List *lP )
	{
    int type;
	List *valueP;
    Inherit cut = lP->u.p.marked;

/*
    if (lP->nextP != NULL)
        if (((List *)lP->nextP)->u.p.type == NOT_USED)
        lP->nextP = NULL;
*/
    lP->u.p.marked = TRUE;
    type = lP->u.p.type;
    if (type == ALPHA || type == LITERAL)
		if ( (valueP = (List *)lP->data.dataP) != NULL)
        valueP->u.p.marked = TRUE;
	return cut;
	}
		
void proMem_garbageCollect( void )
	{
	int i, total = (int)(listSize/(sizeof(List)));
	List *lP, **prevPP, *tempP; 
	char *scnP;
	long length;

	beginTime = clock();
/* Unmarking */
	for (i=0;i<total;i++)
        listPoolP[i].u.p.marked = FALSE;

/* Marking the elements an element points to. */
	for (i=0;i<symTableSize;i++)
		{
		lP= symTablePP[i];
		while (lP != NULL)
			{
			tempP = lP->data.dataP;
			if (tempP != NULL)
                if (tempP->u.p.type != LITERAL &&
                    (List *)tempP->nextP >= listPoolP       &&
                    (List *)tempP->nextP <  (List *)(listPoolP + listSize))
					proMem_processList( (List *)((List *)lP->
						data.dataP)->nextP, mark, nofunc );
			lP=(List *)lP->nextP;
			}		
		}

/* The lists protected by descriptors. */
	for (i=0;i<total;i++)
        if (listPoolP[i].u.p.type == DESCRIPTOR)
			proMem_processList( &listPoolP[i], mark, nofunc );

/* The lists belonging to states. */
	for (i=0;i<stateSpaceSize;i++)
		if (stateSpaceP[i].validState )
			{
			proMem_processList( stateSpaceP[i].execP, mark, nofunc );
			proMem_processList( stateSpaceP[i].returnP, mark, nofunc );
			proMem_processList( stateSpaceP[i].localP, mark, nofunc );
			proMem_processList( stateSpaceP[i].titleP, mark, nofunc );
			proMem_processList( stateSpaceP[i].req.newReadP, mark, nofunc );
			proMem_processList( stateSpaceP[i].inP, mark, nofunc );
			proMem_processList( stateSpaceP[i].outP, mark, nofunc );
			}

/* Getting rid of unused lexemes in the symbol table. */
	for (i=0;i<symTableSize;i++)
		{
		lP    = symTablePP[i];
		prevPP = (List **)&symTablePP[i];
		while (lP != NULL)
			{
            if ( ((List *)lP->data.dataP)->u.p.type == KEYWORD ||
                 ((List *)lP->data.dataP)->u.p.locked )
                 ((List *)lP->data.dataP)->u.p.marked  = TRUE;
            if ( !(((List *)lP->data.dataP)->u.p.marked)
				 && ((List *)lP->data.dataP)->nextP != NULL)
                 ((List *)lP->data.dataP)->u.p.marked  = TRUE;
            if ( ((List *)lP->data.dataP)->u.p.marked)
				{
                lP->u.p.marked = TRUE;
				prevPP = (List **)&lP->nextP;
				}
			else
				{
				*prevPP = lP->nextP;
				scnP    = (char *)((List *)lP->data.dataP)->
						 data.dataP;
				length  = strlen( scnP ) + 1;
				scnP += strlen( scnP ) + 1;
				while (scnP < lexTailP)		 		 
					{
					proMem_lookup( scnP )->data.dataP = 
							(scnP - length);
					memmove( (scnP-length), scnP, strlen( scnP )+1 );
					scnP += strlen(scnP - length)+1;
					}		
				memmove( (lexTailP - length), lexTailP, 
						  LEX_MEM_LEFT );
				lexTailP -= length;
				}
			lP=(List *)lP->nextP;
			}		
		}

/* Collecting the available elements. */
	availableP=NULL;	
	listFree = 0;
	for (i=0;i<total;i++) 
        if (!listPoolP[i].u.p.marked)
			{
			listPoolP[i].nextP = availableP;
            listPoolP[i].u.b = 0;
			availableP = &listPoolP[i];
			listFree++;
			}
		else 
            listPoolP[i].u.p.marked = FALSE;
	garbColTime += clock() - beginTime;
	}		





