// <><><><><><><><><><><><><>  Plot3D.h  <><><><><><><><><><><><><><>
//
// ----------------------------------------------------
// (C) Copyright Clark Dailey, Inc. 1999
//
// C++ Class Name:
//    Plot3D
//
// Purpose:
//    Used as a base class for several Plot3D classes
//
// ----------------------------------------------------

#ifndef	_PLOT3D_H_	 	// avoid repeated including
#define	_PLOT3D_H_	1	// has been included

#include <math.h>
#include <assert.h>

#ifdef _DEBUG
 #define PLOT_ASSERT(v)	assert(v)
#else
 #define PLOT_ASSERT(v)	// do nothing
#endif
#define	DBGPLOT	TRACE	// TRACE for Windows95, printf for DOS

// degrees/radians conversions
inline double Deg2Rad(double deg) { return((deg*2.*3.1415925359)/360.);   }
inline double Rad2Deg(double rad) { return((rad*360.)/(2.*3.1415925359)); }

#pragma warning( disable : 4100 )  // unreferenced formal parameter

// ----------
// axis info
// ----------
typedef struct axis_s
{
	unsigned nxticks;		// number of x tick marks
	unsigned nyticks;		// number of y tick marks
	unsigned nzticks;		// number of z tick marks
	long	 axis_color;	// color of axis
	char	 font_name[128];// name of font (needs to be True-Type for rotating)
	int		 font_height;	// pixels
	int		 font_width;	// pixels
	unsigned tick_length;	// 1-100
	unsigned tick_spacing;	// 1-100
} AXIS_PLOT_INFO;


// ------------------------------------------------------------------------------------------
// abstract class - must be derived from to use
class Plot3D // base class for several Plot3D classes
{
public:
	Plot3D();	// constructor
	virtual ~Plot3D();	// destructor

	// pure virtual functions must be defined by derived class
	virtual void MoveTo(int xv,int yv)=0;	// move to a location
	virtual void DrawTo(int xv,int yv)=0;	// draw to a location
	virtual void PlotText(AXIS_PLOT_INFO* pAxisInfo,double degrees,int xv,int yv,CString text)=0;	// draw text at location
	// justification: 1=left, 2=center, 3=right

	// overridable color functions
	virtual void ColorTo(long color) { } // called when pen color change is required
	virtual long Color(double x,double y,double z) { return(1); } // default is monochrome

	// settings
	int SetRanges(double xmin=-1.,double xmax=1.,
				  double ymin=-1.,double ymax=1.,
				  double zmin=-1.,double zmax=1.);

	int SetViewingPosition(double xytilt=60.,double zrotate=45.);

	int SetVideoRes(int xvmin=0,   int yvmin=0,
					int xvmax=640, int yvmax=480,double scale=1.0);

	void SetVideoAxisInversion(int xv_invert=0,int yv_invert=0);

	void SetAxisParameters(unsigned ShowAxis=0,AXIS_PLOT_INFO* pAxisInfo=0);

	// call back routines
	virtual void cbBegPlot() { }	// called at beginning of plotting
	virtual void cbEndPlot() { }	// called when plotting is complete
	virtual void cbBegDrawAxis() { }// called at beginning of drawing axis
	virtual void cbEndDrawAxis() { }// called when draw axis plotting is complete
	virtual void cbCheckUserAbort() {  }	// called periodically to check for user abort plotting
	void  DoAbort() { m_abort_plotting=1; }	// call this from cbCheckUserAbort() to abort plotting

	// pen location status
	int  CurX()     { return(m_xv_cur);    }	// returns current x location
	int  CurY()     { return(m_yv_cur);    }	// returns current y location
	long CurColor() { return(m_color_cur); }	// returns current plotting color

	// ------------------------------------------------------------
	//             Transformation Equations
	// ------------------------------------------------------------
	// object space to plot space
	double xpf(double xo,double yo,double zo) { return(tm[0][0]*xo + tm[0][1]*yo + tm[0][2]*zo); }
	double ypf(double xo,double yo,double zo) { return(tm[1][0]*xo + tm[1][1]*yo + tm[1][2]*zo); }
	double zpf(double xo,double yo,double zo) { return(tm[2][0]*xo + tm[2][1]*yo + tm[2][2]*zo); } 
	
	// plot space to object space
	double xof(double xp,double yp,double zp) { return(tm[0][0]*xp + tm[1][0]*yp + tm[2][0]*zp); }
	double yof(double xp,double yp,double zp) { return(tm[0][1]*xp + tm[1][1]*yp + tm[2][1]*zp); }
	double zof(double xp,double yp,double zp) { return(tm[0][2]*xp + tm[1][2]*yp + tm[2][2]*zp); }
	
	// plot space to video space
	int xvf(double xp) { return( (int)((m_p2v*(xp-m_xpmid) + (double)m_xvmid)) ); }
	int yvf(double yp) { return( (int)((m_p2v*(yp-m_ypmid) + (double)m_yvmid)) ); }
	void MapPoint(double x,double y,double z,int* xv,int* yv);
	void MapAndDrawLine(double x1,double y1,double z1,double x2,double y2,double z2);

	// ouput control
	void PlotOutputOff() { m_to_plotter=0; }
	void PlotOutputOn()  { m_to_plotter=1; }

	void Dump();

protected:	// routines
	// calculate transformation matrix
	void CalcTM();

	// calculate projections of corners
	void CalcCorners();

	// filter move/draw commands
	void FilterMoveTo(int xv,int yv);
	void FilterDrawTo(int xv,int yv);
	void FilterColorTo(long color);

protected:	// user input data
	// x,y,z min/max ranges
	double	m_xmin, m_xmax, m_ymin, m_ymax, m_zmin, m_zmax;

	// viewing angles
	double	m_xytilt, m_zrotate;	// degrees

	// x,y video min/max ranges
	int m_xvmin, m_xvmax, m_yvmin, m_yvmax;
	double m_scale;	// plotting scale (.01 < scale < 1)

	// axis parameters
	unsigned       m_show_axis;  // 0=no, 1=draw axis
	AXIS_PLOT_INFO m_axis;

protected:	// internal data
	double	m_xmid, m_ymid, m_zmid;

	// transformation matrix
	double tm[3][3];

	// min/mid/max projection values
	double m_xpmin,  m_xpmid,  m_xpmax;
	double m_ypmin,  m_ypmid,  m_ypmax;
	double m_zpmin,  m_zpmid,  m_zpmax;

	// mid points of video coordinate system
	int m_xvmid, m_yvmid;
	double m_p2v;	// maps plot coordinates to video coordinates

	// corner coordinates
	double m_xcorner [8],m_ycorner [8],m_zcorner [8];	// corner values
	double m_xpcorner[8],m_ypcorner[8],m_zpcorner[8];	// projection of corners

	// current pen location and color
	int	 m_xv_cur, m_yv_cur;
	long m_color_cur;

	// output variables
	int m_to_plotter;	 	// 0=no, 1=output to plotter
	int m_pending_moveto;	// 0=no, 1=need to move before drawing

	// control flags
	int	m_abort_plotting;	// 0=no, 1=abort plotting
};

#endif	/* if _PLOT3D_H_ included */

/*	<><><><><><><><><><><><><>  Plot3D.h  <><><><><><><><><><><><><><> */



