(* Copyright (C) 1998, 1999 Stuart King. All rights reserved. *)

(***************************************************************************
** Operating systems vary in how they represent end-of-line in text files.
** For example Unix and it's variants (such as Linux) use a line feed
** character to represent end-of-line. DOS, OS/2, Windows use two characters
** carriage return, and line feed together to represent end-of-line.
** Irie Pascal can read text files with any combination of carriage return,
** line feed characters as end-of-line, but it writes text files in the
** native file format. So converting text files with Irie Pascal is
** very easy all that is necessary is to use read each line of text
** using "readln" and write out each line using "writeln".
****************************************************************************)
program text(output);
var
   param : integer;
   OutDir : filename;
   mode : integer;

   procedure DisplayError(msg : string);
   begin
      writeln('ERROR: ', msg);
      halt
   end;

   function DirectorySeparator : string;
   begin
      if (platform <> platform_linux) and
         (platform <> platform_fbsd) and
         (platform <> platform_solaris) and
         (platform <> platform_solaris_sparc) then
         DirectorySeparator := '\'
      else
         DirectorySeparator := '/'
   end;

   procedure syntax;
   begin
      writeln('Text - Converts text files into the native format');
      writeln('Syntax: ivm text filespec .. filespec output_directory');
      writeln('        The wildcards ''?'' and ''*'' may be used');
      writeln('For example:');
      writeln('        ivm text *.pas ' + DirectorySeparator + 'temp' + DirectorySeparator);
      halt
   end;

   procedure ConvertFile(InName, OutName : filename);
   const
      MaxLine = 1024;   (* Maximum length of a line in the text file to be converted *)
   var
      f, g : text;
      line : string[MaxLine];
   begin
      writeln(InName, ' -> ', OutName);
      reset(f, InName);
      rewrite(g, OutName);
      while not eof(f) do
         begin
            readln(f, line);
            writeln(g, line)
         end;
      close(g);
      close(f)
   end;

   procedure ConvertTextFiles(f : filename);
   var
      DirPart, NamePart, ExtPart : filename;
      fname : filename;
      d : Dir;

      (***************************************************************
      ** PURPOSE: This function tests whether a file name is
      **          specified by a file spec (which may have wild cards).
      **          For example the file spec "test.*" specifies all file
      **          names with "test" as the name and any extension.
      **          So FileMatch("test.*", "test.dat") would return "true".
      ** NOTES: This function may be built-into Irie Pascal in the future
      **        but it's kept separate for now. Before building it in
      **        I want to be reasonably sure I can implement it on all
      **        platforms that Irie Pascal will eventually run on.
      *)
      function FileMatch(fspec, fname : filename) : boolean;
      var
         SpecPos, NamePos : integer;
         next : char;
         done : boolean;
      begin (* FileMatch *)
         (*
         ** File names under Linux and FreeBSD are case sensitive but
         ** they aren't under other platforms. So for those platforms
         ** where file names are not case sensitive the file spec and
         ** file name are converted to all lower case so that their
         ** original case doesn't matter (i.e. "NAME" <> "Name" but by
         ** by converting "NAME" to "name" and "Name" to "name" they match).
         *)
         if (platform <> platform_linux) and (platform <> platform_fbsd) then
            begin
               fspec := lowercase(fspec);
               fname := lowercase(fname)
            end;
         NamePos := 1;
         SpecPos := 1;
         done := false;
         repeat
            if (SpecPos>length(fspec)) and (NamePos>length(fname)) then
               begin
                  FileMatch := true;
                  done := true
               end
            else if (SpecPos>length(fspec)) and (NamePos<=length(fname)) then
               begin
                  FileMatch := false;
                  done := true
               end
            else if (SpecPos<=length(fspec)) and (NamePos>length(fname)) then
               begin
                  FileMatch := false;
                  done := true
               end
            else if fspec[SpecPos] = '?' then
               begin
                  inc(SpecPos);
                  inc(NamePos)
               end
            else if fspec[SpecPos] = '*' then
               begin
                  if SpecPos = length(fspec) then
                     begin
                        FileMatch := true;
                        done := true
                     end
                  else
                     begin
                        next := fspec[SpecPos+1];
                        NamePos := pos(next, fname, NamePos);
                        if NamePos > 0 then
                           inc(SpecPos)
                        else
                           begin
                              FileMatch := false;
                              done := true
                           end
                     end
               end
            else if fspec[SpecPos] = fname[NamePos] then
               begin
                  inc(SpecPos);
                  inc(NamePos)
               end
            else
               begin
                  FileMatch := false;
                  done := true
               end
         until done;
      end; (* FileMatch *)

   begin (* ConvertTextFiles *)
      f := fexpand(f);
      writeln(f);
      fsplit(f, DirPart, NamePart, ExtPart);
      NamePart := NamePart + ExtPart;
      OpenDir(d, DirPart);
      repeat
         ReadDir(d, fname);
         if fname <> '' then    (* if there are more files in directory *)
            begin
               if FileMatch(NamePart, fname) then
                  begin
                     ConvertFile(DirPart+fname, OutDir+fname)
                  end
            end
      until fname = '';
      CloseDir(d)
   end; (* ConvertTextFiles *)

begin
   if paramcount < 2 then
      syntax;
   OutDir := fexpand(paramstr(paramcount));
   getfilemode(OutDir, mode);
   if (mode and dir_bit) = 0 then
      DisplayError("'" + paramstr(paramcount) + "' is not a directory");
      (* fsplit(OutDir, OutDir, , ); *)
   if copy(OutDir, length(OutDir), 1) <> DirectorySeparator then
      OutDir := OutDir + DirectorySeparator;
   for param := 1 to paramcount-1 do
      ConvertTextFiles(paramstr(param));
end.
