/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/

#include <stdio.h>
#include <math.h>

#include "kask.h"
#include "kasktri.h"
#include "kaskass.h"
#include "kaskcmd.h"
#include "kasksol.h"


#define L_SMALL 4
#define L_MEDIUM 6
#define L_LARGE 20
#define BLOCK_SMALL 100
#define BLOCK_MEDIUM 100
#define BLOCK_LARGE 100


int iluExists= 0;
static int failLevel= 10;
static int AX,AY;
static int testPrintP= true;
static long nCount,mCount;


int ILUPrepare();
int ILUAssemble();
int ComputeILU();
int ILUaxMul();
int InvertILU();
int ILUIterate();
int CheckILU();


static int SetNoOne();
static int CountNeighbors();
static int AllocSparse();
static int InitNeighbors();
static int StoreNeighbors();
static int SetSparseZero();
static int ILUAddElem();
static int ILUSetBC();
static int OneLineILU();
static int MulLine();
static int StepDown();
static int StepUp();
static void ILUTest();
static int SparseCheck();
static void ILUPrint();
static int ILUPrintElem();



int ILUPrepare()
{
static int first= true;
if(first)
{
if(!InitList(SPARSE_SMALL,L_SMALL*sizeof(NEIGHBOR),BLOCK_SMALL))
{ZIBStdOut("Not enough memory\n");return false;}
if(!InitList(SPARSE_MEDIUM,L_MEDIUM*sizeof(NEIGHBOR),BLOCK_MEDIUM))
{ZIBStdOut("Not enough memory\n");return false;}
if(!InitList(SPARSE_LARGE,L_LARGE*sizeof(NEIGHBOR),BLOCK_LARGE))
{ZIBStdOut("Not enough memory\n");return false;}
first= false;
}
if((actTriang->noOfPoints)!=iluExists)
{
if(!ApplyP(SetNoOne,all))return false;
if(!ApplyE(CountNeighbors,nodal))return false;
if(!ApplyP(AllocSparse,all))
{ZIBStdOut("Generation of sparse matrix failed\n");return false;}
if(!ApplyP(InitNeighbors,all))return false;
if(!ApplyE(StoreNeighbors,nodal))return false;
iluExists= actTriang->noOfPoints;
}
if(!ApplyP(SetSparseZero,all))return false;

return true;
}


static int SetNoOne(p)
PT*p;
{
p->noOfNeighbors= 1;
return true;
}


static int CountNeighbors(ed)
EDG*ed;
{
PT*p1= ed->p1,*p2= ed->p2;

(p1->noOfNeighbors)++;
(p2->noOfNeighbors)++;

return true;
}


static int AllocSparse(p)
PT*p;
{
NEIGHBOR*neigh= nil;
int no= p->noOfNeighbors,listNo;

if(no<=(p->maxNeighbors))return true;
if(no<=L_SMALL)listNo= SPARSE_SMALL;
else
{
if(no<=L_MEDIUM)listNo= SPARSE_MEDIUM;
else
{
if(no<=L_LARGE)listNo= SPARSE_LARGE;
else
{sprintf(globBuf,"Unexpected many neighbors at point %d\n",p->indexP);ZIBStdOut(globBuf);return false;}
}
}

if((p->neighbors)!=nil)
ReturnElem((p->maxNeighbors==L_SMALL)?SPARSE_SMALL:SPARSE_MEDIUM,(PTR)(p->neighbors));

neigh= (NEIGHBOR*)GetElem(listNo);
if(neigh==nil)return false;
p->maxNeighbors= (listNo==SPARSE_SMALL)?L_SMALL:((listNo==SPARSE_MEDIUM)?L_MEDIUM:L_LARGE);
p->neighbors= neigh;

return true;
}


static int InitNeighbors(p)
PT*p;
{
p->noOfNeighbors= 1;
(p->neighbors)[0].p= p;
return true;
}


static int StoreNeighbors(ed)
EDG*ed;
{
PT*p1= ed->p1,*p2= ed->p2;
NEIGHBOR*nb;
int i,k,lng;

lng= p1->noOfNeighbors;
for(k= 0;k<lng;k++)
{
nb= &((p1->neighbors)[k]);
if(((nb->p)->indexP)>(p2->indexP))break;
}
for(i= lng;i>k;i--)
(p1->neighbors)[i].p= (p1->neighbors)[i-1].p;
(p1->neighbors)[k].p= p2;
(p1->noOfNeighbors)++;

lng= p2->noOfNeighbors;
for(k= 0;k<lng;k++)
{
nb= &((p2->neighbors)[k]);
if(((nb->p)->indexP)>(p1->indexP))break;
}
for(i= lng;i>k;i--)
(p2->neighbors)[i].p= (p2->neighbors)[i-1].p;
(p2->neighbors)[k].p= p1;
(p2->noOfNeighbors)++;

return true;

}


static int SetSparseZero(p)
PT*p;
{
int k,lng= p->noOfNeighbors;
NEIGHBOR*n;

for(k= 0;k<lng;k++)
{
n= &((p->neighbors)[k]);
n->stiff= ZERO;
n->ilu= ZERO;
}
RA(p,R_DIAG)= ZERO;
RA(p,R_RHS)= ZERO;
return true;
}



int ILUAssemble()
{
InitNumAss(N_STD);
partP= P_ALL;
if(!ILUPrepare())return false;
if(!ApplyT(ILUAddElem,all))return false;
if ( !(actSolve->compNormP)) 
   { if(!ApplyP(ILUSetBC,dirichlet))return false; }
actTriang->iluAvailable= true;
return true;
}


static int ILUAddElem(t)
TR*t;
{
PT**pts= &(t->p1),*p;
NEIGHBOR*n;
int i,j,k,lng;

if((actProblem->NumAss)(t)==false)return false;
for(i= 0;i<3;i++)
{
p= pts[i];
RA(p,R_DIAG)+= assA[i][i];
lng= p->noOfNeighbors;
n= p->neighbors;
for(j= 0;j<3;j++)
for(k= 0;k<lng;k++)
if(n[k].p==pts[j])
n[k].stiff+= 
(symP==FULL)?assA[i][j]:(i>j)?assA[i][j]:assA[j][i];
RA(p,R_RHS)+= assB[i];
}
return true;
}


static int ILUSetBC(p)
	PT*p;
{
	NEIGHBOR*n= p->neighbors;
	int k,lng= p->noOfNeighbors;
	int class = p->class;
	REAL bxy, *dirVals=(actProblem->dirVals);

	(actProblem->DirichF)(p->x,p->y,class,dirVals);
	bxy= dirVals[0];

	for(k= 0;k<lng;k++)n[k].stiff= (n[k].p==p)?ONE:ZERO;
	RA(p,R_RHS)= bxy;
	RA(p,R_SOL)= bxy;
	RA(p,R_DIAG)= ONE;

   return true;
}



int ComputeILU()
{
return ApplyP(OneLineILU,allBackward);
}


static int OneLineILU(p)
PT*p;
{
NEIGHBOR*nbk,*nb1,*nb2;
PT*pk,*pj;
double sum,Lij,Rjk;
int r,s,i,k,min,j,lngi= p->noOfNeighbors,lngj;

i= p->indexP;

for(r= 0;r<lngi;r++)
{
nbk= &((p->neighbors)[r]);
pk= nbk->p;
k= pk->indexP;
min= (k<i)?k:i;

sum= ZERO;

for(j= 0;j<lngi;j++)
{
nb1= &((p->neighbors)[j]);
pj= nb1->p;
if((pj->indexP)>=min)break;

Lij= nb1->ilu;
lngj= pj->noOfNeighbors;

for(s= 0;s<lngj;s++)
{
nb2= &((pj->neighbors)[s]);
if(nb2->p==pk)
{
Rjk= nb2->ilu;
break;
}
}
sum= sum+Lij*Rjk;
}

if(k<i)nbk->ilu= ((nbk->stiff)-sum)/RA(pk,R_DIAG);
else nbk->ilu= (nbk->stiff)-sum;
}

return true;
}



int ILUaxMul(x,y)
int x,y;
{
int rc;

AX= x;AY= y;
rc= ApplyP(MulLine,all);

return rc;
}


int InvertILU(x,y)
int x,y;
{
AX= x;AY= y;

SetZeroField(x);
ApplyP(StepDown,all);
ApplyP(StepUp,allBackward);

return true;
}


static int MulLine(p)
PT*p;
{
int k,lng= p->noOfNeighbors;
NEIGHBOR*n;
double sum= ZERO;

for(k= 0;k<lng;k++)
{
n= &((p->neighbors)[k]);
sum+= ((p->neighbors)[k].stiff)*RA(n->p,AX);
}
RA(p,AY)= sum;
return true;
}


static int StepDown(p)
PT*p;
{
NEIGHBOR*nb;
double sum= ZERO;
int m;

m= 0;
nb= &((p->neighbors)[m]);

while(nb->p!=p)
{
sum= sum+(nb->ilu)*RA(nb->p,AX);
m= m+1;
nb= &((p->neighbors)[m]);
}
RA(p,AX)= RA(p,AY)-sum;
return true;
}


static int StepUp(p)
PT*p;
{
NEIGHBOR*nb;
double sum= ZERO;
int m,lng= p->noOfNeighbors;

m= lng-1;
nb= &((p->neighbors)[m]);

while(nb->p!=p)
{
sum= sum+(nb->ilu)*RA(nb->p,AX);
m= m-1;
nb= &((p->neighbors)[m]);
}
RA(p,AX)= (RA(p,AX)-sum)/(nb->ilu);

return true;
}



int ILUIterate(maxIte,adrRes,eps,omega,verboseP)
REAL eps,*adrRes,omega;
int*maxIte,verboseP;
{
int step= 0,maxit= *maxIte;
REAL res= 1.0e+10,lastRes;

ComputeILU();
if(verboseP)
{ sprintf(globBuf,"ILU solver, %d points, requested acc. %e\n",
actTriang->noOfPoints,eps);ZIBStdOut(globBuf);}
while(step<maxit)
{
lastRes= res;
ILUaxMul(R_SOL,R_RES);
lin(R_RES,R_RHS,R_RES,-1.0);
res= sqrt(scalprod(R_RES,R_RES));
if(res<eps)break;
InvertILU(R_ILU,R_RES);
lin(R_SOL,R_ILU,R_SOL,1.0);
step++;
if(verboseP){sprintf(globBuf,"%d: res=%e\n",step,res);ZIBStdOut(globBuf);}
if((res>lastRes)&&(step>10)&&(res>failLevel))
{
ZIBStdOut("ILU solver divergent?\n");
maxit= -1;
break;
}
}

*maxIte= step;
*adrRes= res;
if(step>=maxit)
{
if(verboseP)ZIBStdOut("ILU-Solver failed\n");
return false;
}
else if(verboseP)ZIBStdOut("ILU-Solver O.K.\n");
return true;
}



int ILUCheck(cmd)
COMMAND*cmd;
{
if(ParsCheck(cmd,0,0))return false;

if (actTriang==nil)
{ ZIBStdOut("No triangulation\n"); return false; }
testPrintP= (actTriang->noOfPoints)<100;
if(!ILUPrepare())return false;
ILUTest();
if(!ILUAssemble())return false;
if(!ComputeILU())return false;
ILUPrint();
return true;
}


static void ILUTest()
{
nCount= 0L;mCount= 0L;
ZIBStdOut("Checking sparse storage\n");
ApplyP(SparseCheck,all);
sprintf(globBuf,"%d elements, %4.1f%%filled\n",nCount,(100.0*nCount)/((REAL)mCount));
ZIBStdOut(globBuf);
return;
}


static int SparseCheck(p)
PT*p;
{
int k,lng= p->noOfNeighbors;
NEIGHBOR*n;

nCount+= lng;
mCount+= (p->maxNeighbors);
if(testPrintP)
{
sprintf(globBuf,"%d:(",p->indexP);ZIBStdOut(globBuf);
for(k= 0;k<lng;k++)
{
n= &((p->neighbors)[k]);
sprintf(globBuf,"%d",(n->p)->indexP);ZIBStdOut(globBuf);
if(k<(lng-1))ZIBStdOut(",");
}
ZIBStdOut(")\n");
}
return true;
}


static void ILUPrint()
{
if(testPrintP)ApplyP(ILUPrintElem,all);
return;
}


static int ILUPrintElem(p)
PT*p;
{
int k,lng= p->noOfNeighbors;
REAL x;

for(k= 0;k<lng;k++)
{
x= (p->neighbors)[k].stiff;
if(x==ZERO)continue;
sprintf(globBuf,"(%d,%d):%10.4e %10.4e\n",p->indexP,
(p->neighbors)[k].p->indexP,x,
(p->neighbors)[k].ilu);
ZIBStdOut(globBuf);
}
return true;
}





