 /*
  * Khoros: $Id: File.c,v 1.3 1992/03/20 23:37:37 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: File.c,v 1.3 1992/03/20 23:37:37 dkhoros Exp $";
#endif

 /*
  * $Log: File.c,v $
 * Revision 1.3  1992/03/20  23:37:37  dkhoros
 * VirtualPatch5
 *
  */ 


/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>								<<<<
   >>>>	    file name: File.c					<<<<
   >>>>								<<<<
   >>>>   description: Khoros standard file routines		<<<<
   >>>>								<<<<
   >>>>      routines: file_tempnam				<<<<
   >>>>		       file_open				<<<<
   >>>>		       file_close				<<<<
   >>>>		       file_read				<<<<
   >>>>		       file_write				<<<<
   >>>>		       file_lseek				<<<<
   >>>>		       file_tell				<<<<
   >>>>		       file_access				<<<<
   >>>>		       file_unlink				<<<<
   >>>>		       file_lock				<<<<
   >>>>		       file_descriptors				<<<<
   >>>>								<<<<
   >>>> modifications:						<<<<
   >>>>								<<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


/*
 *  If KFILE is not defined, define it since we want at least file
 *  transport.  For Khoros 2.0 we will allow the ability of not having
 *  file transport, but until things are better defined and a little
 *  stable....
 */
#ifndef KFILE
#define KFILE
#endif


#ifdef KFILE

#include "internals.h"	
#include "File.h"	



/**************************************************************
*
* MODULE NAME: file_tempnam
*
*     PURPOSE: This function initializes a tempnam for a "file"
*	       type transport.  Given a request with a filename of
*	       the following syntax:
*
*			"filename"
*			"/path/filename"
*			"./path/filename"
*			"file=XXXXXXXXXX"
*			     ...etc....
*
*
*       INPUT: 
*
*        
*      OUTPUT:  returns -1 or 0 depending whether we
*		sucessfully created the template
*
* CALLED FROM:  internal routine called from ktempnam()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_tempnam(identifier, template, result)

char *identifier;
char *template;
char *result;
{
	int	fid;
	char	*temp, filename[LENGTH];


	/*
	 *  If the template is "-" or "#" this means that it is stdin,
	 *  stdout, or stderr, which means don't make a tempnam simply
	 *  copy this as the output file
	 */
	if (strcmp(template, "-") == 0 || strcmp(template, "#") == 0)
	{
	   strcpy(result, template);
	   return(0);
	}

	/*
	 *  Make a tempnam from the template
	 */
	(void) strcpy(filename, template);
	if ((temp = vtempnam(filename)) == NULL)
	   return(-1);

	/*
	 *  Create the unique file so that no one steals away from us...
	 */
	if ((fid = open(temp, O_WRONLY | O_CREAT | O_TRUNC, 0666)) <= 0)
	{
	   perror("file_tempnam:  Unable to create temporary file....");
	   return(-1);
	}
	close(fid);

	if (identifier == NULL)
	   (void) strcpy(result, temp);
	else
	   (void) sprintf(result,"%s=%s", identifier, temp);

	kfree(temp);
	return(0);
}



/**************************************************************
*
* MODULE NAME: file_open
*
*     PURPOSE: This function opens a "file" connection.  It is the
*	       internal driver to open a file, which is called by
*	       the user when they call kopen().  kopen() calls when
*	       it gets an open request with a filename of the following
*	       syntax:
*
*			"filename"
*			"/path/filename"
*			"./path/filename"
*			"file=XXXXXXXXXX"
*			     ...etc....
*
*
*       INPUT: 
*
*        
*      OUTPUT:  returns 0 or -1 if an error occurs
*
* CALLED FROM:  internal routine called from kopen()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_open(path, flags, mode, file)

char *path;
int  flags;
int  mode;
kfile *file;
{
	int	   fid, access;
	ResourceStruct *resources;


	if (strcmp(path,"-") == 0)
	{
	   access = flags & O_ACCMODE;
	   if (access == O_RDONLY)
	      fid = fileno(stdin);
	   else if (access == O_WRONLY)
	      fid = fileno(stdout);
	   else
	   {
	      (void) fprintf(stderr,"file_open:  Invalid open access...\n\n");
	      (void) fprintf(stderr,"  '-' can be opened for either reading \
or writing, but open access flags are\n  currently set to (%d)\n", access); 
		 return(-1);
	   }
	}
	else if (strcmp(path,"#") == 0)
	{
	   fid = fileno(stderr);
	}
	else if ((fid = open(path, flags, mode)) < 0)
	{
	   perror("file_open:  Open failure....");
	   return(-1);
	}

	if (!(resources = (ResourceStruct *) calloc(1, sizeof(ResourceStruct))))
	{
	   (void) fprintf(stderr,"file_open:  Not enough memory....\n\n");
	   (void) fprintf(stderr,"  Unable to malloc (%d) bytes for the khoros \
file structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}

	resources->fid = fid;
	file->resources = (caddr_t) resources;
	return(0);
}



/**************************************************************
*
* MODULE NAME: file_close
*
*     PURPOSE: This function closes a "file".  It is the internal driver
*	       to close a data file, which is called by the user
*	       when they call kclose().  kclose() calls the "file" internal
*	       drivers by the nature of the transport, which is dictated
*	       by the initial kopen().
*
*
*       INPUT:  file - the kfile structure.
*
*      OUTPUT:  returns whether we were able to close the file
*
* CALLED FROM:  internal routine called from kclose()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_close(file)

kfile *file;
{
	int	   status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call close() since "kfile" is just a regular file
	 *  descriptor.
	 */
	status = close(resources->fid);
	return(status);
}



/**************************************************************
*
* MODULE NAME: file_read
*
*     PURPOSE: This function reads a "file".  It is the internal driver
*	       to read data from a file, which is called by the user
*	       when they call kread().  kread() calls the "file" internal
*	       drivers by the nature of the transport, which is dictated
*	       by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		ptr  - the pointer to store the data into.
*		nbytes - the number of bytes to read.
*
*      OUTPUT:  returns the number of bytes read from the file
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/

int file_read(file, ptr, nbytes)

kfile *file;
char  *ptr;
int   nbytes;
{
	int	   count;
	int	   numread = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call read() since "kfile" is just a regular file
	 *  descriptor.
	 */
	do
	{
	   count = MIN(nbytes - numread, DefaultPageSize);
	   count = read(resources->fid, (char *) (ptr + numread), count);
	   numread += count;
	}
	while (numread < nbytes && count > 0);

	return(numread);
}



/**************************************************************
*
* MODULE NAME: file_write
*
*     PURPOSE: This function writes to a "file".  It is the internal driver
*	       to write data from the supplied data array to a file, which
*	       is called by the user when they call kwrite().  kwrite() calls
*	       the "file" internal drivers by the nature of the transport,
*	       which is dictated by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		ptr    - the pointer to write the data from.
*		nbytes - the number of bytes to read.
*
*      OUTPUT:  returns the number of bytes written to the kfile
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_write(file, ptr, nbytes)

kfile *file;
char  *ptr;
int   nbytes;
{
	int	   count;
	int	   numwrite = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call write() since "kfile" is just a regular file
	 *  descriptor.
	 */
	do
	{
	    count = MIN(nbytes - numwrite, DefaultPageSize);
	    count = write(resources->fid, (char *) (ptr + numwrite), count);
	    numwrite += count;
	} while (numwrite < nbytes && count > 0);

	return(numwrite);
}



/**************************************************************
*
* MODULE NAME: file_lseek
*
*     PURPOSE: This function is used to do a "lseek".  It is the internal
*	       driver to rewind to a specific point so that the data can be
*	       skipped or re-read.  This is called when the user calls
*	       klseek().  klseek() calls the "file" internal drivers by the
*	       nature of the transport, which is dictated by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		offset - the offset in which to seek
*		whence - the control of how the offset will be applied
*
*      OUTPUT:  returns the number of bytes written to the kfile
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_lseek(file, offset, whence)

kfile *file;
int   offset;
int   whence;
{
	int	   pos;
	ResourceStruct *resources = (ResourceStruct *) file->resources;

	/*
	 *  Simply call lseek() since "kfile" is just a regular file
	 *  descriptor.
	 */
	pos = lseek(resources->fid, offset, whence);
	return(pos);
}



/**************************************************************
*
* MODULE NAME: file_tell
*
*     PURPOSE: This function is used to do a "lseek".  It is the internal
*	       driver to rewind to a specific point so that the data can be
*	       skipped or re-read.  This is called when the user calls
*	       ktell().  ktell() calls the "file" internal drivers by the
*	       nature of the transport, which is dictated by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		offset - the offset in which to seek
*		whence - the control of how the offset will be applied
*
*      OUTPUT:  returns the number of bytes written to the kfile
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_tell(file)

kfile *file;
{
	int	   pos;
	ResourceStruct *resources = (ResourceStruct *) file->resources;

	/*
	 *  Simply call tell() since "kfile" is just a regular file
	 *  descriptor.
	 */
	pos = tell(resources->fid);
	return(pos);
}



/**************************************************************
*
* MODULE NAME: file_access
*
*     PURPOSE: This function is used to do see if a file is accessable.
*	       This is called when the user calls kaccess().  kaccess()
*	       calls the "file" internal drivers by the nature of the
*	       transport, which is dictated by the transport_identifier().
*
*
*       INPUT:  path     - the initial path.
*		filename - the filename that they wish to access
*		mode     - the access mode information
*
*      OUTPUT:  returns the result
*
* CALLED FROM:  internal routine called from kaccess()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_access(path, filename, mode)

char *path;
char *filename;
int  mode;
{
	int	   status;

	/*
	 *  Simply call access() since "filename" is just a regular filename
	 */
	status = access(filename, mode);
	return(status);
}



/**************************************************************
*
* MODULE NAME: file_unlink
*
*     PURPOSE: This function initializes a tempnam for a "file"
*	       type transport.  Given a request with a filename of
*	       the following syntax:
*
*			"filename"
*			"/path/filename"
*			"./path/filename"
*			"file=XXXXXXXXXX"
*			     ...etc....
*
*
*       INPUT: 
*
*        
*      OUTPUT:  returns 0 or -1 depending whether we
*		sucessfully created the template
*
* CALLED FROM:  internal routine called from kunlink()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_unlink(identifier, filename)

char *identifier;
char *filename;
{
	int	status;


	status = unlink(filename);
	return(status);
}



/**************************************************************
*
* MODULE NAME: file_lock
*
*     PURPOSE: This function locks a "file" transport.  Given
*	       a request with a filename of the following syntax:
*
*			"filename"
*			"/path/filename"
*			"./path/filename"
*			"file=XXXXXXXXXX"
*			     ...etc....
*
*
*       INPUT: 
*
*        
*      OUTPUT:  returns 0 or -1 depending whether we
*		sucessfully created the template
*
* CALLED FROM:  internal routine called from kflock()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_lock(file, operation)

kfile *file;
int   operation;
{
	int	status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	status = flock(resources->fid, operation);
	return(status);
}



/**************************************************************
*
* MODULE NAME: file_descriptors
*
*     PURPOSE: This function shouldn't exist, but there is occasion
*	       when the user wants to know the actual file descriptor
*	       associated with the file descriptor.
*
*       INPUT: file structure
*
*        
*      OUTPUT:  inum - the input file descriptor
*
* CALLED FROM:  internal routine called from kdescriptor()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int file_descriptors(file, inum, onum)

kfile *file;
int   *inum, *onum;
{
	int	status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	if (inum != NULL) *inum = resources->fid;
	if (onum != NULL) *onum = resources->fid;
	return(0);
}

#endif  /* KFILE */
/* don`t add after the endif */
