/* module: 4dd_edit.c iris editing and display functions */

static char SccsId[] = "@(#)4dd_edit.c	1.2";

#include "gl.h"
#include "stdio.h"
#include "math.h"
#include "device.h"

#include "4dd_echo.h"
#include "4dd_cyfile.h"
#include "4dd_gt.h"

extern int errno;
int edit_exit;


edit(gs, cmd)

GSPEC *gs;
char *cmd;
{
	long gid;
	long gid_help, gid_wire, gid_render;
	Device event;
	short value;

	static struct Vertex *vtx = 0;			/* rect coord database */

	if (gs->valid == 0) {
		puts("Invalid image");
		return(-1);
	}

	edit_exit = FALSE;
	foreground();
	qreset();

	/* make each window */
	gid_help = help_event("Please standby...", (Device)CREATE, (short)0);

	vtx = makerect(gs, vtx);				/* initialize the vertex tables */

	gid_wire = wire_event(vtx, (Device)CREATE, (short)0);
	gid_render = render_event(vtx, (Device)CREATE, (short)0);

	qdevice((Device)MENUBUTTON);
	qdevice((Device)LEFTMOUSE);
	qdevice((Device)WINQUIT);

	gid = winget();
	while (edit_exit == FALSE) {
		help_event("For menu press and hold right button", (Device)REDRAW, 0);
		event = qread(&value);
		switch (event) {
		default:
			if (gid == gid_wire) {
				wire_event(vtx, event, value);
			} else if (gid == gid_render) {
				render_event(vtx, event, value);
			} else if (gid == gid_help) {
				help_event((char *)0, event, value);
			}
			break;
		case REDRAW:
			if (value == gid_wire) {
				wire_event(vtx, event, value);
			} else if (value == gid_render) {
				render_event(vtx, event, value);
			} else if (value == gid_help) {
				help_event((char *)0, event, value);
			}
			break;
		case INPUTCHANGE:
			gid = value;
			if (gid != 0) {
				winset(gid);
			}
			break;
		case WINQUIT:
			edit_exit = TRUE;
			break;
		}
	}
	help_event((char *)0, (Device)DESTROY, (short)0);
	wire_event(vtx, (Device)DESTROY, (short)0);
	render_event(vtx, (Device)DESTROY, (short)0);
	return(0);
}



struct Vertex *makerect(gs, vtx)

GSPEC *gs;
struct Vertex *vtx;
{
	if (vtx == 0) {
		vtx = (struct Vertex *)malloc(sizeof(struct Vertex));
	}
	gstovtx(gs, vtx);
	return(vtx);
}


gstovtx(gs, vtx)		/* gs to vertex conversion */

GSPEC *gs;
struct Vertex *vtx;
{
	Coord theta, theta_incr;		/* angle of lg and increment */
	short lt, lg;					/* latitude/longitude counters */
	int radius;						/* radius from input image */
	Coord r, x, y;					/* rectangular coords */
	Coord y_incr;					/* latitudnal increment */
	Coord sin_theta, cos_theta;		/* time savers */

	vtx->gs = gs;
	vtx->nlt = gs->nlt;
	vtx->nlg = gs->nlg;
	vtx->ltmin = gs->ltmin;
	vtx->ltmax = gs->ltmax;
	vtx->lgmin = gs->lgmin;
	vtx->lgmax = gs->lgmax;

	if (! (gs->flags & FLAG_CARTESIAN)) {
		theta_incr = (gs->lgincr * 1.e-6);		/* to radians */
		theta = 0.;
		y_incr = gs->ltincr * 1.e-6;			/* to meters */
		for (lg = 0; lg < vtx->nlg; ++lg) {
			y = -(vtx->nlt/2) * y_incr;
			sin_theta = fsin(theta);
			cos_theta = fcos(theta);
			for (lt = 0; lt < vtx->nlt; ++lt) {
				radius = GETR(gs, lt, lg);		/* cyl radius */
				if (radius != VOID) {
					r = radius * 1.e-6;			/* to meters */
					vtx->pnt[lg][lt][LX] = r * sin_theta;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = r * -cos_theta;
					vtx->data[lg][lt] = 0;
				} else {
					vtx->pnt[lg][lt][LX] = 0.;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = 0.;
					vtx->data[lg][lt] = VTX_VOID;
				}
				y += y_incr;
			}
			theta += theta_incr;
		}
	} else {
		for (lg = 0; lg < vtx->nlg; ++lg) {
			x = (lg - vtx->nlg/2) * gs->lgincr * 1.e-6;
			for (lt = 0; lt < vtx->nlt; ++lt) {
				if (gs->flags & FLAG_BILATERAL) {
					y = (lt % (gs->nlt/2) - vtx->nlt) * gs->ltincr * 1.e-6;
				} else {
					y = (lt - vtx->nlt / 2) * gs->ltincr * 1.e-6;
				}
				radius = GETR(gs, lt, lg);
				if (radius != VOID) {
					vtx->pnt[lg][lt][LX] = x;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = radius * 1.e-6;;
				} else {
					vtx->pnt[lg][lt][LX] = x;
					vtx->pnt[lg][lt][LY] = y;
					vtx->pnt[lg][lt][LZ] = -.17;
					vtx->data[lg][lt] = VTX_VOID;
				}
			}
		}
	}
	makenormals(vtx);
}



makenormals(vtx)

struct Vertex *vtx;
{
	int lg;
	int lt;
	int i, j;
	Boolean cart = (vtx->gs->flags & FLAG_CARTESIAN) ? TRUE : FALSE;

	/* special case, low longitude */
	vnormal(vtx->pnt[0][0], vtx->pnt[0][0], vtx->pnt[1][0],
		vtx->pnt[0][0], vtx->pnt[0][1], cart);
	for (lt = 1; lt < vtx->nlt-1; ++lt) {
		vnormal(vtx->pnt[0][lt], vtx->pnt[0][lt], vtx->pnt[1][lt],
			vtx->pnt[0][lt-1], vtx->pnt[1][lt+1], cart);
	}
	vnormal(vtx->pnt[0][lt], vtx->pnt[0][lt], vtx->pnt[1][lt],
		vtx->pnt[0][lt-1], vtx->pnt[0][lt], cart);

	/* each inside longitude */
	for (lg = 1; lg < vtx->nlg-1; ++lg) {
		/* special case, low latitude */
		vnormal(vtx->pnt[lg][0], vtx->pnt[lg-1][0], vtx->pnt[lg+1][0],
				vtx->pnt[lg][0], vtx->pnt[lg][1], cart);
		/* each inside latitude */
		for (lt = 1; lt < vtx->nlt-1; ++lt) {
			vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg+1][lt],
					vtx->pnt[lg][lt-1], vtx->pnt[lg][lt+1], cart);
		}
		/* special case, high latitude */
		vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg+1][lt],
			vtx->pnt[lg][lt-1], vtx->pnt[lg][lt], cart);
	}

	/* special case, high longitude */
	vnormal(vtx->pnt[lg][0], vtx->pnt[lg-1][0], vtx->pnt[lg][0],
		vtx->pnt[lg][0], vtx->pnt[lg][1], cart);
	for (lt = 1; lt < vtx->nlt-1; ++lt) {
		vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg][lt],
			vtx->pnt[lg][lt-1], vtx->pnt[lg][lt+1], cart);
	}
	vnormal(vtx->pnt[lg][lt], vtx->pnt[lg-1][lt], vtx->pnt[lg][lt],
		vtx->pnt[lg][lt-1], vtx->pnt[lg][lt], cart);
}



#define SQ(n) ((n)*(n))

vnormal(p, vtx0, vtx1, vtx2, vtx3, cart)

Coord *p;
Coord *vtx0, *vtx1, *vtx2, *vtx3;
Boolean cart;
{
	Coord v0[3], v1[3];			/* vectors */
#	define X 0
#	define Y 1
#	define Z 2
	Coord magnitude;

	/* vectors */
	v0[X] = vtx1[LX] - vtx0[LX];
	v0[Y] = vtx1[LY] - vtx0[LY];
	v0[Z] = vtx1[LZ] - vtx0[LZ];
	v1[X] = vtx2[LX] - vtx3[LX];
	v1[Y] = vtx2[LY] - vtx3[LY];
	v1[Z] = vtx2[LZ] - vtx3[LZ];

	/* normal of vector pair */
	p[NX] = v0[Y]*v1[Z] - v0[Z]*v1[Y];
	p[NY] = v0[Z]*v1[X] - v0[X]*v1[Z];
	p[NZ] = v0[X]*v1[Y] - v0[Y]*v1[X];

	/* magnitude */
	magnitude = fsqrt(SQ(p[NX])+SQ(p[NY])+SQ(p[NZ]));

	/* make unit magnitude */
	if (magnitude != 0.) {
		if (cart) {
			p[NX] /= -magnitude;
			p[NY] /= -magnitude;
			p[NZ] /= -magnitude;
		} else {
			p[NX] /= magnitude;
			p[NY] /= magnitude;
			p[NZ] /= magnitude;
		}
	} else {
		p[NX] = 0.;
		p[NY] = 0.;
		p[NZ] = 0.;
	}
}


edit_config()
{
	wire_getdefaults(&wire);
	render_getdefaults(&render);
}
