/* cytoslice.c		Convert Cyberware range map to slices */

static char SccsId[] = "%W%";

/*	Given a filled and clipped range map file in Cyberware format and 
 *	several parameters defining vertex pitches; produces a list of
 *	polylines.  The parameters are verticies per slice and distance
 *	between slices (in meters).  The output is an ASCII list of X,Y,Z
 *	triplets defining the slices.  Each slice is followed by a blank line.
 *
 *	No cleverness is used in spacing the verticies.  They will have
 *	equal spacing in X or theta.
 *
 *	Will exit with an error if a VOID is encountered.
 *	Respects clipping limits set by ltrange and lgrange.
 *	Works with either cylindrical or rectangular range maps.
 *
 *	Compiler:
 *		SGI Iris 4d series:
 *			cc -DIRIS -o cytoslice cytoslice.c cyfile.c strings.c -lm
 *			cc -DIRIS -o cytoslice cytoslice.c cyfile.o strings.o -lm
 *
 *		Sun products:
 *			cc -DSUN -o cytoslice cytoslice.c cyfile.c strings.c -lm
 *			cc -DSUN -o cytoslice cytoslice.c cyfile.o strings.o -lm
 *
 *		Hewlett Packard IPC:
 *			cc -DSUN -o cytoslice cytoslice.c cyfile.c -lm
 *			cc -DSUN -o cytoslice cytoslice.o cyfile.o -lm
 */

#include "fcntl.h"
#include "stdio.h"
#include "math.h"
#include "cyfile.h"

#define FALSE	0
#define TRUE	1

extern int errno;
void perror();

float gsinterpolate();


char *usage = "Usage: cytoslice [-asr] [-v nvertex] [-z z_incr] cy_file(s)\n";


main(argc, argv)

int argc;
char **argv;
{
	int c;
	extern char *optarg;
	extern int optind;

	int fd;							/* target image file */
	FILE *fp;
	GSPEC *gs;						/* database descriptor */
	int n, nvertex = -1;			/* points per slice */
	float range;
	float x, x_incr;
	float z, z_limit;
	float z_incr = -1.;
	float y;
	float theta, theta_incr, theta_range;
	int theta_steps;
	int autocad = FALSE;			/* enable AutoCAD script format output */
	int split = FALSE;				/* split, each slice into separate file */
	int closed = FALSE;				/* make polylines closed */
	int rect = FALSE;				/* make image space rectangular */
	int errflg = 0;
	int first = 1;					/* print only one 3DPOLY msg */
	int proengineer = FALSE;

	char filename[128];
	int filenumber = 0;

	while ((c = getopt(argc, argv, "asrv:z:")) != -1) {
		switch (c) {
		case 'a':
			autocad = TRUE;
			break;
		case 's':
			split = TRUE;
			break;
		case 'r':
			rect = TRUE;
			break;
		case 'v':
			nvertex = atoi(optarg);
			if (nvertex < 3) errflg++;
			break;
		case 'z':
			z_incr = atof(optarg);
			if (z_incr <= 0.) errflg++;
			break;
		case '?':
			errflg++;
			break;
		}
	}
	if (errflg) {
		fputs(usage, stderr);
		exit(2);
	}
	for (; optind < argc; optind++) {

#ifdef HP_INTEGRAL
		if (split) {
			puts("Insert image disk, type ENTER to proceed"); fflush(stdout);
			getchar();
		}
#endif

		fd = open(argv[optind], O_RDONLY);
		if (fd < 0) {
			perror("cytoslice");
			continue;
		}
		gs = cyread(NULL, fd);
		if (gs == NULL) {
			close(fd);
			exit(-1);
		}
		close(fd);
		if (nvertex < 0) {
			/* not specified, set to default */
			nvertex = gs->lgmax - gs->lgmin + 1;
		}
		if (z_incr < 0.) {
			/* not specified, set to default */
			z_incr = (float)gs->ltincr * 1.e-6;
		}
		if (rect) {
			gs->flags |= FLAG_CARTESIAN;
		}
		if (split) {

#ifdef HP_INTEGRAL
			puts("Insert first disk, type ENTER to proceed"); fflush(stdout);
			getchar();
#endif

		} else {
			fp = stdout;		/* if not split mode, standard output */
		}
		if (!(gs->flags & FLAG_CARTESIAN)) {
			if (gs->lgmin == 0 && gs->lgmax == gs->nlg-1) {
				closed = TRUE;
				theta_steps = gs->lgmax - gs->lgmin + 1;
			} else {
				theta_steps = gs->lgmax - gs->lgmin;
			}
		} else {
			theta_steps = gs->lgmax - gs->lgmin;
		}
		theta_range = (float)(theta_steps * gs->lgincr) * 1.e-6;
		if (closed) {
			theta_incr = theta_range / (float)nvertex;
		} else {
			theta_incr = theta_range / (float)(nvertex - 1);
		}
		z = (float)(gs->ltmin * gs->ltincr) * 1.e-6;
		z_limit = (float)((gs->ltmax + 1) * gs->ltincr) * 1.e-6;
		while (z <= z_limit) {
			if (split) {
				do {
					sprintf(filename, "%.5s%d.scr", argv[optind], filenumber);
					fp = fopen(filename, "w");
					printf("File: %s", filename); fflush(stdout);
					if (fp == NULL) {

#ifdef HP_INTEGRAL
						puts(", error, maybe disk is full?");
						fputs("Insert new disk, type ENTER to proceed", stdout);
						getchar();
#else
						puts(", error, attempting to clear");
#endif
					}
				} while (fp == NULL);
				clearerr(fp);
				fputs(", opened", stdout); fflush(stdout);
			}
			if (autocad) {
				if (
					((gs->flags & FLAG_CARTESIAN) && first) ||
					((gs->flags & FLAG_CARTESIAN) && split) ||
					!(gs->flags & FLAG_CARTESIAN)
				) {
					fputs("3DPOLY\n", fp);
					if (gs->flags & FLAG_CARTESIAN) {
						first = 0;
					}
				}
			}
			theta = (float)(gs->lgmin * gs->lgincr) * 1.e-6;
			for (n = 0; n < nvertex - closed; ++n) {
				range = gsinterpolate(gs, theta, z, closed);
				if (gs->flags & FLAG_CARTESIAN) {
					x = theta;
					y = range;
				} else {
					x = range * sin(theta);
					y = range * -cos(theta);
				}
				if (autocad) {
					if (gs->flags & FLAG_CARTESIAN) {
						fprintf(fp, "%f,%f,%f\n", x, y, z);
					} else {
						fprintf(fp, "%f,%f,%f\n", x, y, z);
					}
				} else if (proengineer) {
					fprintf(fp, "%d %f %f %f\n", n+1, x, y, z);
				} else {
					fprintf(fp, "%f %f %f\n", x, y, z);
				}
				theta += theta_incr;
			}
			if (autocad && closed) {
				fputs("CLOSE\n", fp);
			} else {
				fputs("\n\n", fp);
			}
			if (split) {
				fflush(fp);
				if (ferror(stderr)) {
					/* output error, pause, repeat slice */
					fclose(fp);
					unlink(filename);
#ifdef HP_INTEGRAL
					puts(", error, maybe disk is full?");
					fputs("Insert new disk, type ENTER to proceed", stdout);
					getchar();
#else
					puts("output error, repeating slice");
#endif

				} else {
					/* continue to next slice */
					puts(", completed");
					z += z_incr;
					++filenumber;
					fclose(fp);
				}
			} else {
				/* next slice */
				z += z_incr;
			}
		}
	}
}


float gsinterpolate(gs, x, y, closed)

GSPEC *gs;
float x, y;
int	closed;
{
	float lt_r;			/* latitude as a float */
	float lt_f;			/* latitude fractional part */
	float lt_i;			/* latitude integer part */
	float lg_r;			/* longitude as a float */
	float lg_f;			/* longitude fractional part */
	float lg_i;			/* longitude integer part */
	int lt_0, lt_1;		/* low/high latitude */
	int lg_0, lg_1;		/* left/right longitud index */
	int r_00, r_01, r_10, r_11;
	float r_a, r_b;

	lg_r =  x / ((float)gs->lgincr * 1.e-6);			/* longitude */
	lg_i = floor(lg_r);									/* left longitude */
	lg_0 = (int)lg_i;									/* longitude index */
	if (lg_0 == gs->lgmax) {
		if (closed) {
			lg_1 = 0;
		} else {
			lg_1 = lg_0;
		}
	} else {
		lg_1 = lg_0 + 1;
	}

	lt_r = y / ((float)gs->ltincr * 1.e-6);				/* latitude */
	lt_i = floor(lt_r);									/* lower longitude */
	lt_0 = lt_i;										/* latitude index */
	if (lt_0 == gs->ltmin || lt_0 == gs->ltmax) {
		lt_1 = lt_0;
	} else {
		lt_1 = lt_0 + 1;									/* upper neighbor */
	}

	if (
		lg_0 < 0 || lg_0 >= gs->nlg ||
		lt_0 < 0 || lt_0 >= gs->nlt
	) {
		fprintf(stderr, "\nRange error: x=%f y=%f\n", x, y);
		exit(-1);
	}

	/* find the interpolation ratio */
	lg_f = lg_r - lg_i;									/* fractional part */
	lt_f = lt_r - lt_i;									/* fractional part */

	/* get radii of the four nearby points from database */
	r_00 = GETR(gs, lt_0, lg_0);
	r_01 = GETR(gs, lt_1, lg_0);
	r_10 = GETR(gs, lt_0, lg_1);
	r_11 = GETR(gs, lt_1, lg_1);

	/* test for void values */
	if (r_00 == VOID || r_01 == VOID || r_10 == VOID || r_11 == VOID) {
		fprintf(stderr, "\nVOID value: x=%f y=%f\n", x, y);
		return(0.);
	}
	
	/* interpolate and return the radius at the given coordinate */
	r_a = (float)r_00 + (float)(r_10 - r_00) * lg_f;
	r_b = (float)r_01 + (float)(r_11 - r_01) * lg_f;
	return (r_a + (r_b - r_a) * lt_f) * 1.e-6;
}
