/*-
 * customs.h --
 *	Header for the customs agent.
 *
 * Copyright (c) 1988, 1989 by the Regents of the University of California
 * Copyright (c) 1988, 1989 by Adam de Boor
 * Copyright (c) 1989 by Berkeley Softworks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any non-commercial purpose
 * and without fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California,
 * Berkeley Softworks and Adam de Boor make no representations about
 * the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 *
 *	"$Id: customs.h,v 1.23 1992/10/05 04:43:11 stolcke Exp $ ICSI (Berkeley)"
 */
#ifndef _CUSTOMS_H_
#define _CUSTOMS_H_

/*
 * First comes stuff needed by both the client and the server...
 */
#include    <sys/types.h>
#ifndef SOCK_STREAM /* ultrix doesn't guard against multiple inclusion ... */
#include    <sys/socket.h>
#endif
#include    <sys/param.h>
#include    <netinet/in.h>
#include    <arpa/inet.h>
#include    <sys/time.h>
#ifdef hpux
# define _KERNEL
# include    <sys/resource.h>
# undef _KERNEL
# define FSIZE_SCALE 512
#else
# include    <sys/resource.h>
#endif

#ifndef INADDR_LOOPBACK
#define	INADDR_LOOPBACK		(u_long)0x7f000001	/* in host order */
#endif /* INADDR_LOOPBACK */

#ifndef RLIM_INFINITY
#define	RLIM_INFINITY	0x7fffffff
#endif /* RLIM_INFINITY */

#include    "config.h"		/* for default timeouts and ports */

#include    "rpc.h"

#define MACHINE_NAME_SIZE   	64  	    	/* Longest machine name */
#define MAX_INFO_SIZE	    	MAX_DATA_SIZE	/* Most data returned by
						 * CUSTOMS_INFO call */
#define MAX_NUM_GROUPS		16  	    	/* Most groups a caller
						 * can be in. Don't use
						 * NGROUP as that varies from
						 * OS to OS. */
#define MAX_NUM_RLIMITS		8		/* Should be RLIM_NLIMITS,
						 * but you never now how
						 * vendors can screw this up. */

typedef enum {
/* CLIENT FUNCTIONS */
    CUSTOMS_PING, 	/* See if server is up */
    CUSTOMS_HOST, 	/* Get address of host to use */
    CUSTOMS_AVAILINTV,	/* Set interval for availability check */
    CUSTOMS_SETAVAIL,	/* Set availability criteria */
    CUSTOMS_INFO, 	/* Find who's registered */
    CUSTOMS_MASTER,	/* Find address of MCA */
    CUSTOMS_LOG,  	/* Log information of rpc socket */
/* AGENT-INTERNAL FUNCTIONS */
    CUSTOMS_AVAIL,	/* Tell master if machine available */
    CUSTOMS_HOSTINT,	/* Agent-internal HOST call */
    CUSTOMS_REG,  	/* Register local machine with master */
    CUSTOMS_ALLOC,	/* Local machine allocated by master */
    CUSTOMS_CONFLICT,	/* More than one master exists */
/* ELECTION BROADCAST FUNCTIONS */
    CUSTOMS_CAMPAIGN,	/* Attempt to become master */
    CUSTOMS_NEWMASTER,	/* Declare oneself master */
/* IMPORT/TCP FUNCTIONS */
    CUSTOMS_IMPORT,	/* Import a job */
    CUSTOMS_KILL, 	/* Kill a running job with a signal */
    CUSTOMS_EXIT, 	/* RETURN CALL: status and id # of exited process */
/* DEBUG FUNCTIONS */
    CUSTOMS_ABORT,	/* Exit */
    CUSTOMS_RESTART,	/* Reexecute with same arguments */
    CUSTOMS_DEBUG,	/* Turn on debugging. */
    CUSTOMS_ELECT,  	/* Start off a new election */
    CUSTOMS_VERSION,	/* Get version number */
    CUSTOMS_JOBS,	/* Get list of imported jobs */
} Customs_Proc;

/*
 * Parameter to CUSTOMS_DEBUG
 */
#define DEBUG_RPC 	1   	/* Debug rpc system */
#define DEBUG_CUSTOMS	2   	/* Debug customs itself */

/*
 * ExportPermits are what the servers use to authenticate exportation of
 * jobs. They are returned to the client from a Customs_Host().
 * CUSTOMS_FAIL may be used to see if the request succeeded. If it did not,
 * CUSTOMS_FAIL(&permit.addr) will be True.
 */
#define CUSTOMS_FAIL(inaddrPtr)	((inaddrPtr)->s_addr == INADDR_ANY)

typedef struct {
    struct in_addr	addr;	    	/* Address of host */
    u_long    	  	id;	  	/* Authentication ID to give it */
} ExportPermit;


/*
 * Host_Data is what is passed to the Customs_Host() function. It contains the
 * UID under which the job will be exported (which must remain constant) and
 * a word of flags indicating criteria to use to determine which host to use.
 */
typedef struct {
    u_short   	  	uid;
    u_short		flags;
} Host_Data;
#define EXPORT_ANY	0x0001	    /* Export to any sort of machine */
#define EXPORT_SAME	0x0002	    /* Export only to same sort of machine */
#define EXPORT_USELOCAL	0x0004	    /* Use local host if available */
#define EXPORT_68020	0x0008	    /* Go only to a 68020 (TEMPORARY) */

/*
 * Avail_Data is what is passed to and returned from the Customs_SetAvail()
 * function. changeMask contains a bitwise-OR of AVAIL_IDLE, AVAIL_SWAP,
 * AVAIL_LOAD and AVAIL_IMPORTS, indicating what fields of the Avail_Data
 * structure are valid. On return, changeMask indicates what fields in the
 * request were invalid. If changeMask is 0, everything was accepted.
 * The returned structure contains the current (after the change, if nothing
 * was wrong) criteria.
 * Load averages are sent as a fixed-point number. The location of the
 * decimal point is given by LOADSHIFT. Divide by LOADSCALE to get the
 * appropriate floating-point number.
 */
typedef struct {
    long    	  changeMask;	    /* Parameters to change */
    long    	  idleTime; 	    /* Idle time (in seconds) */
    long    	  swapPct;  	    /* Percentage of free swap space */
    long    	  loadAvg;  	    /* Maximum load average */
    long    	  imports;  	    /* Greatest number of imported processes */
    long    	  procs;  	    /* Minumum no. of free processes */
    long    	  cpuLimit; 	    /* Maximum CPU time (in seconds) */
    long    	  checkUser; 	    /* Level of user access checking */
    long    	  niceLevel; 	    /* Nice increment for imported jobs */
    long    	  evictDelay; 	    /* Grace period before eviction */
} Avail_Data;

#define AVAIL_IDLE	    1
#define MAX_IDLE  	    (60*60)

#define AVAIL_SWAP	    2
#define MAX_SWAP  	    40

#define LOADSHIFT   	    8
#define LOADSCALE   	    (1<<LOADSHIFT)
#define AVAIL_LOAD	    4
#define MIN_LOAD  	    ((int)(0.25*LOADSCALE))

#define AVAIL_IMPORTS	    8
#define MIN_IMPORTS	    1

#define AVAIL_PROC	    16
#define MAX_PROC  	    20

#define AVAIL_CPU	    32
#define MIN_CPU  	    (1*60)
#define GRACE_CPU	    15		/* Grace period after CPU time expires
					   (time between SIGXCPU and abort) */
#define AVAIL_CHECK         64
#define DONT_CHECK          0		/* Be very generous (or foolish) */
#define PWD_CHECK           1		/* Check for passwd entry */
#define SHELL_CHECK         2		/* Check for authorized shell */
#define LOG_CHECK           3		/* SHELL_CHECK + log violations */
#define MAX_CHECK           LOG_CHECK

#define AVAIL_NICE	    128
#define MAX_NICE  	    20

#define AVAIL_EVICT	    256
#define MIN_EVICT  	    1

#define MIN_CHECK           5		/* Shortest -check interval */

#ifndef EVICT_NOTIFY
#define EVICT_NOTIFY        SIGUSR2	/* Signal to notify imported jobs
					   of impending eviction */
#endif
#ifndef EVICT_SIGNAL
#ifdef SIGXCPU
#define EVICT_SIGNAL        SIGXCPU	/* Signal to evict imported jobs */
#else
#define EVICT_SIGNAL        SIGTERM	/* Signal to evict imported jobs */
#endif
#endif

/*
 * The next few constants are return values, some of them, and are not to
 * be passed to the agent.
 */
#define AVAIL_DOWN	    0x80000000	/* Machine is down -- this is *not* a
					 * parameter */
#define AVAIL_EVERYTHING    (~0)

/*
 * Strings follow the WayBill in the CUSTOMS_IMPORT call arguments as
 * follows:
 *	current-working-directory
 *	file	  	    	    # command to execute
 *	number-o'-arguments 	    # on a 32-bit boundary
 *	argument strings
 *	number-o'-envariables	    # on a 32-bit boundary
 *	environment strings
 *
 * The function Customs_MakeWayBill will create an appropriate buffer...
 */
typedef struct {
    u_long  id;	  	    	/* Identifier returned by the MCA */
    u_short port; 	    	/* UDP Port for callback when process exits */
    long    deadline;		/* Deadline for remote process to start */
    short   ruid;  	    	/* The current real user id */
    short   euid; 	    	/* The current effective user id */
    short   rgid;	    	/* The current real group id */
    short   egid; 	    	/* The current effective group id */
    short   pad;    	    	/* Explicit padding for all architectures */
    long    umask;	    	/* File creation mask */
    long    ngroups;	    	/* Number of groups */
    long    groups[MAX_NUM_GROUPS];	/* Array of group ids */
    long    priority;		/* Process priority */
#ifndef NO_RLIMITS
    struct  rlimit rlimits[MAX_NUM_RLIMITS];	/* Resource limit structures */
#endif
} WayBill;

/*
 * Kill_Data is passed to the CUSTOMS_KILL procedure.
 */
typedef struct {
    u_long  id;	    	    	/* Job ID number (from ExportPermit) */
    long    signo;	    	/* Signal number to deliver */
} Kill_Data;


/*
 * Parameters to the CUSTOMS_EXIT call
 */
typedef struct {
    u_long  	  id;	    	/* ID # of exported job */
    long    	  status;   	/* Exit status */
} Exit_Data;

/*
 * Return values for the CUSTOMS_VERSION call
 */
typedef struct {
    short	majorversion;	/* Major version number */
    short	minorversion;	/* Minor version number */
    short	patchlevel;	/* Patchlevel */
    short	buildno;	/* # of build generating this binary */
    char	builddate[30];	/* String returned by date(1) */
    char	builder[8+1+64];/* login@hostname */
} Version_Data;

/*
 * Data returned by the CUSTOMS_JOBS call
 */
#define MAX_CMD_LEN	(10 * sizeof(long))
typedef struct {
    long	pid;		/* Process (group) id */
    u_long	id;		/* Job identifier */
    long	time;		/* Run time of job */
    short	uid;		/* User id (redundant by id, but convenient */
    short	from_port;	/* Port and ... */
    long	from_addr;	/* ... address job came from */
    char	command[MAX_CMD_LEN];
				/* Command executed */
} Job_Info;
    
/*
 * This is the time within which the daemon is "guaranteed" to respond.
 */
#ifndef CUSTOMS_RETRY
#define CUSTOMS_RETRY	2
#endif
#ifndef CUSTOMS_URETRY
#define CUSTOMS_URETRY	500000
#endif
#ifndef CUSTOMS_NRETRY
#define CUSTOMS_NRETRY	2
#endif

/*
 * Timeout for CUSTOMS_IMPORT call (in secs)
 */
#ifndef CUSTOMS_TIMEOUT
#define CUSTOMS_TIMEOUT	10	/* should be plenty for unloaded machines */
#endif

#define Customs_Align(ptr, type)    (type) (((int)(ptr)+3)&~3)

/*
 * Name under which to look up customs service ports
 */
#ifndef CUSTOMS_SERVICE_NAME
#define CUSTOMS_SERVICE_NAME	"customs"
#endif

/*
 * Default ports if yellow pages f***s us over
 */
#ifndef DEF_CUSTOMS_UDP_PORT
#define DEF_CUSTOMS_UDP_PORT	8231
#endif
#ifndef DEF_CUSTOMS_TCP_PORT
#define DEF_CUSTOMS_TCP_PORT	8231
#endif

/*
 * Rpc front-ends
 */
Rpc_Stat    Customs_Host(),
	    Customs_Ping(),
	    Customs_UseLocal(),
	    Customs_SetAvail(),
	    Customs_AvailInterval(),
	    Customs_Master(),
	    Customs_MasterForHost(),
	    Customs_Info();
void	    Customs_Init();
void	    Customs_PError();
short	    Customs_Port();

/*
 * Return codes from Customs_RawExport, for anyone interested.
 * If > -100, negation of result is Rpc_Stat code from call to local server.
 * if <= -200, -(code + 200) is Rpc_Stat for call to remote server.
 */
#define CUSTOMS_NOEXPORT    -100    	/* Couldn't export */
#define CUSTOMS_NORETURN    -101    	/* Couldn't create return socket */
#define CUSTOMS_NONAME	    -102    	/* Couldn't fetch socket name */
#define CUSTOMS_ERROR	    -104    	/* Remote export error -- message
					 * already printed */
#define CUSTOMS_NOIOSOCK    -105    	/* Couldn't create tcp I/O socket */

#endif _CUSTOMS_H_
