/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

#include "exception.hpp"
#include "cssmerr.h"
#include "misc.hpp"

#include "sign.hpp"
#include "dsasign.hpp"
#include "ibmcylinkcsp.hpp"
#include "random.hpp"

/**********************************************************
 *
 *	Sign class
 *
 **********************************************************/
Sign::Sign(CSSM_CC_HANDLE ctxHandle, uint32 algId)
:Context(ctxHandle, CSSM_ALGCLASS_SIGNATURE, algId) 
{
	switch (algId) {
		//case CSSM_ALGID_DSA :
		case CSSM_ALGID_SHA1WithDSA:
			engine = new DSASignEngine();
			break;
		default:
			throw InputException(CSSM_CSP_UNKNOWN_ALGORITHM);
			break;
	}
}

Sign::~Sign() {
	if (engine != NULL)
		delete engine;
}

void Sign::signInit(Key *key, void *params) {
	//should check for expiration dates also

	if (key == NULL)
		throw InputException(CSSM_CSP_INVALID_KEY_POINTER);
	
	//cannot do it; because CSSM_ALGID_DSA != CSSM_ALGID_SHA1WithDSA
	/*if (key->getAlgId() != this->algId)
		throw InputException(CSSM_CSP_KEY_ALGID_NOTMATCH);*/

	if ((key->getKeyClass() != CSSM_KEYCLASS_PRIVATE_KEY) && 
		(key->getKeyClass() != CSSM_KEYCLASS_PUBLIC_KEY))
		throw InputException(CSSM_CSP_KEY_CLASS_INCORRECT);

	if ((!(key->getUsage() & CSSM_KEYUSE_SIGN)) && (key->getUsage() != CSSM_KEYUSE_ANY))
		throw InputException(CSSM_CSP_KEY_USAGE_INCORRECT);

	engine->signInit(key, params);
}

void Sign::signUpdate(CSSM_DATA_PTR dataBuff, uint32 dataBuffCount) {
	uint32	i;

	if (dataBuff == NULL)
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);
		
	for (i = 0; i < dataBuffCount; i ++) {
		if ((dataBuff[i].Data == NULL) || (dataBuff[i].Length == 0))
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

		CssmData dataIn = CssmData(&dataBuff[i]);
		engine->signUpdate(dataIn);
	}
}

void Sign::signUpdate(CssmData *dataBuff, uint32 dataBuffCount) {
	uint32	i;

	if (dataBuff == NULL)
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);
		
	for (i = 0; i < dataBuffCount; i ++) {
		if (dataBuff[i].getLength() == 0)
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);
					
		engine->signUpdate(dataBuff[i]);
	}
}

void Sign::signFinal(CSSM_DATA_PTR signature) {
	if ((signature == NULL) || (signature->Data == NULL))
		throw InputException(CSSM_CSP_NO_OUTBUF);

	if (signature->Length < getSignatureLength())
		throw InputException(CSSM_CSP_ERR_OUTBUF_LENGTH);

	CssmData dataOut = CssmData(signature);
	engine->signFinal(dataOut);
	memcpy(signature->Data, dataOut.getData(), getSignatureLength());
	signature->Length = dataOut.getLength();
}

void Sign::signFinal(CssmData &signature) {
	if (signature.getLength() < getSignatureLength())
		throw InputException(CSSM_CSP_ERR_OUTBUF_LENGTH);

	engine->signFinal(signature);
}

void Sign::sign(Key *key, void *params, CSSM_DATA_PTR dataBuff, uint32 dataBuffCount, CSSM_DATA_PTR signature) {
	signInit(key, params);
	signUpdate(dataBuff, dataBuffCount);
	signFinal(signature);
}

void Sign::sign(Key *key, void *params, CssmData *dataBuff, uint32 dataBuffCount, CssmData &signature) {
	signInit(key, params);
	signUpdate(dataBuff, dataBuffCount);
	signFinal(signature);
}

void Sign::verifyInit(Key *key, void *params) {
	//should check for expiration dates also

	if (key == NULL)
		throw InputException(CSSM_CSP_INVALID_KEY_POINTER);
		
	//cannot do it; because CSSM_ALGID_DSA != CSSM_ALGID_SHA1WithDSA
	/*if (key->getAlgId() != this->algId)
		throw InputException(CSSM_CSP_KEY_ALGID_NOTMATCH);*/

	if ((key->getKeyClass() != CSSM_KEYCLASS_PUBLIC_KEY) && 
		(key->getKeyClass() != CSSM_KEYCLASS_PRIVATE_KEY))
		throw InputException(CSSM_CSP_KEY_CLASS_INCORRECT);
	
	if ((!(key->getUsage() & CSSM_KEYUSE_VERIFY)) && (key->getUsage() != CSSM_KEYUSE_ANY))
		throw InputException(CSSM_CSP_KEY_USAGE_INCORRECT);

	engine->verifyInit(key, params);
}

void Sign::verifyUpdate(CSSM_DATA_PTR dataBuff, uint32 dataBuffCount) {
	uint32	i;

	if (dataBuff == NULL)
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

	for (i = 0; i < dataBuffCount; i ++) {
		if ((dataBuff[i].Data == NULL) || (dataBuff[i].Length == 0))
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

		CssmData dataIn = CssmData(&dataBuff[i]);
		engine->verifyUpdate(dataIn);
	}
}

void Sign::verifyUpdate(CssmData *dataBuff, uint32 dataBuffCount) {
	uint32	i;

	if (dataBuff == NULL)
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

	for (i = 0; i < dataBuffCount; i ++) {
		if (dataBuff[i].getLength() == 0)
			throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

		engine->verifyUpdate(dataBuff[i]);
	}
}

CSSM_BOOL Sign::verifyFinal(CSSM_DATA_PTR signature) {
	if ((signature == NULL) || (signature->Data == NULL))
		throw InputException(CSSM_CSP_NO_OUTBUF);

	if (signature->Length < getSignatureLength())
		throw InputException(CSSM_CSP_ERR_OUTBUF_LENGTH);
	
	CssmData dataIn = CssmData(signature);
	return (engine->verifyFinal(dataIn));
}

CSSM_BOOL Sign::verifyFinal(CssmData &signature) {
	if (signature.getLength() == 0)
		throw InputException(CSSM_CSP_ERR_OUTBUF_LENGTH);
		
	return (engine->verifyFinal(signature));
}

CSSM_BOOL Sign::verify(Key *key, void *params, CSSM_DATA_PTR dataBuff,
			uint32 dataBuffCount, CSSM_DATA_PTR signature) 
{
	verifyInit(key, params);
	verifyUpdate(dataBuff, dataBuffCount);
	return(verifyFinal(signature));
}

CSSM_BOOL Sign::verify(Key *key, void *params, CssmData *dataBuff,
			uint32 dataBuffCount, CssmData &signature) 
{
	verifyInit(key, params);
	verifyUpdate(dataBuff, dataBuffCount);
	return(verifyFinal(signature));
}

uint32 Sign::getSignatureLength() {
	return(engine->getSignatureLength());
}

/**********************************************************
 *
 *	SignEngine class
 *
 **********************************************************/
SignEngine::SignEngine(uint32 algId) {
	this->algId = algId;
}

SignEngine::~SignEngine() {
}
