/* ***************************************************************** *
 * Copyright 1998 International Business Machines Corporation. All   *
 * Rights Reserved.                                                  *
 *                                                                   *
 * Please read this carefully.  Your use of this reference           *
 * implementation of certain of the IETF public-key infrastructure   *
 * specifications ("Software") indicates your acceptance of the      *
 * following.  If you do not agree to the following, do not install  *
 * or use any of the Software.                                       *
 *                                                                   *
 * Permission to use, reproduce, distribute and create derivative    *
 * works from the Software ("Software Derivative Works"), and to     *
 * distribute such Software Derivative Works is hereby granted to    *
 * you by International Business Machines Corporation ("IBM").  This *
 * permission includes a license under the patents of IBM that are   *
 * necessarily infringed by your use of the Software as provided by  *
 * IBM.                                                              *
 *                                                                   *
 * IBM licenses the Software to you on an "AS IS" basis, without     *
 * warranty of any kind.  IBM HEREBY EXPRESSLY DISCLAIMS ALL         *
 * WARRANTIES OR CONDITIONS, EITHER EXPRESS OR IMPLIED, INCLUDING,   *
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OR CONDITIONS OF       *
 * MERCHANTABILITY, NON INFRINGEMENT AND FITNESS FOR A PARTICULAR    *
 * PURPOSE.  You are solely responsible for determining the          *
 * appropriateness of using this Software and assume all risks       *
 * associated with the use of this Software, including but not       *
 * limited to the risks of program errors, damage to or loss of      *
 * data, programs or equipment, and unavailability or interruption   *
 * of operations.                                                    *
 *                                                                   *
 * IBM WILL NOT BE LIABLE FOR ANY DIRECT DAMAGES OR FOR ANY SPECIAL, *
 * INCIDENTAL, OR  INDIRECT DAMAGES OR FOR ANY ECONOMIC              *
 * CONSEQUENTIAL DAMAGES (INCLUDING LOST PROFITS OR SAVINGS), EVEN   *
 * IF IBM HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.  IBM  *
 * will not be liable for the loss of, or damage to, your records or *
 * data, or any damages claimed by you based on a third party claim. *
 *                                                                   *
 * IBM wishes to obtain your feedback to assist in improving the     *
 * Software.  You grant IBM a world-wide, royalty-free right to use, *
 * copy, distribute, sublicense and prepare derivative works based   *
 * upon any feedback, including materials, error corrections,        *
 * Software Derivatives, enhancements, suggestions and the like that *
 * you provide to IBM relating to the Software (this does not        *
 * include products for which you charge a royalty and distribute to *
 * IBM under other terms and conditions).                            *
 *                                                                   *
 * You agree to distribute the Software and any Software Derivatives *
 * under a license agreement that: 1) is sufficient to notify all    *
 * licensees of the Software and Software Derivatives that IBM       *
 * assumes no liability for any claim that may arise regarding the   *
 * Software or Software Derivatives, and 2) that disclaims all       *
 * warranties, both express and implied, from IBM regarding the      *
 * Software and Software Derivatives.  (If you include this          *
 * Agreement with any distribution of the Software or Software       *
 * Derivatives you will have met this requirement.)  You agree that  *
 * you will not delete any copyright notices in the Software.        *
 *                                                                   *
 * This Agreement is the exclusive statement of your rights in the   *
 * Software as provided by IBM.   Except for the rights granted to   *
 * you in the second paragraph above, You are not granted any other  *
 * patent rights, including but not limited to the right to make     *
 * combinations of the Software with products that infringe IBM      *
 * patents. You agree to comply with all applicable laws and         *
 * regulations, including all export and import laws and regulation. *
 * This Agreement is governed by the laws of the State of New York.  *
 * This Agreement supersedes all other communications,               *
 * understandings or agreements we may have had prior to this        *
 * Agreement.                                                        *
 * ***************************************************************** */

#include "exception.hpp"
#include "cssmerr.h"
#include "misc.hpp"

#include "ibmcylinkcsp.hpp"
#include "dsakeygen.hpp"
#include "toolkit.h"
#include <iostream.h>

/**********************************************************
 *
 *	DSAKeyGenEngine class
 *
 **********************************************************/
DSAKeyGenEngine::DSAKeyGenEngine(Random* randSource)
:KeyGenEngine(CSSM_ALGID_DSA, randSource)
{
}

DSAKeyGenEngine::~DSAKeyGenEngine() {
}

void DSAKeyGenEngine::init(CssmData &seed, void *params) {
	KeyGenEngine::init(seed, NULL);
}

AlgorithmParams* DSAKeyGenEngine::generateParams(void *paramsIn) {
	CssmData	rVal = CssmData(SHA_LENGTH);
	CssmData	q = CssmData(SHA_LENGTH);
	ushort		length;
	DSAParams	*result = NULL;
	int			rc;

	if (paramsIn == NULL)
		throw InputException(CSSM_CSP_INPUT_DATA_ERROR);

	if ((*(ushort*)paramsIn < DSS_LENGTH_MIN) ||
		(*(ushort*)paramsIn > DSS_LENGTH_MAX))
	{
		throw InputException(CSSM_CSP_KEY_LENGTH_UNSUPPORTED);
	}

	length = *(ushort*)paramsIn;
	CssmData p = CssmData(length);
	CssmData g = CssmData(length);

	randSource->generateRandom(rVal);
	rc = GenDSSParameters(length, p.getData(), q.getData(), g.getData(), rVal.getData(), NULL);
	if (rc != SUCCESS) {
		throw InternalException(CSSM_CSP_KEYGEN_FAILED);
	}

	result = new DSAParams(p, q, g);

	return(result);
}


/*void DSAKeyGenEngine::generateParams(Key* key, void *paramsIn) {
	DSAParams	*params = (DSAParams*)generateParams(paramsIn);

	DSAPublicKey* tmpKeyMaterial = (DSAPublicKey*)(key->exportKeyMat());
	if ((tmpKeyMaterial == NULL) || (tmpKeyMaterial->getAlgId() != this->algId))
		throw ObjInitException(CSSM_CSP_KEY_BAD_KEY);
	
	tmpKeyMaterial->importAlgParams(params);
	delete params;
	key->importKeyMat(tmpKeyMaterial);
	delete tmpKeyMaterial;
}*/

void DSAKeyGenEngine::generateKeyMat(Key *key, AlgorithmParams* algParams) {
	KeyMaterial *tmpKeyMat;

	if (key == NULL)
		throw InputException(CSSM_CSP_INVALID_KEY_POINTER);
	
	if (key->getAlgId() != this->getAlgId())
		throw InputException(CSSM_CSP_KEY_ALGID_NOTMATCH);

	if (key->getKeyClass() != CSSM_KEYCLASS_PRIVATE_KEY)
		throw InputException(CSSM_CSP_INVALID_KEYCLASS);

	tmpKeyMat = instantiateKeyMat(*key);
	generateKeyMat(tmpKeyMat, algParams);
	key->importKeyMat(tmpKeyMat);
	delete tmpKeyMat;
}

/***********************************************************************
 *
 * Uses algParams to generate the key material, and then imports these
 * parameters into the keyMat object.
 *
 ***********************************************************************/
void DSAKeyGenEngine::generateKeyMat(KeyMaterial *keyMat, AlgorithmParams* algParams) {
	int			rc;
	DSAParams*	params = NULL;

	if (keyMat == NULL)
		throw InputException(CSSM_CSP_INVALID_KEY_POINTER);

	if (keyMat->getAlgId() != this->getAlgId())
		throw InputException(CSSM_CSP_KEY_ALGID_NOTMATCH);

	if (keyMat->getKeyClass() != CSSM_KEYCLASS_PRIVATE_KEY)
		throw InputException(CSSM_CSP_INVALID_KEYCLASS);

	//think about how to support default parameters (in which case
	//this pointer can be NULL)
	if (algParams == NULL)
		throw InputException(CSSM_CSP_KEY_NO_PARAM);

	if (algParams->getAlgId() != CSSM_ALGID_DSA)
		throw InputException(CSSM_CSP_KEY_ALGID_NOTMATCH);

	params = (DSAParams*)algParams;
	CssmData	x = CssmData(SHA_LENGTH);
	CssmData	y = CssmData(params->getLength());
	CssmData	rand = CssmData(SHA_LENGTH);

	randSource->generateRandom(rand);
	CssmData &p = params->getP();
	CssmData &q = params->getQ();
	CssmData &g = params->getG();
	rc = GenDSSKey(params->getLength(), p.getData(), q.getData(), g.getData(),
				   x.getData(), y.getData(), rand.getData());

	delete &p;
	delete &q;
	delete &g;

	if (rc != SUCCESS)
		throw CryptoLibException(CSSM_CSP_KEYGEN_FAILED);

	((DSAPublicKey*)keyMat)->importAlgParams(algParams);
	((DSAPublicKey*)keyMat)->setY(y);
	((DSAPrivateKey*)keyMat)->setX(x);
}
