/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 *
 * For copying and distribution information, please see the file
 * <uw-copyright.h>.
 */

#include <uw-copyright.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <stdio.h>
#include <sgtty.h>
#include <strings.h>

#include <plog.h>
#include <pfs.h>
#include <pprot.h>
#include <perrno.h>
#include <pmachine.h>

#define DIR_VNO		     0

extern char	root[];
extern char	shadow[];
extern char	dirshadow[];
extern char	dircont[];
extern char	pfsdat[];

extern char	hostname[];

extern char	*acltypes[];

/*
 * dsrdir - Read the contents of a directory from disk
 *
 *   DSRDIR is used by the directory server to read the contents of
 *   a directory off disk.  It takes the host specific part of the system
 *   level name of the directory as an argument, and a directory structure
 *   which it fills in.
 */
dsrdir(name,magic,dir,ul)
    char	*name;
    int		magic;
    VDIR	dir;
    VLINK	ul;      /* Pointer to the vlink currently being expanded */
	                 /* used for location to insert new union links   */
    {
	FILE 		*vfs_dir;
	char		vfs_dirname[MAX_DIR_LINESIZE];
	char		native_dirname[MAX_DIR_LINESIZE];
	char		rel_filename[MAX_DIR_LINESIZE];
	char		line[MAX_DIR_LINESIZE];
	char		*slash;
	VLINK		cur_link = NULL;
	VLINK		cur_fil;
	ACL		cur_acl = NULL;
	ACL		prev_acl = NULL;
	int		tmp;
	char		truefalse[16];
	int		return_value = DSRDIR_NOT_A_DIRECTORY;

	dir->version = -1;
	dir->magic_no = 0;
	dir->inc_native = 1;		/* Include native if can't read */
	dir->f_info = NULL;
	dir->dacl = NULL;
	dir->next = NULL;
	dir->previous = NULL;

	if(ul == NULL) {               /* Don't clear links if expanding */
	    dir->links = NULL;
	    dir->ulinks = NULL;
	}

	/* Determine name of directory contents */
	strcpy(vfs_dirname,shadow);
	strcat(vfs_dirname,name);
	strcat(vfs_dirname,"/");
	strcat(vfs_dirname,dircont);

	strcpy(native_dirname,name);

	/* NOTE: A temporary inefficient directory format is  */
	/* in use.  For this reason, the code supporting it   */
	/* is also interim code, and does not do any checking */
	/* to make sure that the directory actually follows   */
	/* the format.  This, a munged directory will result  */
	/* in unpredictable results.			      */

	/* Read the contents of the VFS directory */
	if((vfs_dir = fopen(vfs_dirname,"r")) != NULL) {
	    return_value = PSUCCESS;
	    while(fgets(line,MAX_DIR_LINESIZE,vfs_dir)) {
		switch(*line) {
		case 'V':
		    tmp = sscanf(line,"VERSION %d",&(dir->version));
		    if(tmp != 1) {
			dir->version = -1;
			plog(L_DATA_FRM_ERR,0,0,"Bad directory format %s: %s",
			     vfs_dirname,line,0);
		    }
		    break;
		case 'M': /* Magic Number */
		    tmp = sscanf(line,"MAGIC-NUMBER %d",&(dir->magic_no));
		    if(tmp != 1) {
			plog(L_DATA_FRM_ERR,0,0,"Bad directory format %s: %s",
			     vfs_dirname,line,0);
		    }
		    break;
		case 'I':
		    tmp = sscanf(line,"INCLUDE-NATIVE %s",truefalse);
		    dir->inc_native = 1;
		    if((*truefalse == 'F') || (*truefalse == 'f'))
			dir->inc_native = 0;
		    /* REAL-ONLY - Do not include . and .. */
		    if((*truefalse == 'R') || (*truefalse == 'r'))
			dir->inc_native = -1;
		    if(tmp != 1) {
			dir->inc_native = 1;
			plog(L_DATA_FRM_ERR,0,0,"Bad directory format %s: %s",
			     vfs_dirname,line,0);
		    }
		    break;
		case 'A': {
		    char 	a_acltype[MAX_DIR_LINESIZE];
		    char	a_atype[MAX_DIR_LINESIZE];
		    char 	a_rights[MAX_DIR_LINESIZE];
		    char 	a_principals[MAX_DIR_LINESIZE];

		    cur_acl = acalloc();
		    tmp = sscanf(line,"ACL %s %s %s %[^\n]",
				 a_acltype,a_atype,a_rights,a_principals);
		    if(tmp >= 1) {
			for(cur_acl->acetype = 0;acltypes[cur_acl->acetype];
			    (cur_acl->acetype)++) {
			    if(strcmp(acltypes[cur_acl->acetype],a_acltype)==0)
				break;
			}
			if(acltypes[cur_acl->acetype] == NULL)
			    cur_acl->acetype = 0;
		    }
		    else {
			plog(L_DATA_FRM_ERR,0,0,"Bad directory format %s: %s",
			     vfs_dirname,line,0);
			acfree(cur_acl);
			break;
		    }

		    if(tmp >= 2) {
			cur_acl->atype = stcopyr(unquoten(a_atype),
						 cur_acl->atype);
		    }
		    if(tmp >= 3) {
			cur_acl->rights = stcopyr(unquoten(a_rights),
						  cur_acl->rights);
		    }
		    if(tmp >= 4) {
			cur_acl->principals = stcopyr(unquoten(a_principals),
						 cur_acl->principals);
		    }

		    if(prev_acl) {
			prev_acl->next = cur_acl;
			cur_acl->previous = prev_acl;
		    }
		    else if(cur_link) cur_link->acl = cur_acl;
		    else dir->dacl = cur_acl;

		    prev_acl = cur_acl;
		}
		    break;

		case 'L': {
		    char 	l_name[MAX_DIR_LINESIZE];
		    char	l_type[MAX_DIR_LINESIZE];
		    char 	l_htype[MAX_DIR_LINESIZE];
		    char 	l_host[MAX_DIR_LINESIZE];
		    char 	l_ntype[MAX_DIR_LINESIZE];
		    char 	l_fname[MAX_DIR_LINESIZE];

		    prev_acl = NULL; /* So next acl entry is for a new link */

		    cur_link = vlalloc();
		    tmp = sscanf(line,"LINK %s %c %s %d %d %s %s %s %s %d %d",
				 l_name, &(cur_link->linktype), l_type,
				 &(cur_link->dest_exp),&(cur_link->link_exp),
				 l_htype,l_host,
				 l_ntype,l_fname,
				 &(cur_link->version),
				 &(cur_link->f_magic_no));
		    if(tmp >= 10) {
			cur_link->name = stcopy(unquote(l_name));
			stfree(cur_link->type); /* Should punt if same */
			cur_link->type = stcopy(l_type);
			stfree(cur_link->hosttype); /* Should punt if same */
			cur_link->hosttype = stcopy(l_htype);
			cur_link->host = stcopy(l_host);
			stfree(cur_link->nametype); /* Should punt if same */
			cur_link->nametype = stcopy(l_ntype);
			cur_link->filename = stcopy(l_fname);
			/* if ul, then vl_insert will call ul_insert */
			if(ul && cur_link->linktype == 'U') {
			    tmp = ul_insert(cur_link,dir,ul);
			    if(!tmp) ul = cur_link;
			}
			else vl_insert(cur_link,dir,VLI_ALLOW_CONF);
		    }
		    else {
			plog(L_DATA_FRM_ERR,0,0,"Bad directory format %s: %s",
			     vfs_dirname,line,0);
			vlfree(cur_link);
		    }
		}
		    break;

		case 'F': { /* Filter attached to previous link */
		    char	f_type;
		    char 	f_htype[MAX_DIR_LINESIZE];
		    char 	f_host[MAX_DIR_LINESIZE];
		    char 	f_ntype[MAX_DIR_LINESIZE];
		    char        f_fname[MAX_DIR_LINESIZE];
		    int		f_vers;
		    int		f_mno;
		    char 	f_args[MAX_DIR_LINESIZE];

		    if(!cur_link) {
			plog(L_DATA_FRM_ERR,0,0, "No reference to previous link.  Can't add filter %s: %s",
			     vfs_dirname,line,0);
		       break; }

		    cur_fil = vlalloc();

		    f_vers = 0; f_mno = 0;
		    tmp = sscanf(line,"FILTER %c %s %s %s %s %d %d ARGS ' %[^']",
				 &f_type, f_htype, f_host, f_ntype, f_fname,
				 &f_vers, &f_mno, f_args);
		    if(tmp >= 5) {
			cur_fil->linktype = f_type;
			cur_fil->hosttype = stcopyr(f_htype,cur_fil->hosttype);
			cur_fil->host = stcopy(f_host);
			stfree(cur_fil->nametype); /* Should punt if same */
			cur_fil->nametype = stcopy(f_ntype);
			cur_fil->filename = stcopy(f_fname);
			cur_fil->version = f_vers;
			cur_fil->f_magic_no = f_mno;
			if(tmp == 8) cur_fil->args = stcopy(f_args);

			fl_insert(cur_fil,cur_link);
		    }
		    else {
			plog(L_DATA_FRM_ERR,0,0,"Bad directory format %s: %s",
			     vfs_dirname,line,0);
			vlfree(cur_fil);
		    }
		}
		}
	    }

	    fclose(vfs_dir);
	}
	/* Note that this directory is entirely native */
	else (dir->inc_native = 2);

	/* Read the contents of the native directory */
	if(dir->inc_native) {
	    DIR			*dirp;
	    struct direct	*dp;

	    if(dirp = opendir(native_dirname)) {
		return_value = PSUCCESS;

		for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)) {
		    strcpy(rel_filename,native_dirname);

		    /* We must special case unix concept of . and .. */
		    /* if we want to eliminate useless parts of path */
		    /* If the filename is "..", then                 */
		    if(!strcmp(dp->d_name,"..")) {
			if(dir->inc_native < 0) continue;
			slash = rindex(rel_filename,'/');
			if(slash) *slash = NULL;
			else strcpy(rel_filename,"/");
			if(!(*rel_filename)) strcpy(rel_filename,"/");
		    }
		    /* Else if ".", do nothing.  If not "." add to   */
		    /* rel_filename					 */
		    else if(strcmp(dp->d_name,".")) {
			if(*(native_dirname + strlen(native_dirname) - 1)!='/')
			    strcat(rel_filename,"/");
			strcat(rel_filename,dp->d_name);
		    }
		    /* If ".", must still decide if we include it */
		    else if(dir->inc_native < 0) continue;
		    

		    cur_link = vlalloc();

		    cur_link->name = stcopy(dp->d_name);
		    cur_link->filename = stcopy(rel_filename);
		    cur_link->host = stcopy(hostname);
		    cur_link->linktype = 'N';
		    
		    vl_insert(cur_link,dir,VLI_ALLOW_CONF);
		}
		closedir(dirp);
	    }
	}
	if((return_value != PSUCCESS) || (dir->magic_no != magic)) {
	    PFILE		dfi = pfalloc();

	    tmp = dsrfinfo(name,magic,dfi);
	    if(tmp == DSRFINFO_FORWARDED) {
		dir->f_info = dfi;
		return(DSRFINFO_FORWARDED);
	    }
	    else if((dir->magic_no < 0) && dfi->forward) {
		dir->f_info = dfi;
		return(DSRFINFO_FORWARDED);
	    }
	    pffree(dfi);
	}
	return(return_value);
    }

/*
 * dswdir - Write directory contents to disk
 *
 */
dswdir(name,dir)
    char	*name;
    VDIR	dir;
    {

	char		vfs_dirname[MAX_DIR_LINESIZE];
	FILE 		*vfs_dir;
	VLINK		cur_link;
	ACL		cur_acl;
	char		qatype[MAX_DIR_LINESIZE];
	char		qrights[MAX_DIR_LINESIZE];
	char		qprin[MAX_DIR_LINESIZE];


	/* if a VFS directory, then create it if necessary */
	if(!strncmp(pfsdat,name,strlen(pfsdat)))
	    mkdirs(name);

	/* Determine name of shadow */
	strcpy(vfs_dirname,shadow);
	strcat(vfs_dirname,name);

	/* Create the shadow directory if necessary */
	mkdirs(vfs_dirname);

	/* Determine name of directory contents */
	strcat(vfs_dirname,"/");
	strcat(vfs_dirname,dircont);


	/* NOTE: A temporary inefficient directory format is  */
	/* in use.  For this reason, the code supporting it   */
	/* is also interim code, and does not do any checking */
	/* to make sure that the directory actually follows   */
	/* the format.  This, a munged directory will result  */
	/* in unpredictable results.			      */

	/* Write the contents of the VFS directory */
	if((vfs_dir = fopen(vfs_dirname,"w")) != NULL) {
	    VLINK flt;
	    fprintf(vfs_dir,"VERSION %d\n",DIR_VNO);
	    fprintf(vfs_dir,"INCLUDE-NATIVE ");
	    if(dir->inc_native == -1)
		fprintf(vfs_dir,"REAL-ONLY\n");
	    else if(dir->inc_native)
		fprintf(vfs_dir,"TRUE\n");
	    else fprintf(vfs_dir,"FALSE\n");

	    cur_acl = dir->dacl;
	    while(cur_acl) {
		strcpy(qatype,quote(cur_acl->atype));
		strcpy(qrights,quote(cur_acl->rights));
		strcpy(qprin,quote(cur_acl->principals));
		fprintf(vfs_dir,"ACL %s %s %s %s\n",acltypes[cur_acl->acetype],
			qatype, qrights, qprin);
		cur_acl = cur_acl->next;
	    }

	    cur_link = dir->links;
	    while(cur_link) {
		if(cur_link->linktype == 'L')
		    fprintf(vfs_dir,"LINK %s L %s %d %d %s %s %s %s %d %d\n",
			    quote(cur_link->name),
			    cur_link->type,cur_link->dest_exp,
			    cur_link->link_exp,cur_link->hosttype,
			    cur_link->host,cur_link->nametype,
			    cur_link->filename,cur_link->version,
			    cur_link->f_magic_no);

		cur_acl = cur_link->acl;
		while(cur_acl) {
		    strcpy(qatype,quote(cur_acl->atype));
		    strcpy(qrights,quote(cur_acl->rights));
		    strcpy(qprin,quote(cur_acl->principals));
		    fprintf(vfs_dir,"ACL %s %s %s %s\n",
			    acltypes[cur_acl->acetype],
			    qatype, qrights, qprin);
		    cur_acl = cur_acl->next;
		}

		flt = cur_link->filters;
		while(flt) {
		    fprintf(vfs_dir,"FILTER %c %s %s %s %s %d %d",
			    flt->linktype,flt->hosttype,flt->host,
			    flt->nametype,flt->filename,
			    flt->version, flt->f_magic_no);
		    if(flt->args) fprintf(vfs_dir," ARGS '%s'",flt->args);
		    fprintf(vfs_dir,"\n");
		    flt = flt->next;
		}
		cur_link = cur_link->next;
	    }

	    cur_link = dir->ulinks;
	    while(cur_link) {
		if(cur_link->linktype == 'U')
		    fprintf(vfs_dir,"LINK %s U %s %d %d %s %s %s %s %d %d\n",
			    quote(cur_link->name),
			    cur_link->type,cur_link->dest_exp,
			    cur_link->link_exp,cur_link->hosttype,
			    cur_link->host,cur_link->nametype,
			    cur_link->filename,cur_link->version,
			    cur_link->f_magic_no);
		flt = cur_link->filters;
		while(flt) {
		    fprintf(vfs_dir,"FILTER %c %s %s %s %s",
			flt->linktype,flt->hosttype,flt->host,
			flt->nametype,flt->filename);
		    if(flt->args) fprintf(vfs_dir," ARGS '%s'",flt->args);
		    fprintf(vfs_dir,"\n");
		    flt = flt->next;
		}
		cur_link = cur_link->next;
	    }
	    
	    fclose(vfs_dir);
	    return(PSUCCESS);
	}
	else return(PFAILURE);
    }
