/*
 * Copyright (c) 1991 by the University of Washington
 *
 * For copying and distribution information, please see the file
 * <uw-copyright.h>.
 */

#include <uw-copyright.h>

#include <strings.h>

#include <pfs.h>
#include <psite.h>
#include <pprot.h>
#include <pauthent.h>
#include <plog.h>
#include <pmachine.h>

#ifndef IPPORT_RESERVED
#define IPPORT_RESERVED 1024
#endif

static	ACL_ST default_iacl = {ACL_DEFAULT, NULL};
static	ACL_ST system_iacl = {ACL_SYSTEM, NULL};

#ifdef DEFAULT_ACL_1
static	ACL_ST default_acl1 = DEFAULT_ACL_1;
#endif

#ifdef DEFAULT_ACL_2
static	ACL_ST default_acl2 = DEFAULT_ACL_2;
#endif

#ifdef SYSTEM_ACL_1
static	ACL_ST system_acl1 = SYSTEM_ACL_1;
#endif

#ifdef SYSTEM_ACL_2
static	ACL_ST system_acl2 = SYSTEM_ACL_2;
#endif

#ifdef OVERRIDE_ACL_1
static	ACL_ST override_acl1 = OVERRIDE_ACL_1;
#endif

#ifdef OVERRIDE_ACL_2
static	ACL_ST override_acl2 = OVERRIDE_ACL_2;
#endif

static	int	initialized = 0;

static ACL	default_acl = NULL;
static ACL	system_acl = NULL;
static ACL	override_acl = NULL;
static ACL	nulldir_acl = NULL;

initialize_defaults()
    {
	initialized++;

	/* The default ACL includes the entries from  */
	/* DEFAULT_ACL 1 and 2                        */
#ifdef DEFAULT_ACL_1
	default_acl = &default_acl1;
#endif 
#ifdef DEFAULT_ACL_2
	default_acl2.previous = &default_acl1;
	default_acl1.next = &default_acl2;
#endif

	/* The system ACL includes the entries from   */
	/* SYSTEM_ACL 1 and 2                         */
#ifdef SYSTEM_ACL_1
	system_acl = &system_acl1;
#endif 
#ifdef SYSTEM_ACL_2
	system_acl2.previous = &system_acl1;
	system_acl1.next = &system_acl2;
#endif

	/* The override ACL includes the entries from */
	/* OVERRIDE_ACL 1 and 2                       */
#ifdef OVERRIDE_ACL_1
	override_acl = &override_acl1;
#endif 
#ifdef OVERRIDE_ACL_2
	override_acl2.previous = &override_acl1;
	override_acl1.next = &override_acl2;
#endif

	nulldir_acl = &default_iacl;
	default_iacl.next = &system_iacl;
	system_iacl.previous = &default_iacl;

    }

/*
 * check_acl - check access control list
 *
 *     CHECK_ACL checks an access control list to see if a particular
 *     user is authorized to perform a particular operation.  It returns
 *     a positive number if the operation is authorized, and zero if
 *     not authorized.  CHECK_ACL actually checks up to three access
 *     control lists.  First, the access control list associated with
 *     a link is checked (if specified).  If the operation would not
 *     be authorized, the directory ACL is checked to see if it overrides
 *     the individual entry.  If sill not authorized,  an override ACL is 
 *     checked to see if the operation is performed.
 *
 *     NOTE on negative rights:  Negative rights apply within
 *     a particular access control list only.  Thus, a negative
 *     entry in the link ACL can override other entries in the
 *     link ACL, but it will not prevent access if the user
 *     is authorized to perform the operation by the directory
 *     or override ACL's.
 */
check_acl(dacl,lacl,id,op)
    ACL		dacl, lacl;  /* Directory and link ACLs          */
    char	*op;	     /* Operation                        */
    CINFO	id;	     /* Client identification            */
    {
	int	answer = 0;

	if(!initialized) initialize_defaults();

	if(!dacl) dacl = nulldir_acl;

	/* Check link ACL or dir ACL as link if link ACL is NULL */
	if(!lacl) answer = checkl_acl_r(dacl,dacl,id,op,ACL_NESTING,2);
	else answer = checkl_acl_r(lacl,dacl,id,op,ACL_NESTING,0);

	if(answer) return(answer);
	
	/* Check to see if directory overrides */
	answer = checkl_acl_r(dacl,dacl,id,op,ACL_NESTING,1);

	if(answer) return(answer);

	/* Check to see if absolute override applies */
	return(checkl_acl_r(override_acl,override_acl,id,op,ACL_NESTING,2));

    }


static checkl_acl_r(acl,dacl,id,op,depth,ld)
    ACL		acl;  	     /* Access control list              */
    ACL		dacl;  	     /* Directory access control list    */
    char	*op;	     /* Operation                        */
    CINFO	id;	     /* Client identification            */
    int		depth;	     /* Maximum nesting                  */
    int		ld;	     /* 0 = link, 1 = dir, 2 = both      */
    {
	int	retval = 0;    /* Would this entry authorize op  */
	int	answer = 1;    /* How to answer if match         */

	if(depth == 0) return(NOT_AUTHORIZED);

	while(acl) {
	    retval = check_permissions(op,acl->rights,ld);
	    if(retval||((acl->rights==NULL)&&((acl->acetype == ACL_DEFAULT)
  					 ||(acl->acetype == ACL_SYSTEM)
					 ||(acl->acetype == ACL_DIRECTORY)))) {
		if(retval == NEG_AUTHORIZED) answer = NOT_AUTHORIZED;
		else answer = AUTHORIZED;

		switch(acl->acetype) {

		case ACL_NONE:
		    break;
		case ACL_DEFAULT:
		    retval = checkl_acl_r(default_acl,dacl,id,op,depth-1,ld);
		    if(retval) return(answer);
		    break;
		case ACL_SYSTEM:
		    retval = checkl_acl_r(system_acl,dacl,id,op,depth-1,ld);
		    if(retval) return(answer);
		    break;
		case ACL_OWNER:
		    /* Check if user is the owner of the dirtectory */
#ifdef NOTDEF
		    /* We need to find the owner of the file/directory  */
		    /* for which this ACL applies, but unfortunately we */
		    /* don't even have the name of the file/directory   */
		    ownacl->acetype = ACL_ASRTHOST;
		    ownacl->atype = NULL;
		    ownacl->rights = acl->rights;
		    sprintf(ownaclst,"%s@%%.%%.%%.%%"/*,dirowner*/);
		    ownacl->principals = ownaclst;
		    ownacl->restrictions = acl->restrictions;
#endif NOTDEF
		    break;
		case ACL_DIRECTORY:
		    retval = checkl_acl_r(dacl,dacl,id,op,depth-1,ld);
		    if(retval) return(answer);
		    break;
		case ACL_ANY:
		    return(answer);
		case ACL_AUTHENT: /* Not yet implemented */
		    break;
		case ACL_LGROUP: /* Not yet implemented */
		    break;
		case ACL_GROUP: /* Not yet implemented */
		    break;
		case ACL_ASRTHOST:  /* Check host and asserted userid */
		    retval = check_assertion(acl,id,0);
		    if(retval) return(answer);
		    break;
		case ACL_TRSTHOST:  /* Check host and userid */
		    retval = check_assertion(acl,id,1);
		    if(retval) return(answer);
		    break;
		default: /* Not implemented */
		    break;
		}
	    }
	    acl = acl->next;
	}
	return(NOT_AUTHORIZED);
    }

/* 
 * check_permissions - Check if operation is authorized
 *
 *           CHECK_PERMISIONS takes a string with letters representing
 *           the permissions required for the current opration, and
 *           a string listing operating authorized by the current ACL
 *           entry.  It returns a 1 if the operation would be
 *           authorized by the current entry, a 0 if not, and a -1
 *           if the ACL entry would have authorized the operation,
 *           but began with a "-" indicating negative authorization.
 *  
 *   ARGS:   op  - String with operations to be performed
 *           p   - Permissions from ACL entry
 *           ld  - Whther ACL entry is for directory or link (or both)
 *
 *   RETURNS: 1 if authorized
 *            0 if not authorized
 *           -1 if negative authorization
 *
 * Protections
 *
 * For directory:                For individual links:      
 *   A Administer                  a Administer           
 *   V View                        v View                 
 *   L List                        l List                 
 *   R Read                        r Read                 
 *   M Modify                      m Modify               
 *   D Delete                      d Delete               
 *   I Insert                               
 *   B Administer                    (does not override link)
 *   Y View                          (does not override link)
 *   > Add rights                  ] Add rights
 *   < Remove rights               [ Remove rights
 *   ) Add rights                    (does not override link)
 *   ( Remove rights                 (does not override link)
 *
 *  When a small letter is associated with a directory, it is the default
 *  used for those links in the directory which do not otherwise specify
 *  protections.  The large letter for a directory indicates that the 
 *  right applies to ALL entries in the directory, regardless of the ACLs
 *  associated with the individual link.  
 *
 *  These rights apply to the directory and individual links.  The ability
 *  to read a link does not grant any rights to read the file that the
 *  link points to.  Instead, it grants the right to read the binding
 *  of the link.  The protection mechanisms for the objects themselves
 *  depend on the underlying access mechanism.
 *
 *  The Administer right is required to change the ACL.  "A" allows one
 *  to change the ACLs for the directory as a whole, as well as those 
 *  for individual links.  It does not, however, grant any rights to 
 *  the object pointed to by those links (e.g. it doesn't allow one
 *  to change the permissions on subdirectories.
 *
 *  List allows one to list an entry in a wildcarded search.  Read allows
 *  one to learn the binding of a link.  If one can read a link, but
 *  not list it, then it can only be seen when the user specifies the
 *  exact name of the link in a query.
 *
 *  Modify allows one to change the binding of a link.  It does not
 *  allow one to create a new link or delete an existing one.  Insert
 *  or delete access are necessary for that. 
 *
 *  View allows one to view the contents of the ACL.  Administer implies view.
 *  Add rights and remove rights, ><][)(, allow one to add or remove the
 *  other rights that are specified as part of the same ACL entry.  It is
 *  a limited form of administer.  The other rights included in the
 *  same ACL entry do not apply, but only restrict which rights may be
 *  added or removed.  The add or remove indicators must appear in the
 *  first one or two positions in the rights field, and can not themselves
 *  be added or removed unless they also appear later in the rights field.
 *
 *  If the permission string begins with a "-" and if the specified operation
 *  would otherwise be authorized, check_permissions returns -1 indicating 
 *  that an applicable negative right has been found, and that the operation
 *  should be denied even if subsequent ACL entries authorizing it are found.
 *  If an ACL entry preceding this one has already authorized the operation,
 *  the operation will be performed.
 *
 *  BUGS: For now, only the first character in ><][])( means add or
 *        delete the following rights, and all rights included in the
 *        entry including the first ><][)( may be added or deleted.
 *        Eventually, we will check the first two positions to see
 *        if adding and deleting is allowed, and the matching
 *        characters in those positions will be removed before 
 *        checking subsequent characters.
 */
check_permissions(op,p,ld)
    char	*op;	/* Operation                        */
    char	*p;	/* Permissions                      */
    int		ld;     /* 0 = link, 1 = directory, 2= both */
    {
	int	polarity = 1; /* -1 = neg authorization */

	if(!p || !(*p)) return(NOT_AUTHORIZED);

	if(*p == '-') polarity = -1;

	if(index("><][)(",*p) && !index("><][",*op))
	    return(NOT_AUTHORIZED);

	while(*op) {
	    switch(*(op++)) {
	    case 'a':
		if((ld != 1) && index(p,'a')) continue;
		if((ld != 0) && index(p,'A')) continue;
		else return(NOT_AUTHORIZED);
	    case 'A': 
		if((ld != 0) && index(p,'A')) continue;
		if((ld != 0) && index(p,'B')) continue;
		else return(NOT_AUTHORIZED);

	    case 'v':
		if((ld != 1) && index(p,'v')) continue;
		if((ld != 0) && index(p,'V')) continue;
		if((ld != 1) && index(p,'a')) continue;
		if((ld != 0) && index(p,'A')) continue;
		else return(NOT_AUTHORIZED);
	    case 'V': 
		if((ld != 0) && index(p,'V')) continue;
		if((ld != 0) && index(p,'Y')) continue;
		if((ld != 0) && index(p,'A')) continue;
		if((ld != 0) && index(p,'B')) continue;
		else return(NOT_AUTHORIZED);

	    case 'l':
		if((ld != 1) && index(p,'l')) continue;
	    case 'L': /* and fall through */
		if((ld != 0) && index(p,'L')) continue;
		else return(NOT_AUTHORIZED);

	    case 'r':
		if((ld != 1) && index(p,'r')) continue;
	    case 'R': /* and fall through */
		if((ld != 0) && index(p,'R')) continue;
		else return(NOT_AUTHORIZED);

	    case 'm':
		if((ld != 1) && index(p,'m')) continue;
	    case 'M': /* and fall through */
		if((ld != 0) && index(p,'M')) continue;
		else return(NOT_AUTHORIZED);

	    case 'd':
		if((ld != 1) && index(p,'d')) continue;
	    case 'D': /* and fall through */
		if((ld != 0) && index(p,'D')) continue;
		else return(NOT_AUTHORIZED);

	    case 'I':
		if((ld != 0) && index(p,'I')) continue;
		else return(NOT_AUTHORIZED);

	    case ']':
		if((ld != 1) && index(p,']')) continue;
		if((ld != 0) && index(p,'>')) continue;
		else return(NOT_AUTHORIZED);
	    case '>': 
		if((ld != 0) && index(p,'>')) continue;
		if((ld != 0) && index(p,')')) continue;
		else return(NOT_AUTHORIZED);

	    case '[':
		if((ld != 1) && index(p,'[')) continue;
		if((ld != 0) && index(p,'<')) continue;
		else return(NOT_AUTHORIZED);
	    case '<': 
		if((ld != 0) && index(p,'<')) continue;
		if((ld != 0) && index(p,'(')) continue;
		else return(NOT_AUTHORIZED);

	    default:
		return(NOT_AUTHORIZED);
	    }
	}
	return(polarity);
    }

static char *pr_inet_ntoa(a)
    long a;
    {
	static char	astring[20];

#if BYTE_ORDER == BIG_ENDIAN
	sprintf(astring,"%d.%d.%d.%d",(a >> 24) & 0xff,
		(a >> 16) & 0xff,(a >> 8) & 0xff, a & 0xff);
#else
	sprintf(astring,"%d.%d.%d.%d", a & 0xff,
		(a >> 8) & 0xff,(a >> 16) & 0xff, (a >> 24) & 0xff);
#endif
	
	return(astring);
    }

static char *inet_def_local(s)
    char	*s;
    {
	static char	adstring[50];
	static long	myaddr = 0;
	char		intstring[10];
	static long		o[4];
	char		*ads;
	long		*octet;

	if(!myaddr) {
	    myaddr = myaddress();
#if BYTE_ORDER == BIG_ENDIAN
	    o[0] = (myaddr >> 24) & 0xff;
	    o[1] = (myaddr >> 16) & 0xff;
	    o[2] = (myaddr >> 8) & 0xff;
	    o[3] = myaddr & 0xff;
#else
	    o[0] = myaddr & 0xff;
	    o[1] = (myaddr >> 8) & 0xff;
	    o[2] = (myaddr >> 16) & 0xff;
	    o[3] = (myaddr >> 24) & 0xff;
#endif
	}

	octet = o;

	ads = adstring;
	while(*s && ((ads - adstring) < 40) ) {
	    switch(*s) {
	    case '.':
		octet++;
		*ads++ = *s;
		break;
	    case '%':
		sprintf(intstring,"%d",*octet);
		bcopy(intstring,ads,strlen(intstring));
		ads += strlen(intstring);
		break;
	    default:
		*ads++ = *s;
		break;
	    }
	    s++;
	}
	*ads++ = '\0';
	return(adstring);

    }

/*
 * check_assertion - check for membership in list of principals
 *                   and check optional host ID
 *
 */
check_assertion(a,id,pr)
    ACL		a;   /* Access control list */
    CINFO	id;
    int		pr;  /* Flag 1 = require privileged port */
    {
	char		pcopy[MAX_DIR_LINESIZE];
	char		*pptr;

	/* If pr, then check to see if from privileged port */
	if(pr && (id->port > IPPORT_RESERVED)) return(FALSE);

	/* First check the authentication type */
	if(a->atype && *(a->atype) &&
	   (strcmp(id->auth_type,a->atype) != 0))
	    return(FALSE);

	strcpy(pcopy,a->principals);

	for(pptr = strtok(pcopy," ,\n");pptr;pptr = strtok(NULL," ,\n")) {
	    char	*host;

	    host = index(pptr,'@');
	    if(host == NULL) {
		if(wcmatch(id->userid,pptr)) return(TRUE);
		else continue;
	    }
	    *(host++) = '\0';

	    if(!wcmatch(id->userid,pptr)) continue;

	    if(index("%123456789.",*host)) { /* Host address */
		if(wcmatch(pr_inet_ntoa(id->haddr),inet_def_local(host)))
		    return(TRUE);
		continue;
	    }
	    else if(*host == '0') { /* Host address with leading 0 */
		if(wcmatch(pr_inet_ntoa(id->haddr),inet_def_local(host+1)))
		    return(TRUE);
		continue;
	    }
	    else { /* Host name - not implemented */
		continue;
	    }
	}

	return(FALSE);
    }


