#include <stdio.h>
#include <sys/types.h>
#include <string.h>

#ifdef MMAP
#include <sys/mman.h>
#endif

#include <ndbm.h>
#include <defines.h>
#include <structs.h>
#include <database.h>

#include <pfs.h>
#include <perrno.h>
#include <plog.h>

#include "prarch.h"

VLINK	atoplink();

char *re_comp();
char *make_lcase();
int get_match_list();

extern char *strings_begin;
extern long strings_table_size;
extern DBM *fast_strings;

#define MATCH_CACHE_SIZE   10

struct match_cache {
    char               *program_name;/* Matched regular expression          */
    int			max_hits;    /* Maximum number of entries           */
    int			offset;      /* Offset                              */
    search_sel 		search_type; /* Search method                       */
    VDIR_ST		vd;          /* Directory with results              */
    int			archiedir;   /* Flag: directory links are to archie */
    struct match_cache 	*next;       /* Next entry in cache                 */
};

static struct match_cache *mcache = NULL;

static int		  cachecount = 0;

/*
 * prarch_match - Search archie database for specified file
 *
 * 	PRARCH_PROG searches the archie database and returns
 *      a list of files matching the provided regular expression
 *      
 *  ARGS:  program_name - regular expression for files to match
 *             max_hits - maximum number of entries to return (max hits)
 *               offset - start the search after this many hits
 *          search_type - search method 
 *                   vd - pointer to directory to be filled in
 *            archiedir - flag - directory links should be to archie
 *
 *   Search method is one of:   S_FULL_REGEX
 *		                S_EXACT 
 *                              S_SUB_NCASE_STR 
 *                              S_SUB_CASE_STR 
 */
int prarch_match(program_name,max_hits,offset,search_type,vd,archiedir)
    char	*program_name; 	/* Regular expression to be matched       */
    int		max_hits;	/* Maximum number of entries to return    */
    int		offset;		/* Skip this many matches before starting */
    search_sel search_type;	/* Search method                          */
    VDIR	vd;		/* Directory to be filled in              */
    int		archiedir;	/* Flag: directory links s/b to archie    */
{
  /*
   * Search the database for the string specified by 'program_name'.  Use the
   * fast dbm strings database if 'is_exact' is set, otherwise search through
   * the strings table.  Stop searching after all matches have been found, or
   * 'max_hits' matches have been found, whichever comes first.  
   */
  char 		s_string[MAX_STRING_LEN];
  char		*strings_ptr;
  char		*strings_curr_off;
  strings_header str_head;
  datum 	search_key, key_value;
  int 		hits_exceeded = FALSE;	/* should be boolean? */
  int 		stop_search = FALSE;
  char 		*strings_end;
  int 		match_number;
#ifdef STRFIND
  int 		patlen;
#endif
  site_out 	**site_outptr;
  site_out 	site_outrec;
  int 		i;
  VLINK		cur_link;

  /* For caching */
  VLINK		tmp_link;
  static struct match_cache *cachep = NULL;
  static struct match_cache *pcachep = NULL;
  static struct match_cache *newresults = NULL;

  if(!program_name || !(*program_name)) return(PRARCH_BAD_ARG);
  strcpy(s_string, program_name);

  /* The caching code assumes we are handed an empty directory */
  /* if not, return an error for now.  Eventually we will get  */
  /* rid of that assumption                                    */
  if(vd->links) {
      plog(L_DIR_INFO, NULL, NULL, "Prarch_match handed non empty dir",0);
      return(PRARCH_BAD_ARG);
  }

  /* Check for cached results */

  cachep = mcache;
  pcachep = NULL;

  while(cachep) {
      if((strcmp(cachep->program_name,program_name) == 0) &&
	 (cachep->max_hits == max_hits) &&
	 (cachep->offset == offset) &&
	 (cachep->search_type == search_type) &&
	 (cachep->archiedir == archiedir)) {
	  /* We have a match.  Move to front of list */
	  if(pcachep) {
	      pcachep->next = cachep->next;
	      cachep->next = mcache;
	      mcache = cachep;
	  }
	  vd->links = cachep->vd.links;
	  plog(L_DIR_INFO, NULL, NULL, "Responding with cached data",0);

	  /* We now have to clear the expanded bits or the links that   */
	  /* have been returned in previous queries will not be retured */
	  cur_link = vd->links;
	  while(cur_link) {
	      tmp_link = cur_link;
	      while(tmp_link) {
		  tmp_link->expanded = FALSE;
		  if(tmp_link == cur_link) tmp_link = tmp_link->replicas;
		  else tmp_link = tmp_link->next;
	      }
	      cur_link = cur_link->next;
	  }
	  return(PRARCH_SUCCESS);
      }
      pcachep = cachep;
      cachep = cachep->next;
  }
  /* Didn't find it, find cache entry to use */
  if(cachecount < MATCH_CACHE_SIZE) { /* Create a new entry */
      newresults = (struct match_cache *) malloc(sizeof(struct match_cache));
      cachecount++;
      newresults->next = mcache;
      mcache = newresults;
      newresults->program_name = stcopy(program_name);
      newresults->max_hits = max_hits;
      newresults->offset = offset;
      newresults->search_type = search_type;
      newresults->archiedir = archiedir;
      vdir_init((&(newresults->vd)));
  }
  else { /* Use last entry */
      newresults = pcachep;

      /* move to front of list */
      newresults->next = mcache;
      mcache = newresults;

      /* Fix the last entry so we don't have a cycle */
      while(pcachep->next != newresults) pcachep = pcachep->next;
      pcachep->next = NULL;

      /* Free the old results */
      if(newresults->vd.links) {
	  newresults->vd.links->dontfree = FALSE;
	  vllfree(newresults->vd.links);
	  newresults->vd.links = NULL;
      }

      newresults->program_name = stcopyr(program_name,newresults->program_name);
      newresults->max_hits = max_hits;
      newresults->offset = offset;
      newresults->search_type = search_type;
      newresults->archiedir = archiedir;
  }

  site_outptr = (site_out **) malloc((unsigned)(sizeof(site_out) * 
						(max_hits + offset)));
  if(!site_outptr) return(PRARCH_OUT_OF_MEMORY);

  strings_ptr = strings_begin;
  strings_end = strings_begin + (int) strings_table_size;

  match_number = 0;

  switch(search_type){

  case S_FULL_REGEX:
	
    if(re_comp(s_string) != (char *)NULL){
      return (PRARCH_BAD_REGEX);
    }

    str_head.str_len = -1;

    check_for_messages();

    while((strings_curr_off = strings_ptr + str_head.str_len + 1) < strings_end){

      strings_ptr = strings_curr_off;

      bcopy(strings_ptr,(char *)&str_head,sizeof(strings_header));

      strings_ptr += sizeof(strings_header);
	    
      if(re_exec( strings_ptr ) == 1 ){ /* TRUE */
	strings_curr_off = strings_ptr;

	if(str_head.filet_index != -1){
	  get_match_list((int) (str_head.filet_index), max_hits + offset, 
			 &match_number, site_outptr, FALSE);
	  check_for_messages();
	  if( match_number >= max_hits + offset ){
	    hits_exceeded = TRUE;
	    stop_search = TRUE;
	  }
	}
      }

      if(stop_search == TRUE){
	stop_search = FALSE;
	break;
      }
    }

    break;

  case S_SUB_NCASE_STR:
  case S_SUB_CASE_STR:
    str_head.str_len = -1;

#ifdef STRFIND
    patlen = strlen(s_string ) ;
    initskip(s_string, patlen, search_type == S_SUB_NCASE_STR) ;
#endif
    while((strings_curr_off = strings_ptr + str_head.str_len + 1) < strings_end){
      strings_ptr = strings_curr_off;

      bcopy(strings_ptr,(char *)&str_head,sizeof(strings_header));

      strings_ptr += sizeof(strings_header);
           
#ifndef STRFIND
      if(strstr(strings_ptr, s_string) != (char *) NULL){
#else
	if(strfind(strings_ptr,str_head.str_len)){
#endif
	  strings_curr_off = strings_ptr;
	  if(str_head.filet_index != -1){
	    get_match_list((int) str_head.filet_index, max_hits + offset, 
			   &match_number, site_outptr, FALSE );
	    check_for_messages();
	    if( match_number >= max_hits + offset){
	      hits_exceeded = TRUE;
	      stop_search = TRUE;
	    }
	  }
	}

	if(stop_search == TRUE){
	  stop_search = FALSE;
	  break;
	}
      }

      break;

    case S_EXACT:

      search_key.dptr = s_string;
      search_key.dsize = strlen(s_string) + 1;

      key_value = dbm_fetch(fast_strings, search_key) ;
      check_for_messages();

      if(key_value.dptr != (char *)NULL){ /* string in table */

	int string_pos;

	bcopy(key_value.dptr,(char *)&string_pos, key_value.dsize);

	strings_ptr += string_pos;

	bcopy(strings_ptr,(char *)&str_head,sizeof(strings_header));

	if(str_head.filet_index != -1){
	  get_match_list((int) str_head.filet_index, max_hits + offset, 
			 &match_number, site_outptr, FALSE);
	}
	check_for_messages();

	if(stop_search == TRUE){
	  stop_search = FALSE;
	  break;
	}

      }
      break;

    default:
      return(PRARCH_BAD_ARG);
      break;
    }

    for(i =  offset;(i !=  match_number) && (stop_search == FALSE); i++){
      site_outrec = *site_outptr[i];
      cur_link = atoplink(site_outrec,archiedir);
      free((char *)site_outptr[i]);
      if(cur_link) vl_insert(cur_link,&(newresults->vd),VLI_ALLOW_CONF);
    }
    free((char *)site_outptr);

    if(hits_exceeded) {
      /* Insert a continuation entry */
    }
    
    if(newresults->vd.links) {
	newresults->vd.links->dontfree = TRUE;
	vd->links = newresults->vd.links;
    }
    return(PRARCH_SUCCESS);
}
