/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 *
 * For copying and distribution information, please see the file
 * <uw-copyright.h>.
 */

#include <uw-copyright.h>
#include <stdio.h>
#include <errno.h>
#include <sys/time.h>
#include <netdb.h>
#include <strings.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <pfs.h>
#include <pprot.h>
#include <pcompat.h>
#include <perrno.h>
#include <pmachine.h>


static int notprived = 0;
extern int errno;
extern int perrno;
extern int pfs_debug;
extern int pfs_disable_flag;

char	*nlsindex();

#define max(X, Y)  ((X) > (Y) ? (X) : (Y))

static char 		*prog = "dirsend";
static int		dir_udp_port = 0;	/* Remote UDP port number */

static int client_dirsrv_timeout = CLIENT_DIRSRV_TIMEOUT;
static int client_dirsrv_retry = CLIENT_DIRSRV_RETRY; 

/*
 * dirsend - send packet and receive response
 *
 *   DIRSEND takes a pointer to a structure of type PTEXT, a hostname,
 *   and a pointer to a host address.  It then sends the supplied
 *   packet off to the directory server on the specified host.  If
 *   hostaddr points to a valid address, that address is used.  Otherwise,
 *   the hostname is looked up to obtain the address.  If hostaddr is a
 *   non-null pointer to a 0 address, then the address will be replaced
 *   with that found in the hostname lookup.
 *
 *   DIRSEND will wait for a response and retry an appropriate
 *   number of times as defined by timeout and retries (both static
 *   variables).  It will collect however many packets form the reply, and
 *   return them in a structure (or structures) of type PTEXT.
 *
 *   BUGS: If multi packet responses come back from two separate attempts
 *         at sending a request, the packets may be intermixed (if packets
 *         had been lost for example)  As long as both responses are identical
 *         this is fine, but if they are different, confusion will result.
 */
PTEXT
dirsend(pkt,hostname,hostaddr)
    PTEXT pkt;
    char *hostname;
    struct sockaddr_in	*hostaddr;
{
    PTEXT		first = NULL;	/* First returned packet	 */
    PTEXT		next;		/* The one we are waiting for 	 */
    PTEXT		vtmp;           /* For reorganizing linked list  */
    PTEXT		comp_thru;	/* We have all packets though    */
    int			lp = -1;	/* Opened UDP port	         */
    int			hdr_len;	/* Header Length                 */
    int			nd_pkts;	/* Number of packets we want     */
    int			no_pkts;	/* Number of packets we have     */
    int			pkt_cid;        /* Connection identifier         */
    char		*seqtxt;	/* Pointer to text w/ sequence # */
    struct sockaddr_in  us;		/* Our address                   */
    struct sockaddr_in	to;		/* Address to send query	 */
    struct sockaddr_in	from;		/* Reply received from		 */
    int			from_sz;	/* Size of from structure	 */
    struct hostent	*host;		/* Host infor from gethostbyname */
    int			ns;		/* Number of bytes actually sent */
    int			nr;		/* Number of bytes received      */
    fd_set		readfds;	/* Used for select		 */
    int			tmp;
    char		*ctlptr;	/* Pointer to control field      */
    short		stmp;		/* Temp short for conversions    */
    int			backoff;	/* Server requested backoff      */

    struct timeval	timeout;
    int			retries = client_dirsrv_retry;

    timeout.tv_sec = client_dirsrv_timeout;
    timeout.tv_usec = 0;

    comp_thru = NULL;
    perrno = 0;
    nd_pkts = 0;
    no_pkts = 0;
    pkt_cid = 0;

    /* If necessary, find out what udp port to send to */
    if (dir_udp_port == 0) {
        register struct servent *sp;
	tmp = pfs_enable; pfs_enable = PMAP_DISABLE;
        if ((sp = getservbyname("dirsrv","udp")) == 0) {
	    if (pfs_debug)
		fprintf(stderr, "%s: udp/dirsrv unknown service - using %d\n", 
			prog,DIRSRV_PORT);
	    dir_udp_port = htons((u_short) DIRSRV_PORT);
        }
	else dir_udp_port = sp->s_port;
	pfs_enable = tmp;
        if (pfs_debug > 3)
            fprintf(stderr,"dir_udp_port is %d\n", ntohs(dir_udp_port));
    }

    /* If we were given the host address, then use it.  Otherwise  */
    /* lookup the hostname.  If we were passed a host address of   */
    /* 0, we must lookup the host name, then replace the old value */
    if(!hostaddr || (hostaddr->sin_addr.s_addr == 0)) {
	/* I we have a null host name, return an error */
	if((hostname == NULL) || (*hostname == '\0')) {
            if (pfs_debug)
                fprintf(stderr, "%s: Null hostname specified\n",prog);
	    perrno = DIRSEND_BAD_HOSTNAME;
            return(NULL);
	}
	tmp = pfs_enable; pfs_enable = PMAP_DISABLE;
        if((host = gethostbyname(hostname)) == NULL) {
            if (pfs_debug)
                fprintf(stderr, "%s: Can't resolve host %s\n",prog,hostname);
	    perrno = DIRSEND_BAD_HOSTNAME;
	    pfs_enable = tmp;
            return(NULL);
	}
	pfs_enable = tmp;
	bzero((char *)&to, S_AD_SZ);
        to.sin_family = host->h_addrtype;
        bcopy(host->h_addr, (char *)&to.sin_addr, host->h_length);
	if(hostaddr) bcopy(&to,hostaddr, S_AD_SZ);
    }
    else bcopy(hostaddr,&to, S_AD_SZ);
    to.sin_port = dir_udp_port;

    /* Must open a new port each time. we do not want to see old */
    /* responses to messages we are done with                    */
    if ((lp = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        if (pfs_debug)
            fprintf(stderr,"%s: Can't open socket\n", prog);
	perrno = DIRSEND_UDP_CANT;
        return(NULL);
    }

    /* Try to bind it to a privileged port - loop through candidate */
    /* ports trying to bind.  If failed, that's OK, we will let the */
    /* system assign a non-privileged port later                    */
    if(!notprived) {
	for(tmp = PROS_FIRST_PRIVP; tmp < PROS_FIRST_PRIVP+PROS_NUM_PRIVP; 
	    tmp++) {
	    bzero((char *)&us, sizeof(us));
	    us.sin_family = AF_INET;
	    us.sin_port = htons((u_short) tmp);
	    if (bind(lp, (struct sockaddr *)&us, sizeof(us))) {
		if(errno != EADDRINUSE) {
		    notprived++;
		    break;
		}
	    }
	    else break;
	}
    }

    if (pfs_debug > 2) {
        if (to.sin_family == AF_INET)
            fprintf(stderr,"Sending message to %s...", inet_ntoa(to.sin_addr));
        else
            fprintf(stderr,"Sending message...");
        (void) fflush(stderr);
    }

    first = ptalloc();
    next = first;

 retry:

    ns = sendto(lp,(char *)(pkt->start), pkt->length, 0, &to, S_AD_SZ);
    if(ns != pkt->length) {
	if (pfs_debug) {
	    fprintf(stderr,"\nsent only %d/%d: ",ns, pkt->length);
	    perror("");
	}
	close(lp);
	perrno = DIRSEND_NOT_ALL_SENT;
	ptlfree(first);
        return(NULL);
    }

    if (pfs_debug > 2) fprintf(stderr,"Sent.\n");

    /* We come back to this point (by a goto) if the packet */
    /* received is only part of the response, or if the     */
    /* response came from the wrong host		    */

 keep_waiting:	

    if (pfs_debug > 2) fprintf(stderr,"Waiting for reply...");

    FD_ZERO(&readfds);
    FD_SET(lp, &readfds);

    /* select - either recv is ready, or timeout */
    /* see if timeout or error or wrong descriptor */
    tmp = select(lp + 1, &readfds, (fd_set *)0, (fd_set *)0, &timeout);
    if((tmp == 0) && (retries-- > 0)) {
	timeout.tv_sec = CLIENT_DIRSRV_BACKOFF(timeout.tv_sec);
	if (pfs_debug > 2) {
            fprintf(stderr,"Timed out.  Setting timeout to %d seconds.\n",
		    timeout.tv_sec);
	}
	goto retry;
    }

    if((tmp < 1) || !FD_ISSET(lp, &readfds)) {
	if (pfs_debug) {
	    fprintf(stderr, "select failed: readfds=%x ",
                    readfds);
            perror("");
        }
	close(lp);
	perrno = DIRSEND_SELECT_FAILED;
	ptlfree(first);
        return(NULL);
    }


    from_sz = sizeof(from);
    next->start = next->dat;
    if ((nr = recvfrom(lp, next->start, BUFSIZ, 0, &from, &from_sz)) < 0) {
        if (pfs_debug) perror("recvfrom");
	close(lp);
	perrno = DIRSEND_BAD_RECV;
	ptlfree(first);
        return(NULL);
    }

    next->length = nr;
    
    *(next->start + next->length) = NULL; 

    /* In preparation for a future change, if the first byte is less than */
    /* 20, then the first two bits are a version number and the next six  */
    /* are the header length (including the first byte)                   */
    /* process the new header format then go to*/
    /* the header.                                                        */
    if((hdr_len = (unsigned char) *(next->start)) < 20) {
	ctlptr = next->start + 1;
	next->seq = 0;
	if(hdr_len >= 3) { 	/* Connection ID */
	    bcopy(ctlptr,&stmp,2);
	    if(stmp) pkt_cid = ntohs(stmp);
	    ctlptr += 2;
	}
	if(hdr_len >= 5) {	/* Packet number */
	    bcopy(ctlptr,&stmp,2);
	    next->seq = ntohs(stmp);
	    ctlptr += 2;
	}
	else { /* No packet number specified, so this is the only one */
	    next->seq = 1;
	    nd_pkts = 1;
	}
	if(hdr_len >= 7) {	    /* Total number of packets */
	    bcopy(ctlptr,&stmp,2);  /* 0 means don't know      */
	    if(stmp) nd_pkts = ntohs(stmp);
	    ctlptr += 2;
	}
	if(hdr_len >= 9) {	/* Receievd through */
	    /* Not supported by clients */
	    ctlptr += 2;
	}
	if(hdr_len >= 11) {	/* Backoff */
	    bcopy(ctlptr,&stmp,2);
	    if(stmp) {
		backoff = ntohs(stmp);
		if(pfs_debug > 2) 
		    fprintf(stderr,"Backing off to %d seconds\n", backoff);
		timeout.tv_sec = backoff;
	    }
	    ctlptr += 2;
	}
	if(next->seq == 0) goto keep_waiting;
	next->length -= hdr_len;
	next->start += hdr_len;
	goto done_old;
    }

    if (pfs_debug > 2) 
        fprintf(stderr,"Received packet from %s\n", inet_ntoa(from.sin_addr));

    if (to.sin_addr.s_addr != from.sin_addr.s_addr) {
	if (pfs_debug) {
	    fprintf(stderr, "dirsend: received packet from wrong host! (%x)\n",
		    from.sin_addr.s_addr);
	    (void) fflush(stdout);
	}
	goto keep_waiting;
    }

    pkt_cid = 0;

    /* if new format, then process and go to done_old */
    ctlptr = next->start + max(0,next->length-20);
    while(*ctlptr) ctlptr++;
    /* Control fields start after the terminating null */
    ctlptr++;
    /* Until old version are gone, must be 4 extra bytes minimum */
    /* When no version 3 servers, can remove the -4              */
    if(ctlptr < (next->start + next->length - 4)) {
	/* Connection ID */
	bcopy(ctlptr,&stmp,2);
	if(stmp) pkt_cid = ntohs(stmp);
	ctlptr += 2;
	/* Packet number */
	if(ctlptr < (next->start + next->length)) {
	    bcopy(ctlptr,&stmp,2);
	    next->seq = ntohs(stmp);
	    ctlptr += 2;
	}
	/* Total number of packets */
	if(ctlptr < (next->start + next->length)) {
	    bcopy(ctlptr,&stmp,2);
	    if(stmp) nd_pkts = ntohs(stmp);
	    ctlptr += 2;
	}
	/* Receievd through */
	if(ctlptr < (next->start + next->length)) {
	    /* Not supported by clients */
	    ctlptr += 2;
	}
	/* Backoff */
	if(ctlptr < (next->start + next->length)) {
	    bcopy(ctlptr,&stmp,2);
	    backoff = ntohs(stmp);
	    if(pfs_debug > 2) 
		fprintf(stderr,"Backing off to %d seconds\n", backoff);
	    if(backoff) timeout.tv_sec = backoff;
	    ctlptr += 2;
	}
	if(next->seq == 0) goto keep_waiting;
	goto done_old;

    }

    /* Notes that we have to start searching 11 bytes before the    */
    /* expected start of the MULTI-PACKET line because the message  */
    /* might include up to 10 bytes of data after the trailing null */
    /* The order of those bytes is two bytes each for Connection ID */
    /* Packet-no, of, Received-through, Backoff                     */
    seqtxt = nlsindex(next->start + max(0,next->length - 40),"MULTI-PACKET"); 
    if(seqtxt) seqtxt+= 13;

    if((nd_pkts == 0) && (no_pkts == 0) && (seqtxt == NULL)) {
	close(lp);
	return(first);
    }

    tmp = sscanf(seqtxt,"%d OF %d", &(next->seq), &nd_pkts);
    
    if (pfs_debug && (tmp == 0)) 
	fprintf(stderr,"Cant read packet sequence number: %s", seqtxt);    

 done_old:

    if(pfs_debug > 2) fprintf(stderr,"Packet %d of %d\n",next->seq,nd_pkts);

    if ((first == next) && (no_pkts == 0)) {
	no_pkts = 1;
	next = ptalloc();
	if(first->seq == 1) comp_thru = first;
	goto keep_waiting;
    }
	
    if(comp_thru && (next->seq <= comp_thru->seq))
	ptfree(next);
    else if (next->seq < first->seq) {
	vtmp = first;
	first = next;
	first->next = vtmp;
	first->previous = NULL;
	vtmp->previous = first;
	if(first->seq == 1) comp_thru = first;
	no_pkts++;
    }
    else {
	vtmp = (comp_thru ? comp_thru : first);
	while (vtmp->seq < next->seq) {
	    if(vtmp->next == NULL) {
		vtmp->next = next;
		next->previous = vtmp;
		next->next = NULL;
		no_pkts++;
		goto ins_done;
	    }
	    vtmp = vtmp->next;
	}
	if(vtmp->seq == next->seq)
	    ptfree(next);
	else {
	    vtmp->previous->next = next;
	    next->previous = vtmp->previous;
	    next->next = vtmp;
	    vtmp->previous = next;
	    no_pkts++;
	}
    }   

ins_done:
	
    while(comp_thru && comp_thru->next && 
	  (comp_thru->next->seq == (comp_thru->seq + 1))) {
	comp_thru = comp_thru->next;
	if(pfs_debug > 2) 
	    fprintf(stderr,"Packets now received through %d\n",comp_thru->seq);
    }
    if ((nd_pkts == 0) || (no_pkts < nd_pkts)) {
	next = ptalloc();
	goto keep_waiting;
    }

    close(lp);
    return(first);

}


set_dirsend_retry(to,rt)
    int	to;
    int rt;
    {
	client_dirsrv_timeout = to;
	client_dirsrv_retry = rt;
    }
