static char rcsid[] = "$Id: util.c,v 1.9 1992/08/20 17:56:40 kadhim Exp $";

/* util.c

   Copyright (C) 1987 Free Software Foundation, Inc.

   This file is part of GNU Info.

   GNU Info is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.  No author or distributor accepts
   responsibility to anyone for the consequences of using it or for
   whether it serves any particular purpose or works at all, unless he
   says so in writing.  Refer to the GNU Emacs General Public License
   for full details.

   Everyone is granted permission to copy, modify and redistribute
   GNU Info, but only under the conditions described in the GNU Emacs
   General Public License.   A copy of this license is supposed to
   have been given to you along with GNU Emacs so you can know your
   rights and responsibilities.  It should be in a file named COPYING.
   Among other things, the copyright notice and this notice must be
   preserved on all copies.  
*/

/* **************************************************************** */
/*								    */
/*			Utility Functions			    */
/*								    */
/* **************************************************************** */

#include "info.h"
#include "option.h"

char *search_buffer;		/* area in ram to scan through. */
int buffer_bottom;		/* Length of this area. */

void
set_search_constraints (buffer, extent)
     char *buffer;
     int extent;
{
  /* Set the global variables that all of these routines use. */

  search_buffer = buffer;
  buffer_bottom = extent;
}

int
to_beg_line (from)
     int from;
{
  extern int COLS;
  int i;

  i = 0;
  /* Move back to the start of this line. */
  while (from && search_buffer[from - 1] != '\n') {
      from--;
      i++;
  }
  while (i > COLS) {
    from += COLS;
    i -= COLS;
  }
  return (from);
}

int
to_end_line (from)
     int from;
{
  extern int COLS;
  int i;

  /* Move forward to the end of this line. */
  i = 0;
  while (from < buffer_bottom && search_buffer[from] != '\n' && i < COLS) {
    from++;
    i++;
  }
  return (from);
}

int
to_end_file_line (from)
     int from;
{
  /* Move forward to the end of this line. */
  while (from < buffer_bottom && search_buffer[from] != '\n')
    from++;
  return (from);
}

int
back_lines (count, starting_pos)
     int count, starting_pos;
{
  /* Move back count lines in search_buffer starting at starting_pos.
     Returns the start of that line. */
  starting_pos = to_beg_line (starting_pos);
  while (starting_pos && count)
    {
      starting_pos = to_beg_line (starting_pos - 1);
      count--;
    }
  return (starting_pos);
}

int
forward_lines (count, starting_pos)
     int count, starting_pos;
{
  /* Move forward count lines starting at starting_pos.
     Returns the start of that line. */
  starting_pos = to_end_line (starting_pos);
  while (starting_pos < buffer_bottom && count)
    {
      starting_pos = to_end_line (starting_pos + 1);
      count--;
    }
  return (to_beg_line (starting_pos));
}

int
search_forward (string, starting_pos)
     char *string;
     int starting_pos;
{
  /* Search for STRING in SEARCH_BUFFER starting at STARTING_POS.
     Return the location of the string, or -1 if not found. */

  int len = strlen (string);

  extern int strnicmp();

  int insensitive = strnicmp(options[SEARCHINSENSITIVE].value,"yes",3) == 0;
  int (*func)() = (insensitive ? strnicmp : strncmp);

  while ((starting_pos + len) <= buffer_bottom)
    {
      if (func (search_buffer + starting_pos, string, len) == 0)
	return (starting_pos);
      else
	starting_pos++;
    }
  return (-1);
}

int
search_backward (string, starting_pos)
     char *string;
     int starting_pos;
{
  /* Search for STRING in SEARCH_BUFFER starting at STARTING_POS.
     Return the location of the string, or -1 if not found. */

  int len = strlen (string);

  extern int strnicmp();

  int insensitive = strnicmp(options[SEARCHINSENSITIVE].value,"yes",3) == 0;
  int (*func)() = (insensitive ? strnicmp : strncmp);

  while (starting_pos - len > -1)
    {
      if (func (search_buffer + (starting_pos - len), string, len) == 0)
	return (starting_pos - len);
      else
	starting_pos--;
    }
  return (-1);
}

int
string_in_line (string, pointer)
     char *string;
     int pointer;
{
  /* Only search for STRING from POINTER to end of line.  Return offset
     of string, or -1 if not found. */
  int old_buffer_bottom = buffer_bottom;

  set_search_constraints (search_buffer, to_end_file_line (pointer));
  pointer = search_forward (string, pointer);
  buffer_bottom = old_buffer_bottom;
  return (pointer);
}

/* Skip whitespace characters at OFFSET in SEARCH_BUFFER.
   Return the next non-whitespace character or -1 if BUFFER_BOTTOM
   is reached. */
int
skip_whitespace (offset)
     int offset;
{
  int character;

  while (offset < buffer_bottom)
    {
      character = search_buffer[offset];
      if (character == ' ' || character == '\t')
	offset++;
      else
	return (offset);
    }
  return (-1);
}

/* Skip whitespace characters including <CR> at OFFSET in
   SEARCH_BUFFER.  Return the position of the next non-whitespace
   character, or -1 if BUFFER_BOTTOM is reached. */
int
skip_whitespace_and_cr (offset)
     int offset;
{
  while (true)
    {
      offset = skip_whitespace (offset);
      if (offset > 0 && search_buffer[offset] != '\n')
	return (offset);
      else
	offset++;
    }
}

/* Extract the node name part of the of the text after the FIELD.
   Place the node name into NODENAME.  Assume the line starts at
   OFFSET in SEARCH_BUFFER. */
boolean
extract_field (field_name, nodename, offset)
     char *field_name, *nodename;
     int offset;
{
  int temp, character;

  temp = string_in_line (field_name, offset);
  if (temp < 0)
    return (false);

  temp += strlen (field_name);
  temp = skip_whitespace (temp);

  /* Okay, place the following text into NODENAME. */

  while ((character = search_buffer[temp]) != ','
	 && character != '\n'
	 && character != '\t')
    {
      *nodename = character;
      nodename++;
      temp++;
    }
  *nodename = '\0';
  return (true);
}

boolean
looking_at (string, pointer)
     char *string;
     int pointer;
{
  /* Return true if pointer is exactly at string, else false. */

  extern int strnicmp();

  if (strnicmp (search_buffer + pointer, string, strlen (string)) == 0)
    return (true);
  else
    return (false);
}

/* Whoops, Unix doesn't have strnicmp. */

int
strnicmp (string1, string2, count)
     char *string1, *string2;
     int count;
{
  /* Compare at most COUNT characters from string1 to string2.  Case
     doesn't matter. */
  char ch1, ch2;

  while (count)
    {
      ch1 = *string1++;
      ch2 = *string2++;
      if (to_upper (ch1) == to_upper (ch2))
	count--;
      else
	break;
    }
  return (count);
}

void
clean_up (string)
     char *string;
{
  /* Remove <CR> and whitespace from string, replacing them with
     only one space.  Exception:  <CR> at end of string disappears. */

  char *to = string;
  char last_char = 0;
  boolean result;

  while (*to = *string++)
    {
      if (*to == '\n')
	{
	  *to = SPACE;
	  if (!(*(to + 1)))
	    {
	      *to = '\0';
	      return;
	    }
	}
      result = (last_char == SPACE);
      last_char = *to;
      if (last_char != SPACE)
	to++;
      else if (!result)
	to++;
    }
}
