static char RCSid[] = "$Id: envmod.c,v 1.9 91/05/06 10:32:07 tony Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */


/* environment module library */


/*LINTLIBRARY*/


#include <stdio.h>
#include "err.h"	/* error report module interface */
#include "envmod.h"	/* environment implementation types */


/* definition table interface: */

/* NewDef: returns a new definition table key */
extern char *NewDef();

/* NoDef: returns the distinguished definition table key
	  representing "no definition"
*/
extern char *NoDef();



/* state variables of the environment module */

static PSTACK *freeassoc = (PSTACK *) 0;	/* Free association elements */


/* error messages */

char *em_E_badsym = "symbol argument (%d) out of range!";
char *em_E_badenv = "null environment!";
char *em_E_alloc  = "allocation failure!";




/* emError: report an error from the environment module */
#if defined(__cplusplus) || defined(__STDC__)
static void
emError (int severity, char *function, char *msg)
#else
static void
emError (severity, function, msg)

	int   severity;
	char *function;
	char *msg;
#endif
{
	char buf[512];

	(void) sprintf(buf, "environment module (%s): %s", function, msg);
	message(severity, buf, 0, &curpos);

} /* emError */



/* MakeScope: create a new scope */
static SCOPE *
MakeScope ()
{
	SCOPE *s;
	char *fname = "MakeScope";

	s = (SCOPE *) malloc(sizeof(SCOPE));
	if (!s) {
		emError(DEADLY, fname, em_E_alloc);
		exit(1);
	}
	s->poss = (POSSESSION *) 0;
	return(s);

} /* MakeScope */



/* AddToScope: add a new possession to a given scope --
   on entry -
      s = scope in which the declaration was made
      i = identifier to be added
      k = definition table key
   on exit -
      AddToScope returns the new possession
*/
#if defined(__cplusplus) || defined(__STDC__)
static POSSESSION *
AddToScope (SCOPE *s, int i, char *k)
#else
static POSSESSION *
AddToScope (s, i, k)

	SCOPE *s;
	int i;
	char *k;
#endif
{
	register POSSESSION *p;
	char *fname = "AddToScope";

	/* create possession element with new definition and link to scope */
	p 	= (POSSESSION *) malloc(sizeof(POSSESSION));
	if (!p) {
		emError(DEADLY, fname, em_E_alloc);
		exit(1);
	}
	p->idn 	= i;
	p->key 	= k;
	p->nxt 	= s->poss;
	s->poss = p;
	return(p);

} /* AddToScope */




/* AddPoss: add a single possession in the current environment */
#if defined(__cplusplus) || defined(__STDC__)
static void
AddPoss (POSSESSION *p, PSTACK **stk, ENVIRONMENT *env)
#else
static void
AddPoss (p, stk, env)

	POSSESSION   *p;
	PSTACK 	    **stk;
	ENVIRONMENT  *env;
#endif
{
	char            *fname = "AddPoss";
	register PSTACK *old, *new;

	/* multiply defined if top of stack points to possession in same env. */
	if ((old = *stk) != (PSTACK *) 0)
		if (old->env == env)
			emError(FATAL, fname, "Multiply-defined identifier");
	
	/* get a stack element */
	if (freeassoc == (PSTACK *) 0) {
		new = (PSTACK *) malloc(sizeof(PSTACK));
		if (!new) {
			emError(DEADLY, fname, em_E_alloc);
			exit(1);
		}
	}
	else {
		new 	  = freeassoc;
		freeassoc = freeassoc->out;
	}

	/* push p onto possession stack */
	new->cur = p;
	new->env = env;
	new->out = old;
	*stk 	 = new;

} /* AddPoss */




/* Enter: enter an environment --
   on exit -
      the possessions of env->scp have been made current
*/
#if defined(__cplusplus) || defined(__STDC__)
static void
Enter (ENVIRONMENT *env)
#else
static void
Enter (env)

	ENVIRONMENT *env;
#endif
{
	POSSESSION *p;

	/* push all possessions in env's scope on their stacks */
	for (p = env->scp->poss; p; p = p->nxt)
		AddPoss(p, &env->acc->ptbl[p->idn], env);

	env->acc->cur = env;	/* mark env as current */

} /* Enter */



/* Leave: leave an environment --
   on exit -
      the possessions of env->scp have been made obsolete
*/
#if defined(__cplusplus) || defined(__STDC__)
static void
Leave (ENVIRONMENT *env)
#else
static void
Leave (env)

	ENVIRONMENT *env;
#endif
{
	register POSSESSION *p;

	/* pop the possession stacks for all symbols in env's scope */
	for (p = env->scp->poss; p; p = p->nxt) {

		register PSTACK *old;
		register PSTACK **stk = &env->acc->ptbl[p->idn];

		old 	  = *stk;
		*stk 	  = old->out;
		old->out  = freeassoc;
		freeassoc = old;
	}
	env->acc->cur = env->env;	/* enclosing env. becomes current */
	env->acc = (ACCESS *) 0;	/* mark as inactive */

} /* Leave */



/* CurrEnv: guarantee that env is the current environment */
#if defined(__cplusplus) || defined(__STDC__)
static void
CurrEnv (ENVIRONMENT *env)
#else
static void
CurrEnv (env)

	ENVIRONMENT *env;
#endif
{
	register ACCESS *a;

	if ((a = env->acc) == (ACCESS *) 0) {
		/* env is on a currently inactive branch:
		   we must first leave all environments in the currently
		   active branch and then enter all the environments that
		   enclose env;
		   a recursive call on the enclosing environment will
		   "automagically" take care of that:
		*/
		CurrEnv(env->env);
		env->acc = env->env->acc;	/* same access table */
		Enter(env);			/* make possessions current */
	}
	else
		/* the easy case - env is on the active branch, but perhaps
		   not current; we leave all inner environments until env
		   becomes current:
		*/
		while (a->cur != env)
			Leave(a->cur);

} /* CurrEnv */



/* NewEnv: create a new environment --
   on exit -
      NewEnv describes an environment consisting of an empty scope
*/
ENVIRONMENT *
NewEnv ()
{
	register int 	 i;
	register ACCESS *a;
	ENVIRONMENT 	*e;
	char *fname = "NewEnv";

	/* allocate new environment and access table */
	e = (ENVIRONMENT *) malloc(sizeof(ENVIRONMENT));
	if (!e) {
		emError(DEADLY, fname, em_E_alloc);
		exit(1);
	}
	a = (ACCESS *) malloc(sizeof(ACCESS));
	if (!a) {
		emError(DEADLY, fname, em_E_alloc);
		exit(1);
	}

	e->env = (ENVIRONMENT *) 0;	/* no enclosing environment */
	e->scp = MakeScope();		/* new empty scope */
	e->acc = a;			/* access table */
	a->cur = e;			/* this environment is current */
	for (i = 0; i < MAXIDN; i++)	/* initialize possession stacks */
		a->ptbl[i] = (PSTACK *) 0;
	return(e);

} /* NewEnv */




/* NewScope: create a new scope within a given environment */
#if defined(__cplusplus) || defined(__STDC__)
ENVIRONMENT *
NewScope (ENVIRONMENT *e)
#else
ENVIRONMENT *
NewScope (e)

	ENVIRONMENT *e;
#endif
{
	ENVIRONMENT *new;
	char *fname = "NewScope";

	/* check that environment e is non-null */
	if (e == (ENVIRONMENT *) 0) {
		emError(DEADLY, fname, em_E_badenv);
		exit(1);
	}

	/* make e the current environment */
	CurrEnv(e);

	/* allocate new environment */
	new = (ENVIRONMENT *) malloc(sizeof(ENVIRONMENT));
	if (!new) {
		emError(DEADLY, fname, em_E_alloc);
		exit(1);
	}

	new->scp      = MakeScope();	/* create new scope */
	new->env      = e;		/* e is enclosing environment */
	new->acc      = e->acc;		/* same access table */
	new->acc->cur = new;		/* now new is current environment */
	return(new);

} /* NewScope */




/* KeyInEnv: obtain the definition of an identifier in a given environment --
   on entry -
      e = environment in which the definition is sought
      i = identifier whose definition is sought
   on exit -
	if i has a definition in e
	   the definition table key of i in e is returned,
        otherwise,
	   the reserved key NoDef() is returned
*/
#if defined(__cplusplus) || defined(__STDC__)
char *
KeyInEnv (ENVIRONMENT *e, int i)
#else
char *
KeyInEnv (e, i)

	ENVIRONMENT *e;
	int i;
#endif
{
	char		*fname = "KeyInEnv";
	register PSTACK *ps;

	/* check that environment e is non-null */
	if (e == (ENVIRONMENT *) 0) {
		emError(DEADLY, fname, em_E_badenv);
		exit(1);
	}

	/* check if symbol i is out of range */
	if (i < 0  ||  i >= MAXIDN) {
		char msg[64];
		(void) sprintf(msg, em_E_badsym, i);
		emError(DEADLY, fname, msg);
		exit(1);
	}

	/* make e the current environment */
	CurrEnv(e);

	/* check if i has a definition in e */
	if ((ps = e->acc->ptbl[i]) != (PSTACK *) 0)
		return(ps->cur->key);	/* it does */
	else
		return(NoDef());	/* it doesn't */

} /* KeyInEnv */




/* KeyInScope: obtain definition from the inner scope of an environment --
   on entry -
      e = environment in which the definition is sought
      i = identifier whose definition is sought
   on exit -
      if i is defined in the inner scope of e
         the definition table key of i is returned,
      otherwise,
	 the reserved key NoDef() is returned
*/
#if defined(__cplusplus) || defined(__STDC__)
char *
KeyInScope (ENVIRONMENT *e, int i)
#else
char *
KeyInScope (e, i)

	ENVIRONMENT *e;
	int i;
#endif
{
	char *fname = "KeyInScope";

	/* check that environment e is non-null */
	if (e == (ENVIRONMENT *) 0) {
		emError(DEADLY, fname, em_E_badenv);
		exit(1);
	}

	/* check if symbol i is out of range */
	if (i < 0  ||  i >= MAXIDN) {
		char msg[64];
		(void) sprintf(msg, em_E_badsym, i);
		emError(DEADLY, fname, msg);
		exit(1);
	}

	/* if e is the current environment, use constant time access */
	if (e->acc  &&  e->acc->cur == e) {
		register PSTACK *ps = e->acc->ptbl[i];
		if (ps == (PSTACK *) 0)
			return(NoDef());	/* not defined in e */
		else if (ps->env != e)
			return(NoDef());	/* not in inner scope of e */
		else
			return(ps->cur->key);
	}

	/* else (e is not the current env.), search e's inner scope */
	else {
		register POSSESSION *p;
		for (p = e->scp->poss; p; p = p->nxt)
			if (p->idn == i)
				break;
		return(p ? p->key : NoDef());
	}

} /* KeyInScope */




/* DefineIdn: define an identifier in the inner scope of an environment --
   on entry -
      e = environment in which to define the identifier
      i = identifier to be defined
   on exit -
      if i was already defined in the inner scope of e,
	 the corresponding definition table key is returned
      otherwise,
	 a new definition table key is returned and entered
	 into the inner scope of e
*/
#if defined(__cplusplus) || defined(__STDC__)
char *
DefineIdn (ENVIRONMENT *e, int i)
#else
char *
DefineIdn (e, i)

	ENVIRONMENT *e;
	int	     i;
#endif
{
	register PSTACK     *ps;
	register POSSESSION *p;
	char *fname = "DefineIdn";

	/* check that environment e is non-null */
	if (e == (ENVIRONMENT *) 0) {
		emError(DEADLY, fname, em_E_badenv);
		exit(1);
	}

	/* check if symbol i is out of range */
	if (i < 0  ||  i >= MAXIDN) {
		char msg[64];
		(void) sprintf(msg, em_E_badsym, i);
		emError(DEADLY, fname, msg);
		exit(1);
	}

	/* make e the current environment */
	CurrEnv(e);

	/* check if i already has a definition in the inner scope of e */
	if ((ps = e->acc->ptbl[i]) == (PSTACK *) 0) {
		/* no definition in e yet:
		   create one with a new key in the current scope (AddToScope),
		   then push it onto the possession stack for i (AddPoss)
		*/
		p = AddToScope(e->scp, i, NewDef());
		AddPoss(p, & e->acc->ptbl[i], e);
	}
	else if (ps->env != e) {
		/* i is defined in e, but not in the inner scope */
		p = AddToScope(e->scp, i, NewDef());
		AddPoss(p, & e->acc->ptbl[i], e);
	}
	else
		/* i already has a definition in the inner scope of e */
		p = ps->cur;
	
	return(p->key);

} /* DefineIdn */




/* AddIdn: define an identifier in the inner scope of an environment
   with a given key --
   on entry -
      e = environment in which to define the identifier
      i = identifier to be defined
      k = definition table key
   on exit -
      if i was already defined in the inner scope of e,
	 the corresponding definition table key is returned
      otherwise,
	 i becomes defined in the inner scope of e with the
	 given definition table key k
*/
#if defined(__cplusplus) || defined(__STDC__)
char *
AddIdn (ENVIRONMENT *e, int i, char *k)
#else
char *
AddIdn (e, i, k)

	ENVIRONMENT *e;
	int	     i;
	char	    *k;
#endif
{
	register PSTACK     *ps;
	register POSSESSION *p;
	char *fname = "AddIdn";

	/* check that environment e is non-null */
	if (e == (ENVIRONMENT *) 0) {
		emError(DEADLY, fname, em_E_badenv);
		exit(1);
	}

	/* check if symbol i is out of range */
	if (i < 0  ||  i >= MAXIDN) {
		char msg[64];
		(void) sprintf(msg, em_E_badsym, i);
		emError(DEADLY, fname, msg);
		exit(1);
	}

	/* make e the current environment */
	CurrEnv(e);

	/* check if i already has a definition in the inner scope of e */
	if ((ps = e->acc->ptbl[i]) == (PSTACK *) 0) {
		/* no definition in e yet:
		   create one with key k in the current scope (AddToScope),
		   then push it onto the possession stack for i (AddPoss)
		*/
		p = AddToScope(e->scp, i, k);
		AddPoss(p, & e->acc->ptbl[i], e);
	}
	else if (ps->env != e) {
		/* i is defined in e, but not in the inner scope */
		p = AddToScope(e->scp, i, k);
		AddPoss(p, & e->acc->ptbl[i], e);
	}
	else
		/* i already has a definition in the inner scope of e */
		p = ps->cur;
	
	return(p->key);

} /* AddIdn */

