static char RCSid[] = "$Id: source.c,v 1.1 1991/09/11 10:02:09 cogito Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "source.h"

#define READSIZ 1024	/* primitive I/O in this size bytes */
#define MAXLINE 256	/* May be increased to READSIZ/2 */
/*
** Source text buffering.
** Designed to be fast! And guarantee that full lines are always in memory.
** The idea is to read many lines into READBUF.  There might be the head
** of a partial line at the end of the buffer.  Scan in reverse from the
** end of the buffer, looking for newline. 
** OLD:  Zap it with NUL.
** NEW: Save the character AFTER the newline, and zap its position with NUL
**	This new way keeps newlines with their line - necessary for the way
**	aux scanners work for multi-line arbitrary length strings.
**      See gla.impl and CcharString.c for more details.
** Now the user
** will stop processing when he references this NUL.  refillBuf is called.
** The head of the partial line is copied in front of READBUF, and the
** next READSIZ is read into READBUF.  This will contain the tail of
** the partial line, plus more full lines, plus possibly another partline.
**
** initBuf takes an open file and returns with TokenEnd pointing to the
** first char in file, or pointing to NUL if empty.
**
** refillBuf(p) is called when the last character in the buffer is
** reached.  This is known because *p == NUL.  The argument p is
** used only for assertion checking.  It must be pointing at the last
** character of the buffer -- that is, the one is NUL and signals the end.
** Upon return, TokenEnd will be set to the next character of the file.
**
** The source module guarantees that whole lines will be available in
** the buffer if no line is longer than MAXLINE.  The input file is
** assumed to be ascii, with no embedded NULs.  Each line must be
** newline terminated.
**
**   memblock		 READBUF
**       |		 |
**	|---------------|---------------------------------------------------|
**	| <--MAXLINE-->	|  <--------READSIZ ----------------------------->  |
**	|	^	|				^	^	    |
**	|TokenEnd	|			     charend  READBUF+count |
**	|-------------------------------------------------------------------|
**
*/

/* Exported variables: updated by source module, and scanners */

#define NAMSIZ 250
char SourceName[NAMSIZ] = {0};	/* to remember the name of file */
int LineNum = 1;                /* source line number */
char *StartLine=0;      	/* pointer to beginning of line */
char *TokenStart;
char *TokenEnd;


static char sentinelSav;	/* when we drop in a sentinel, need to save original*/
static char *memblock;
static int maxline = MAXLINE;	/* reads occur at this offset */
#define READBUF (memblock+maxline)
static char *charend;		/* delimits boundary between useful lines
				  and a last partial line */
static int count;		/* number of bytes read */
static int fd;
#define NUL '\0'		/* the ASCII nul character */

#if defined(__cplusplus) || defined(__STDC__)
void
refillBuf(char *p)
#else
void
refillBuf(p)
char *p;
#endif
{
	extern char *malloc();

	if( SourceName[0] == 0 )  /* make sure initBuf was called first!! */
	  fprintf(stderr,"Source module never initialized with initBuf\n");

	/* refillBuf only called when necessary and file doesn't contain NUL */
	if( p != charend ){	/* make sure user and I agree on the end */
	  fprintf(stderr,"refillBuf:sentinel problem, may be NUL in file\n");
	  TokenEnd = p+1;	/* skip over unexpected null */
	  return;
	  }

	/* Calculate new beginning of buffer:
	** 		 ...# chars to be copied.............  ALGEBRA
	** TokenEnd = READBUF - (READBUF+count-1 - charend + 1)
	** TokenEnd = READBUF - (READBUF+count   - charend    )
	** TokenEnd = READBUF - READBUF - count   + charend 
	*/
	TokenEnd = 			 - count   + charend;

	if(charend <= READBUF+count-1) {	/* need to copy down bytes */
		/*
		** copy backwards and let nul byte terminate loop,
		** then replace this nul with the saved character.
		**
		**		|<---------------READBUF------------>
		**		|	charend
		**		|	|
		**		|	0ABC	upon entry to refillBuf
		**	    0ABC|		after copy down
		**	    sABC|		after replacing nul with char s
		**          ^
		**      TokenEnd
		*/
		register char *src = READBUF+count;	/* one beyond */
		register char *dest = READBUF;		/* one beyond */

		if( charend <= READBUF+count-maxline) {	/* re-size buffer */
			char *temp = memblock;

			maxline = READBUF + count - charend;
			memblock = malloc((unsigned)(maxline+READSIZ+1));
			dest = READBUF;
			while( *--dest = *--src) ;
			(void)free(temp);
		} else while( *--dest = *--src) ;

		*dest = sentinelSav;
		TokenEnd = dest;
	}




	*READBUF = NUL;		/* drop in nul incase empty read */
	count = read(fd, READBUF, READSIZ);
	if (count < 0) {
		fprintf(stderr, "RefillBuf:FATAL reading file %s\n", SourceName);
		perror("READ");
		exit(1);
	}
	charend = READBUF+count;	/* make this consistent with XXX below */

	if(count > 0){
		/*
		** Scan from rear, looking for newline, 
		** exchange the character that follows with the NUL sentinel
		*/
		charend--;			/* XXX make consistend w/ above */
		while(*charend != '\n')
			charend--;
		charend++;
		sentinelSav = *charend;
		*charend = NUL;
	}
}


#if defined(__cplusplus) || defined(__STDC__) 
void
initBuf(const char *name, int f)
#else
void
initBuf(name, f)
char *name; int f;
#endif
/* Set to the first source text line
 *    On entry-
 *       name=name of the file to be processed
 *       f=file descriptor of the file to be processed
 *    On exit-
 *       If the source text is empty then on exit-
 *           TokenEnd addresses a location containing the null character
 *	 else
 *           TokenEnd addresses the first character of the line
 ***/
{
	char *malloc();

	fd = f;
	(void)strncpy(SourceName,name,NAMSIZ);
	memblock = malloc(MAXLINE+READSIZ+1);	/* one extra
				in case the last character read is a newline
				(then we want to zap the next location) */
	count=0;		/* no characters have been previously read */
	*READBUF = '\0';	/* in case of empty file */
	TokenEnd = charend = READBUF;  /* make refillBuf happy */
	refillBuf(TokenEnd);
	StartLine = TokenEnd - 1;	/* initialize the start of line 1 */
	LineNum = 1;
}
