/* $Id: expconstit.c,v 2.8 1992/11/19 11:57:57 cogito Exp $ */
static char rcs_id[]= "$Id: expconstit.c,v 2.8 1992/11/19 11:57:57 cogito Exp $";

/***********************************************************************\
*									*
*		Implementation of LIGA expand pass			*
*									*
*	Module:		expconstit.c					*
*	File:		expconstit.c					*
*	Contents:	functions to expand CONSTITUENT and		*
*			CONSTITUENTS					*
*	Author:		Hans Detlef Siewert				*
*	Creation date:	02 Apr 1991					*
*	Last change:	26 Sep 1991					*
*									*
\***********************************************************************/

#include <stdio.h>
#include <string.h>

#include "LIGA.h"
#include "LIGAMacros.h"

/* Includes for option handling */
#include "option_enums.h"
#include "option_types.h"

#include "ligaconsts.h"
#include "expconst.h"
#include "exp_types.h"
#include "exp_io.h"
#include "expref.h"
#include "expidl.h"
#include "expand.h"
#include "exptrav.h"
#include "expconstit.h"

/*
***** EXPANSION OF CONSTITUENT(S) CONSTRUCTS
**
** To expand the CONSTITUENT and the CONSTITUENTS constructs this pass first
** call the function find_constits(). It searches for all constructs by calling
** the recursive function constits_in_call() for all call rules of all
** attributions. This function appends all constructs that are found to a global
** list by calling the append_const(). All constructs are compared to other
** constructs by calling equal_constits() to combine the expansion of similar
** constituents; they are collected in special lists.
**  In the next step expand_constits() is called. It uses the previous collected
** list and examines all constructs. For each construct and its identical
** constructs first all flags for productions and symbols are cleared and then
** mark_constituent() is called. All productions containing a construct are
** marked as target productions. In the recursive function mark_constit_trans()
** all symbols and productions on paths from the constructs to occurrences of
** the constituent symbol are marked for transport. All productions containing a
** source symbol (the symbol given in the construct) on the right hand side are
** marked as source productions. In contrast to the traversing functions used
** while expanding INCLUDING and CHAIN constructs, mark_constit_trans() starts
** at the root of the subtrees and traverses the tree downwards. This is
** necessary because of the specific subtree accesses. An additional flag
** is used to mark visited productions to avoid endless loops.
**  When the productions and symbols have been marked, the function
** expand_constituent() is called by the function expand_constits(). It examines
** all attributions of the productions. In source and transport productions it
** generates appropriate assignments for transports. This is done by calling
** gen_constrans() which calls the function rhs_const() to create an expression
** that retrieves the values from the rhs symbols. For each symbol symb_const()
** is called to get its values. The values from the rhs symbols are combined by
** generating calls of the appropriate concatTl() function. An error results if
** the users wanted to retrieve only one CONSTITUENT but multiple symbols are
** marked on the right hand side. The function symb_const() further uses the
** function gen_create() to generate calls of the creatTl() function.
**  If only the lhs symbol of a production is marked but no symbols from the rhs
** then a call to the nullTl() function is generated by the function
** expand_contituent() by calling the function gen_empty(). This is also an
** error if one CONSTITUENT attribute should have been retrieved.
**  The function repl_constit() replaces all constructs in target productions.
** It uses the function rhs_const() to create the expression and replaces the
** construct with it.
**
*****
*/


/*
** GLOBAL VARIABLES
*/

ConstNode	constituents;	/* global list of all CONSTITUENT(S)s	*/

/*
** LOCAL VARIABLES
*/

int	newassigns;	/* total number of new assignments 		*/

/*
** LOCAL FUNCTIONS
*/

static void
#ifdef __STDC__
show_constituent (ConstNode cptr)
#else
show_constituent (cptr)
ConstNode cptr;
#endif
/* show this CONSTITUENT(S) contruct and it's similar constructs	*/
{
	CoSyNode	csn;
	SyAttrNode	san;
	ConstNode	sameconstit;
	SEQExpr		stmp;
	Expr		shexp;

	if (!cptr)
		return;

	(void) fprintf (ProtocolFile,
		"\n    --> %s in production %s, line %d, col %d\n",
		singleOfConstit(cptr->constit) ? "single" : "list",
		dnameOfProd(prodref(cptr->prodid)),
		rowOfConstit(cptr->constit),
		colOfConstit(cptr->constit));

	if (!singleOfConstit(cptr->constit))
	{
		(void) fprintf (ProtocolFile,
			"\ttype of transport attributes is %s\n",
			cptr->list_name);
		(void) fprintf (ProtocolFile,
			"\tfunctions are %s, %s, %s\n",
			cptr->concat_name, cptr->creat_name, cptr->empty_name);
	} /* if */

	/* show all source symbols */
	(void) fprintf (ProtocolFile, "\tsource attributes are:\n");
	for (csn= cptr->src; csn; csn= csn->next)
	{
	    /* show all source attributes for this symbol */
	    for (san= csn->attrs; san; san= san->next)
	    {
		(void) fprintf (ProtocolFile,
			"\t\t%s.%s\n",
			dnameOfSymb (symbref (csn->symbdid)),
			nameOfAttrdef (san->ad));
	    } /* for */
	} /* for */

	/* show all source symbols */
	(void) fprintf (ProtocolFile, "\tshielded subtrees:\n");
	if (!cptr->shield_symbs)
		(void) fprintf (ProtocolFile, "\t\tnone\n");
	else
	foreachinSEQExpr (cptr->shield_symbs, stmp, shexp) {
		(void) fprintf (ProtocolFile,
			"\t\t%s\n", dnameOfSymb (symbref (
					vOfVal (ExprToVal (shexp)))));
	}

	if (sameconstit= cptr->same)	/* assignment intended */
	{
		(void) fprintf (ProtocolFile, "\tidentical constructs:\n");
		/* show all similar constructs */
		for (; sameconstit; sameconstit= sameconstit->same)
		{
			(void) fprintf (ProtocolFile,
				"\t\tin production %s, line %d, col %d\n",
				dnameOfProd(prodref(sameconstit->prodid)),
				rowOfConstit(sameconstit->constit),
				colOfConstit(sameconstit->constit));
		} /* for */
	} /* if */
} /* show_constituent() */

static void
#ifdef __STDC__
mark_constituent (ConstNode cptr)
#else
mark_constituent (cptr)
ConstNode cptr;
#endif
/* mark symbols and productions for expansion of CONSTITUENT(S)		*/
{
	ConstNode	same;
	SNode		symbs;
	CoSyNode	csn;
	int		found;
	SEQExpr		tmpse;
	Expr		shexp;
	int		shdid;

	markedsymbs= 0;
	markedprods= 0;
	
	/* mark all source symbols */
	for (csn= cptr->src; csn; csn= csn->next)
		symbflag(csn->symbdid) |= src_symb;

	/* mark all shielded symbols (replace by new shield semantic) */
	foreachinSEQExpr (cptr->shield_symbs, tmpse, shexp) {
		shdid = vOfVal (ExprToVal (shexp));
		symbflag(shdid) |= shield_symb;
	}

	/* mark all start productions as target productions */
	for (same= cptr; same; same= same->same)
	{
		prodflag(same->prodid) |= target_prod;
	} /* for */

	protout (ProtocolFile, "\ttransport productions and symbols:\n");

	/* mark all subtrees in start productions containing the attribute */
	for (same= cptr; same; same= same->same)
	{
		found= FALSE;
		if (subtreeOfConstit(same->constit))
		{
			found= mark_constit_trans (same->subdid, same);

			if (symbflag (same->subdid) & src_symb)
			{
				/* mark as source production */
				prodflag(same->prodid) |= start_prod;

				/* symbol found at the root of the subtree */
				found= TRUE;
			} /* if */

		} /* if */
		else
		{
			/* examine all subtrees below rhs symbols */
			for (symbs= prodright(same->prodid)->right;
				symbs;
				symbs= symbs->right)
			{
				found |= mark_constit_trans(
						symbs->symbdid, same);

				if (symbflag (symbs->symbdid) & src_symb)
				{
					/* mark as source production */
					prodflag(same->prodid) |= start_prod;

					/* symbol found at the root of the subtree */
					found= TRUE;
				} /* if */
			} /* for */
		} /* else */

		if (!found)
		{
		    if (singleOfConstit(same->constit))
		    {
			/* no CONSTITUENT symbol reachable */
			print_err (
				rowOfConstit(same->constit),
				colOfConstit(same->constit),
				"unreachable CONSTITUENT symbol",
				CONR_ERRID);
			(void) fprintf (ProtocolFile, "\n*** ERROR  ");
			(void) fprintf (ProtocolFile,
				"no CONSTITUENT symbol found in subtree ");
			(void) fprintf (ProtocolFile,
				"of production %s, in line %d, col %d\n\n",
				dnameOfProd(prodref(same->prodid)),
				rowOfConstit(same->constit),
				colOfConstit(same->constit));
			same->cancelled= TRUE;
		    } /* if */
		    else
		    {
			/* empty CONSTITUENTS list */
			print_warn (
				rowOfConstit(same->constit),
				colOfConstit(same->constit),
				"no CONSTITUENTS symbol in subtree",
				CONS_WARNID);
			(void) fprintf (ProtocolFile, "\n*** WARNING  ");
			(void) fprintf (ProtocolFile,
				"no CONSTITUENTS symbols found in subtree ");
			(void) fprintf (ProtocolFile,
				"of production %s, in line %d, col %d\n\n",
				dnameOfProd(prodref(same->prodid)),
				rowOfConstit(same->constit),
				colOfConstit(same->constit));
		    } /* else */
		} /* if */
	} /* for */

	protout (ProtocolFile,
		"\tmarked for transport: %d symbols, %d productions\n",
		markedsymbs, markedprods);
} /* mark_constituent() */

static void
#ifdef __STDC__
gen_empty (Attribution attrib, ConstNode cptr)
#else
gen_empty (attrib, cptr)
Attribution attrib;
ConstNode cptr;
#endif
/* create assignment of empty list to transport attribute of lhs symbol	*/
{
	int	prod;
	Symb	symb;
	Attrdef	ad;
	Call	emptycall, new;
	Expr	expr;
	Attracc	access;
	Name	v_name;

	prod= prodidOfAttribution(attrib);

	symb= symbref(prodright(prod)->symbdid);

	/* transport attribute should be the first attribute */
	retrievefirstSEQAttrdef (attrsOfSymb(symb), ad);
	if (strcmp(genattrOfConstit(cptr->constit), nameOfAttrdef(ad)))
	{
		print_err (
			rowOfConstit(cptr->constit),
			colOfConstit(cptr->constit),
			"expconstit.c: internal error #1", NO_ERRID);
		cptr->cancelled= TRUE;
		return;
	} /* if */

	protout (ProtocolFile, "\tempty list in production %s\n",
		dnameOfProd(prodref(prod)));
	protout (ProtocolFile, "\t\t%s<1>.%s := %s();\n",
		dnameOfSymb (symb),
		genattrOfConstit (cptr->constit),
		cptr->empty_name);

	/* initialize call of assignment */
	new= NCall;
	nameOfCall(new)= ASSIGNFCT;
	initializeSEQExpr (paramsOfCall(new));
	rowOfCall(new)= rowOfProd(prodref(prod));
	colOfCall(new)= colOfProd(prodref(prod));

	/* initialize left hand side of the new assignment */
	access= NAttracc;
	symbnoOfAttracc(access)= 0;	/* lhs symbol of production */
	attridOfAttracc(access)= didOfAttrdef(ad);
	rowOfAttracc(access)= rowOfProd(prodref(prod));
	colOfAttracc(access)= colOfProd(prodref(prod));
	ExprToAttracc(expr)= access;
	appendrearSEQExpr (paramsOfCall(new), expr);

	/* initialize call of empty function */
	emptycall= NCall;
	nameOfCall(emptycall)= cptr->empty_name;
	initializeSEQExpr (paramsOfCall(emptycall));
	rowOfCall(emptycall)= rowOfProd(prodref(prod));
	colOfCall(emptycall)= colOfProd(prodref(prod));
	ExprToCall(expr)= emptycall;
	appendrearSEQExpr (paramsOfCall(new), expr);

	/* append special name to mark assignments to VOID attributes */
	if (cptr->list_tid == DIDVOID
	    || cptr->list_tid == DIDVOLI
	    || cptr->dep)
	{
		v_name= NName;
		nOfName(v_name)= TNVOID;
		rowOfName(v_name)= rowOfProd(prodref(prod));
		colOfName(v_name)= colOfProd(prodref(prod));
		ExprToName(expr)= v_name;
		appendrearSEQExpr (paramsOfCall(new), expr);
	} /* if */

	/* append new call to the list of attribution rules */
	ExprToCall(expr)= new;
	appendrearSEQAttrrule(attrrulesOfAttribution(attrib), expr);
	newassigns++;
} /* gen_empty() */

static Expr
#ifdef __STDC__
gen_create (ConstNode cptr, Attracc acc, Symb sd, Attrdef ad, char **info)
#else
gen_create (cptr, acc, sd, ad, info)
ConstNode cptr;
Attracc acc;
Symb sd;
Attrdef ad;
char **info;
#endif
/* generate list create function call from attribute access		*/
{
	Call	call;
	Expr	result;

	/* build call to create list */
	call= NCall;
	nameOfCall(call)= cptr->creat_name;
	initializeSEQExpr (paramsOfCall(call));
	rowOfCall(call)= 0;
	colOfCall(call)= 0;

	/* append attribute access to call parameter list */
	ExprToAttracc(result)= acc;
	appendrearSEQExpr (paramsOfCall(call), result);

	/* result is the new call */
	ExprToCall(result)= call;

	/* create information */
	*info= malloc (
		strlen (dnameOfSymb(sd)) +
		strlen (nameOfAttrdef(ad)) +
		strlen (nameOfCall(call)) + 12);
	(void) sprintf (*info, "%s(%s<%d>.%s)",
		nameOfCall(call),
		dnameOfSymb(sd),
		symbnoOfAttracc(acc)+1,
		nameOfAttrdef(ad));

	return (result);
} /* gen_create() */

static Expr
#ifdef __STDC__
gen_concat (
	Expr expr1, char *info1, int infolen1,
	Expr expr2, char *info2, int infolen2,
	ConstNode cptr, char **info, int *infolen)
#else
gen_concat (expr1, info1, infolen1, expr2, info2, infolen2, cptr, info, infolen)
Expr expr1;
char *info1;
Expr expr2;
char *info2;
ConstNode cptr;
char **info;
int *infolen;
#endif
/* generate concatenation of two constituent list expressions		*/
{
	Call	call;
	Expr	result;

	/* build call to concatenate lists */
	call= NCall;
	nameOfCall(call)= cptr->concat_name;
	initializeSEQExpr (paramsOfCall(call));
	rowOfCall(call)= 0;
	colOfCall(call)= 0;

	/* include both expressions in call parameter list */
	appendrearSEQExpr (paramsOfCall(call), expr1);
	appendrearSEQExpr (paramsOfCall(call), expr2);

	/* expression is the new call */
	ExprToCall(result)= call;

	/* create information */
	*info= malloc (infolen1 + infolen2 + strlen (nameOfCall(call)) + 6);
	(void) sprintf (*info, "%s(%s, %s)",
		nameOfCall(call), info1, info2);
	*infolen= infolen1 + infolen2 + strlen(nameOfCall(call)) + 6;

	return (result);
} /* gen_concat() */

static Expr
#ifdef __STDC__
collect_attrs (ConstNode cptr, int symbno, Symb sd, CoSyNode csn, char **info)
#else
collect_attrs (cptr, symbno, sd, csn, info)
ConstNode cptr;
int symbno;
Symb sd;
CoSyNode csn;
char **info;
#endif
/* collect source attributes from a source symbol			*/
{
	SyAttrNode	san;
	Attracc		newaccess;
	Expr		collection, source_expr;
	char		*collection_info, *source_info;
	int		infolen;
	int		first;

	first= TRUE;
	/* loop through all source attributes of this symbol */
	for (san= csn->attrs; san; san= san->next)
	{
		/* create source attribute access */
		newaccess= NAttracc;
		symbnoOfAttracc(newaccess)= symbno;
		attridOfAttracc(newaccess)= didOfAttrdef(san->ad);
		rowOfAttracc(newaccess)= 0;
		colOfAttracc(newaccess)= 0;

		source_expr= gen_create (
			cptr, newaccess, sd, san->ad, &source_info);

		if (first)
		{
			first= FALSE;
			collection= source_expr;
			collection_info= source_info;
			*info= source_info;
			infolen= strlen(source_info)+1;
		} /* if */
		else
		{
			collection= gen_concat (
				collection, collection_info, infolen,
				source_expr, source_info, strlen(source_info)+1,
				cptr, info, &infolen);
			free (collection_info);
			collection_info= *info;
		} /* else */
	} /* for */

	return (collection);
} /* collect_attrs() */

static Expr
#ifdef __STDC__
symb_const (	int subtreerepl,
		Attribution attrib,
		ConstNode cptr,
		int symbno,
		int symbdid,
		char **info,
		int *infolen)
#else
symb_const (subtreerepl, attrib, cptr, symbno, symbdid, info, infolen)
int subtreerepl;
Attribution attrib;
ConstNode cptr;
int symbno, symbdid;
char **info;
int *infolen;
#endif
/* create a new expression to get the constituents from the rhs symbol	*/
{
	int		direct_source;
	int		trnsp_source;
	Expr		result;
	CoSyNode	csn;
	Attrdef		trans_ad;
	Attracc		newaccess;
	Expr		trans_expr, source_expr;
	char		*trans_info, *source_info;

	ExprToCall(result)= NULL;
	trans_info= (char *) NULL;
	source_info= (char *) NULL;

	direct_source = symbflag(symbdid) & src_symb;
	trnsp_source = (symbflag(symbdid) & trnsp_symb) &&
			(!(symbflag(symbdid) & shield_symb)
				|| subtreerepl);

	if (direct_source)
	{	for (csn= cptr->src; csn; csn= csn->next)
			if (csn->symbdid == symbdid) break;

		if (singleOfConstit(cptr->constit))
		{	/* create source attribute access */
			newaccess= NAttracc;
			symbnoOfAttracc(newaccess)= symbno;
			attridOfAttracc(newaccess)= didOfAttrdef(csn->attrs->ad);
			rowOfAttracc(newaccess)= 0;
			colOfAttracc(newaccess)= 0;

			/* expression is simply the access */
			ExprToAttracc(source_expr)= newaccess;

			/* generate information */
			source_info= malloc (
				strlen (dnameOfSymb(symbref(symbdid))) +
				strlen (nameOfAttrdef(csn->attrs->ad)) + 10);
			(void) sprintf (source_info, "%s<%d>.%s",
				dnameOfSymb(symbref(symbdid)),
				symbno+1,
				nameOfAttrdef(csn->attrs->ad));
		} /* if */
		else
		{
			source_expr= collect_attrs (
				cptr, symbno, symbref(symbdid),
				csn, &source_info);
		} /* else */
	} /* if */

	/* retrieve transport list from actual symbol if necessary */
	if (trnsp_source)
	{
		/* get definition of the transport attribute */
		retrievefirstSEQAttrdef (
			attrsOfSymb(symbref(symbdid)),
			trans_ad);

		/* create attribute access to transport attribute */
		newaccess= NAttracc;
		symbnoOfAttracc(newaccess)= symbno;
		attridOfAttracc(newaccess)= didOfAttrdef(trans_ad);
		rowOfAttracc(newaccess)= 0;
		colOfAttracc(newaccess)= 0;

		/* transport expression is access to transport attribute */
		ExprToAttracc(trans_expr)= newaccess;

		/* generate information */
		trans_info= malloc (
			strlen (dnameOfSymb(symbref(symbdid))) +
			strlen (nameOfAttrdef(trans_ad)) + 10);
		(void) sprintf (trans_info, "%s<%d>.%s",
			dnameOfSymb(symbref(symbdid)),
			symbno+1,
			nameOfAttrdef(trans_ad));
	} /* if */

	/* combine source attributes and transport list if necessary */
	if (trnsp_source)
	{
	    if (direct_source)
	    {
		if (singleOfConstit(cptr->constit))
		{
			/* multiple CONSTITUENT symbols reachable */
			print_err (
				rowOfProd(prodref(prodidOfAttribution(attrib))),
				colOfProd(prodref(prodidOfAttribution(attrib))),
				"multiple CONSTITUENT symbols",
				COML_ERRID);
			(void) fprintf (ProtocolFile, "\n*** ERROR  ");
			(void) fprintf (ProtocolFile,
				"multiple CONSTITUENT symbols ");
			(void) fprintf (ProtocolFile,
				"in production %s in line %d, col %d\n\n",
				dnameOfProd( prodref (
					prodidOfAttribution(attrib))),
				rowOfProd(prodref(prodidOfAttribution(attrib))),
				colOfProd(prodref(prodidOfAttribution(attrib))));
			cptr->cancelled= TRUE;
			ExprToCall(result)= NULL;
			return (result);
		} /* if */

		/* concatenate lists of transport and source attribute */
		result= gen_concat (
				trans_expr, trans_info, strlen(trans_info)+1,
				source_expr, source_info, strlen(source_info)+1,
				cptr, info, infolen);
	    } /* if */
	    else
	    {
		/* result is access to transport attribute */
		result= trans_expr;
		*info= trans_info;
		*infolen= strlen(trans_info)+1;
	    } /* else */
	} /* if */
	else
	{
	    if (direct_source)
	    {
		/* result is access to source attribute */
		result= source_expr;
		*info= source_info;
		*infolen= strlen (source_info)+1;
	    } /* if */
	    else
	    {
		/* symbol is neither transport nor source symbol */
		/* so the result is empty */
		ExprToCall(result)= NULL;
		*info= (char *) NULL;
		*infolen= 0;
	    } /* else */
	} /* else */

	return (result);
} /* symb_const() */

static Expr
#ifdef __STDC__
rhs_const (Attribution attrib, ConstNode cptr, char **info)
#else
rhs_const (attrib, cptr, info)
Attribution attrib;
ConstNode cptr;
char **info;
#endif
/* create a new expression to get the constituents from rhs symbols	*/
{
	int		pid;
	Expr		result;
	SNode		sn;
	int		sno;
	Expr		concexpr;
	char		*concinfo;
	int		concinfolen;
	Expr		newexpr;
	char		*newinfo;
	int		newinfolen;
	int		infolen;

	ExprToCall(result)= NULL;
	pid= prodidOfAttribution(attrib);

	/* examine all symbols on the rhs of the production */
	/* there must be something to transport! */
	sno= 0;
	for (sn= prodright(pid)->right; sn; sn= sn->right)
	{
		sno++;
		/* consider only src_symb and not shielded trnsp_symb */
		if (! ((symbflag(sn->symbdid) & src_symb) ||
			((symbflag(sn->symbdid) & trnsp_symb) &&
				!(symbflag(sn->symbdid) & shield_symb))))
			continue;
		
		/* create expression to get constituents from rhs symbol */
		newexpr= symb_const (
				FALSE, attrib, cptr,
				sno, sn->symbdid,
				&newinfo, &newinfolen);

		/* new expression must not be empty */
		if (!ExprToCall(newexpr))
		{
			if (!cptr->cancelled)
			{
				print_err (
					rowOfProd(prodref(pid)),
					colOfProd(prodref(pid)),
					"expconstit.c: internal error #4",
					NO_ERRID);
			} /* if */
			cptr->cancelled= TRUE;
			return(newexpr);
		} /* if */

		if (!ExprToCall(result))
		{
			/* result is the single expression */
			result= newexpr;
			*info= (char *) malloc (newinfolen);
			(void) strcpy (*info, newinfo);
			infolen= newinfolen;
		} /* if */
		else
		{
		    if (singleOfConstit(cptr->constit))
		    {
			/* multiple CONSTITUENT symbols reachable */
			print_err (
				rowOfProd(prodref(pid)),
				colOfProd(prodref(pid)),
				"multiple CONSTITUENT symbols",
				COML_ERRID);
			(void) fprintf (ProtocolFile, "\n*** ERROR  ");
			(void) fprintf (ProtocolFile,
				"multiple CONSTITUENT symbols ");
			(void) fprintf (ProtocolFile,
				"in production %s in line %d, col %d\n\n",
				dnameOfProd(prodref(pid)),
				rowOfProd(prodref(pid)),
				colOfProd(prodref(pid)));
			cptr->cancelled= TRUE;
			ExprToCall(result)= NULL;
			return (result);
		    } /* if */
		    else
		    {
			/* concatenate previous and new expression */
			concexpr= gen_concat (
					result, *info, infolen,
					newexpr, newinfo, newinfolen,
					cptr, &concinfo, &concinfolen);
			result= concexpr;
			free (*info);
			*info= (char *) malloc (concinfolen);
			(void) strcpy (*info, concinfo);
			infolen= concinfolen;
		    } /* else */
		} /* else */
	} /* for */

	return (result);
} /* rhs_const() */

static void
#ifdef __STDC__
repl_constit (Attribution attrib, ConstNode cptr)
#else
repl_constit (attrib, cptr)
Attribution attrib;
ConstNode cptr;
#endif
/* replace CONSTITUENT(S) construct by a transport expression		*/
{
	int		prod;
	ConstNode	cn;
	int		rules;
	Call		emptycall;
	Expr		rhsexpr;
	Expr		conexpr;
	SEQExpr		se;
	Expr		nextexpr;
	char		*info;
	int		infolen;
	int		parmcnt;

	if (cptr->cancelled)
		return;

	prod= prodidOfAttribution(attrib);

	protout (ProtocolFile, "\ttarget production %s\n",
		dnameOfProd(prodref(prod)));

	/* handle all constructs in this production */
	rules= 0;
	for (cn= cptr; cn; cn= cn->same)
	{
	    if (cn->prodid == prod)
	    {
		rules++;
		info= NULL;

		/* generate new expression for transport from the rhs symbols */
		if (subtreeOfConstit(cn->constit))
		{
			rhsexpr= symb_const (TRUE, attrib, cn,
					subtreeOfConstit(cn->constit),
					cn->subdid,
					&info, &infolen);
		} /* if */
		else
		{
			rhsexpr= rhs_const (attrib, cn, &info);
		} /* else */

		/* no expression has been generated due to cancellation */
		/* or because there are not constituents below */
		if (!ExprToCall(rhsexpr))
		{
			if (cn->cancelled)
			{
				return;
			} /* if */

			/* initialize call of empty function */
			emptycall= NCall;
			nameOfCall(emptycall)= cn->empty_name;
			initializeSEQExpr (paramsOfCall(emptycall));
			rowOfCall(emptycall)= rowOfProd(prodref(prod));
			colOfCall(emptycall)= colOfProd(prodref(prod));
			ExprToCall(rhsexpr)= emptycall;
			info= malloc (strlen (nameOfCall(emptycall)) + 3);
			(void) sprintf (info, "%s()", nameOfCall(emptycall));
		} /* if */

		/* replace the construct */
		if (cn->parmno == 1) {
			removefirstSEQExpr (paramsOfCall(cn->constcall));
			appendfrontSEQExpr (
				paramsOfCall(cn->constcall),
				rhsexpr);
		} /* if */
		else
		{
		    /* search the parameter of call that precedes	*/
		    /* the parameter containing the construct		*/
		    parmcnt = 0;
		    foreachinSEQExpr (paramsOfCall(cn->constcall), se, conexpr)
		    {	parmcnt++;
			if (emptySEQExpr(tailSEQExpr(se)))
			{
				print_err (
					rowOfConstit(cn->constit),
					colOfConstit(cn->constit),
					"expconstit.c: internal error #5",
					NO_ERRID);
				cn->cancelled= TRUE;
				return;
			} /* if */
			retrievefirstSEQExpr(tailSEQExpr(se), nextexpr);
			if (parmcnt+1 == cn->parmno) {
				removefirstSEQExpr (se->next);
				appendfrontSEQExpr (se->next, rhsexpr);
				break;
			} /* if */
		    } /* foreachinSEQExpr */
		} /* else */

		/* print info */
		protout (ProtocolFile,
			"\tconstruct in line %d, col %d ",
			rowOfConstit(cn->constit),
			colOfConstit(cn->constit));
		protout (ProtocolFile,
			"replaced by the following expression:\n\t\t%s;\n",
			info);
		free(info);
	    } /* if */
	} /* for */

	/* construct must have been found */
	if (!rules)
	{
		print_err (0, 0, "expconstit.c: internal error #6", NO_ERRID);
		cptr->cancelled= TRUE;
		return;
	} /* if */
} /* repl_constit() */

static void
#ifdef __STDC__
gen_constrans (Attribution attrib, ConstNode cptr)
#else
gen_constrans (attrib, cptr)
Attribution attrib;
ConstNode cptr;
#endif
/* generate CONSTITUENT(S) transport					*/
{
	int	pid;
	Expr	rhsexpr;
	Expr	expr;
	Symb	symb;
	Attrdef	ad;
	Attracc	access;
	Call	new;
	Name	v_name;
	char	*info;

	pid= prodidOfAttribution(attrib);

	/* generate new expression for non empty transport from rhs symbols */
	info= NULL;
	rhsexpr= rhs_const (attrib, cptr, &info);

	/* expression must have been generated */
	if (!ExprToCall(rhsexpr))
	{
	    if (!cptr->cancelled)
	    {
		print_err (
			rowOfConstit(cptr->constit),
			colOfConstit(cptr->constit),
			"expconstit.c: internal error #10", NO_ERRID);
		cptr->cancelled= TRUE;
	    } /* if */
	    return;
	} /* if */

	symb= symbref(prodright(pid)->symbdid);

	/* generate new assignment */

	/* transport attribute should be the first attribute */
	retrievefirstSEQAttrdef (attrsOfSymb(symb), ad);
	if (strcmp(genattrOfConstit(cptr->constit), nameOfAttrdef(ad)))
	{
		print_err (
			rowOfConstit(cptr->constit),
			colOfConstit(cptr->constit),
			"expconstit.c: internal error #11", NO_ERRID);
		cptr->cancelled= TRUE;
		return;
	} /* if */

	/* initialize call of assignment */
	new= NCall;
	nameOfCall(new)= ASSIGNFCT;
	initializeSEQExpr (paramsOfCall(new));
	rowOfCall(new)= rowOfProd(prodref(pid));
	colOfCall(new)= colOfProd(prodref(pid));

	/* initialize left hand side of the new assignment */
	access= NAttracc;
	symbnoOfAttracc(access)= 0;	/* lhs symbol of production */
	attridOfAttracc(access)= didOfAttrdef(ad);
	rowOfAttracc(access)= rowOfProd(prodref(pid));
	colOfAttracc(access)= colOfProd(prodref(pid));
	ExprToAttracc(expr)= access;
	appendrearSEQExpr (paramsOfCall(new), expr);

	/* append expression from rhs constituents to assignment call */
	appendrearSEQExpr (paramsOfCall(new), rhsexpr);

	/* append  special name to mark assignments to VOID attributes */
	if (cptr->list_tid == DIDVOID
	    || cptr->list_tid == DIDVOLI
	    || cptr->dep)
	{
		v_name= NName;
		nOfName(v_name)= TNVOID;
		rowOfName(v_name)= rowOfProd(prodref(pid));
		colOfName(v_name)= colOfProd(prodref(pid));
		ExprToName(expr)= v_name;
		appendrearSEQExpr (paramsOfCall(new), expr);
	} /* if */

	/* append new call to the list of attribution rules */
	ExprToCall(expr)= new;
	appendrearSEQAttrrule(attrrulesOfAttribution(attrib), expr);
	newassigns++;

	/* print informations */
	protout (ProtocolFile, "\ttransport in production %s\n",
		dnameOfProd(prodref(pid)));
	protout (ProtocolFile, "\t%s<1>.%s :=\n\t\t%s;\n",
		dnameOfSymb (symb),
		genattrOfConstit (cptr->constit),
		info);
	free(info);
} /* gen_constrans() */

static void
#ifdef __STDC__
expand_constituent (ConstNode cptr)
#else
expand_constituent (cptr)
ConstNode cptr;
#endif
/* expand CONSTITUENT(S) construct by generating assignments and calls	*/
{
	SEQAttribution	tmp;
	Attribution	attrib;
	int		pid;

	if (cptr->cancelled)
	{
		protout (ProtocolFile, "\t\texpansion cancelled\n");
		return;
	}

	/* examine all attributions of productions */
	foreachinSEQAttribution (attrrulesOfAttrEval(IdlStruct), tmp, attrib)
	{
		pid= prodidOfAttribution(attrib);
		if (prodflag(pid) & target_prod)
			/* replace CONSTITUENT(S) constructs */
			repl_constit (attrib, cptr);

		if (prodflag(pid) & trans_prod)
			/* retrieve source and transport attributes	*/
			/* from the rhs symbols and collect them in the	*/
			/* transport attribute of the lhs symbol	*/
			gen_constrans (attrib, cptr);

		else if (symbflag (lhsOfProd(prodref(pid))) & trnsp_symb)
			/* nothing to transport here */
			{
		    if (singleOfConstit(cptr->constit))
		    {
			/* CONSTITUENT symbol not reachable */
			print_err (
				rowOfProd(prodref(pid)),
				colOfProd(prodref(pid)),
				"CONSTITUENT symbol not reachable",
				0);
			(void) fprintf (ProtocolFile, "\n*** ERROR  ");
			(void) fprintf (ProtocolFile,
				"CONSTITUENT symbol not reachable ");
			(void) fprintf (ProtocolFile,
				"in production %s in line %d, col %d\n\n",
				dnameOfProd(prodref(pid)),
				rowOfProd(prodref(pid)),
				colOfProd(prodref(pid)));
			cptr->cancelled= TRUE;
		    } else
			gen_empty (attrib, cptr);
			}
	} /* foreachinSEQAttribution */
} /* expand_constituent() */

/*
** GLOBAL FUNCTIONS
*/

void
#ifdef __STDC__
find_constits (void)
#else
find_constits()
#endif
/* search all CONSTITUENT(S) constructs and build a list for them	*/
{
	SEQAttribution	tmp1;
	Attribution	attrib;
	SEQAttrrule	tmp2;
	Attrrule	rule;
	int		counter;

	newattrname= (char *) malloc (strlen (CONST_ATTR) + 9);
	if (!newattrname)
		deadly ("expand: out of memory\n");
	constscount= 0;
	constituents= (ConstNode) NULL;
	counter= 0;

	foreachinSEQAttribution (attrrulesOfAttrEval(IdlStruct), tmp1, attrib)
	{
	    foreachinSEQAttrrule (attrrulesOfAttribution(attrib), tmp2, rule)
	    {
		if (typeof(rule) == KCall)
		{
			counter+= constits_in_call (
					AttrruleToCall(rule),
					prodidOfAttribution(attrib),
					FALSE);
		} /* if */
	    } /* foreachinSEQAttrrule */
	} /* foreachinSEQAttribution */

	switch (counter)
	{
	    case 0:
		(void) fprintf(ProtocolFile, "\nNo CONSTITUENT(S)s found\n");
		break;
	    case 1:
		(void) fprintf(ProtocolFile, "\nOne CONSTITUENT(S) found\n");
		break;
	    default:
		(void) fprintf(ProtocolFile,
			"\n%d CONSTITUENT(S)s found\n", counter);
	} /* switch */

	free (newattrname);
} /* find_constits() */

void
#ifdef __STDC__
expand_constits (void)
#else
expand_constits()
#endif
/* expand all CONSTITUENT(S) constructs					*/
{
	ConstNode	cptr, cptr2, cptr3;

	if (!constscount)
		return;

	newattrs= 0;
	newassigns= 0;
	protout (ProtocolFile, "\nCONSTITUENT(S) expansion:");

	/* examine all constituent(s) constructs in the list	*/
	for (cptr= constituents; cptr; cptr=cptr->next)
	{
		clear_psflags();

		if (EXP_PROT)
			show_constituent (cptr);

		mark_constituent (cptr);
		expand_constituent (cptr);
	} /* for */

	/* release memory occupied by nodes in global list */
	for (cptr= constituents; cptr; cptr=cptr2)
	{
		for (cptr2= cptr->same; cptr2; cptr2= cptr3)
		{
			cptr3= cptr2->same;
			free (cptr2);
		} /* for */
		cptr2= cptr->next;
		free (cptr);
	} /* for */
	constituents= (ConstNode) NULL;

	if (constscount==1)
	{
		(void) fprintf (ProtocolFile, "One CONSTITUENT(S) expanded; ");
	} /* if */
	else
	{
		(void) fprintf (ProtocolFile,
			"%d different CONSTITUENT(S)s expanded; ", constscount);
	} /* else */

	switch (newattrs)
	{
	    case 0:
		(void) fprintf (ProtocolFile, "no new attributes, ");
		break;
	    case 1:
		(void) fprintf (ProtocolFile, "one new attribute, ");
		break;
	    default:
		(void) fprintf (ProtocolFile, "%d new attributes, ",
			newattrs);
		break;
	} /* switch */

	switch (newassigns)
	{
	    case 0:
		(void) fprintf (ProtocolFile, "no new assignments\n");
		break;
	    case 1:
		(void) fprintf (ProtocolFile, "one new assignment\n");
		break;
	    default:
		(void) fprintf (ProtocolFile, "%d new assignments\n",
			newassigns);
	} /* switch */
} /* expand_constits() */

/***********************\
* end of expconstit.c	*
\***********************/

								/* HaDeS */

