/*
** $Id: ccomar.h,v 1.2 90/10/23 11:35:21 cogito Exp $
*/

/* ccomar.h - definition of C COMAR internal data types, primitive operations
 *
 *	Author: Karl - Josef Prott
 *		University of Paderborn
 *
 *	This module defines an implementation for COMAR internal structures.
 *	Included are primitives and other functions for creating a new
 *	empty COMAR structure, creating components to be added to a structure,
 *	storing and fetching components and their attributes, and deleting
 *	components from a structure.
 *
 *	Most of this file consists of primitive operations; it is expected 
 *	that most COMAR manipulations will occur via higher level routines
 *	built upon the primitives. In particular, the functions provided 
 *	to create and initialize a new COMAR component ("cmr_new_xxx") should
 *	usually be used rather than the primitives ("cmr_xxx"). The former is
 *	the only way to create new symb_, def_, and attr_entry's although
 *	the subtrees of def_entry's and attr_entry's can be built up on a
 *	per-component basis.
 * Last Change: 15.02.90	From: Kalle
 */

#ifndef ccomar_DEF
#define ccomar_DEF


#include "comar.h"


/* Definitions exported by COMAR */

/* TABLE of CONTENTS:
 *
 * 	1. Condition codes returned by some COMAR functions.
 * 	2. COMAR structure definitions.
 *	3. Creating and initializing new COMAR components.
 *	4. Building COMAR subtrees "bottom-up".
 *	5. Manipulating COMAR lists.
 *	6. Deleting COMAR components and subtrees.
 *	7. Fetching COMAR attributes of COMAR components.
 *	8. Storing new attribute values to COMAR attributes.
 *
 */


/* SECTION 1. Condition codes returned by some COMAR functions.*/

#define CMR_SUCCESS	 0
#define CMR_UNKERR	-1
#define CMR_OPNERR	-2
#define CMR_WRTERR	-3
#define CMR_RDERR	-4
#define CMR_INPERR	-5

#ifdef DEBUG
#define DBG_INPERR	-5	/* ccomario.c may redefine CMR_INPERR */
#endif				/*     for debugging purposes.        */

/* a boolean type: */
typedef char boolean;

#ifndef FALSE
#define FALSE   0 
#endif

#ifndef TRUE
#define TRUE    1
#endif

/* make this a proper pointer */
#ifndef NULL
#define NULL (char *)0
#endif



/* new types for working with generic functions */

/* The type 'Gen_class' represents as a generic type all subtypes,   */
/* from the union-types 'unit', 'value' and 'prop_entry'. It is	     */
/* allowed to use 'Gen_class' as a result-type of a function. If a   */
/* formal parameter is of a subtype from 'value', 'unit' or          */
/* 'prop_entry', the actual one can be of type 'Gen_class'.          */
typedef int   *Gen_class;

/* 'Gen_unit' represents as a generic type 'p_alt',...,'p_elunit'.    */
typedef p_alt  Gen_unit;
/* 'Gen_val' represents as a generic type 'p_sval',...,'p_lval'.      */
typedef p_sval Gen_val;
/* 'Gen_prop' represents as a generic type 'p_prop' and 'p_prop_val'. */
typedef p_prop Gen_prop;


/* SECTION 2. COMAR structure definitions. */

/* Basic COMAR node attributes: */
typedef int SID;	/* non-negative symbol encoding */
typedef int DID;	/* non-negative definition encoding */

/* return maximum SID, DID in COMAR structure: */
extern SID cmr_get_maxsid( /* p_comar c; */ );
extern DID cmr_get_maxdid( /* p_comar c; */ );

/* define constants for two predefined COMAR symbols: */
#define S_NONAME	1
#define S_EMPTY		2





/* COMAR node tags: */
#define P_COMAR     Kp_comar    /* jetzt Wert 04 frueher 01 */

/* Symbol part */
#define P_STRING    Kp_string   /* jetzt Wert 34 frueher 02 */
#define P_NAME	    Kp_name     /* jetzt Wert 14 frueher 03 */

/* Definition part */
#define P_TERM	    Kp_term      /* jetzt Wert 38 frueher 04 */
#define P_NTERM     Kp_nterm     /* jetzt Wert 16 frueher 05 */
#define P_OTHER     Kp_other     /* jetzt Wert 22 frueher 06 */
#define P_PROD	    Kp_prod      /* jetzt Wert 26 frueher 07 */

/* Production subtrees */
#define P_ALT	    Kp_alt       /* jetzt Wert 02 frueher 08 */
#define P_OPT	    Kp_opt       /* jetzt Wert 20 frueher 09 */
#define P_PLUS	    Kp_plus      /* jetzt Wert 24 frueher 10 */
#define P_STAR	    Kp_star      /* jetzt Wert 32 frueher 11 */
#define P_DELREP    Kp_delrep    /* jetzt Wert 06 frueher 12 */
#define P_ELUNIT    Kp_elunit    /* jetzt Wert 10 frueher 13 */

/* Property part */
#define P_PROP	   Kp_prop       /* jetzt Wert 28 frueher 16 */
#define P_PROP_VAL Kp_prop_val   /* jetzt Wert 30 frueher 17 */
#define P_SVAL	   Kp_sval       /* jetzt Wert 36 frueher 18 */
#define P_NVAL	   Kp_nval       /* jetzt Wert 18 frueher 19 */
#define P_DVAL	   Kp_dval       /* jetzt Wert 08 frueher 20 */
#define P_LVAL	   Kp_lval       /* jetzt Wert 12 frueher 21 */


/*  Primitives to create new COMAR structure node variants.
 *  No attribute initialization is performed (but variants are tagged).
 *  It is necessary to test return value for (p_comar)NULL to determine
 *  if an allocation error occurred.
 */

#define cmr_alt()     (Np_alt)
#define cmr_opt()     (Np_opt)
#define cmr_plus()    (Np_plus)
#define cmr_star()    (Np_star)
#define cmr_delrep()  (Np_delrep)
#define cmr_elunit()  (Np_elunit)
#define cmr_sval()    (Np_sval)
#define cmr_nval()    (Np_nval)
#define cmr_dval()    (Np_dval)
#define cmr_lval()    (Np_lval)
/* These macros produce the same effect as functions of the following */
/* form :                                                             */ 
/* extern  p_alt    cmr_alt();                                        */
/* extern  p_opt    cmr_opt();                                        */
/* extern  p_plus   cmr_plus();                                       */
/* extern  p_star   cmr_star();                                       */
/* extern  p_delrep cmr_delrep();                                     */
/* extern  p_elunit cmr_elunit();                                     */
/* extern  p_sval   cmr_sval();                                       */
/* extern  p_nval   cmr_nval();                                       */
/* extern  p_dval   cmr_dval();                                       */
/* extern  p_lval   cmr_lval();                                       */



/* SECTION 3. Creating and initializing new COMAR components. */


/* Return new empty COMAR structure (initialized for pre-defined symbols) */
extern p_comar cmr_new_structure();


/*  Insert new symbol into set of symb_entry's. Repeat call with same sym is
 *  ignored. A symb_entry can be neither deleted nor modified.
 */
extern SID cmr_new_symb_entry( /* p_comar c; int tag; String str; */ );


/*  Insert new term, nterm, or other into list of def_entry's. Repeat call  
 *  with same SID results in a second entry with the same SID. This stands 
 *  in contrast to Plummer's implementation !
 */
extern DID cmr_new_def_entry( /* p_comar c; int tag; SID sid; */);


/*  Insert new production into list of def_entry's. Repeat call with 
 *  same SID results in a second entry with the same SID. This stands 
 *  in contrast to Plummer's implementation !
 */
extern DID cmr_new_prod( /* p_comar c; SID sid; DID lhs; SEQunit units; */ );


/*  Insert new property identified by SID into set of properties of def_entry
 *  identified by DID. Repeat call of either with same SID and DID is ignored.
 *  If DID does not identify a def_entry then return == is CMR_UNKERR
 *  else return == argument did.
 */
extern DID cmr_new_prop( /* p_comar c; DID did; SID sid; */ ); 
extern DID cmr_new_prop_val( /*p_comar c; DID did; SID sid; Gen_val value;*/ ); 



/* SECTION 4. Building COMAR subtrees "bottom-up". */

/*  COMAR has two kinds of explicit subtrees - Property subtrees and
 *  Production subtrees. Use the following to build them "bottom-up".
 */

/* Property sub-trees: */
/* extern p_sval  cmr_new_sval( SID sid );	 */
/* extern p_nval  cmr_new_nval( int val );	 */
/* extern p_dval  cmr_new_dval( DID did );	 */
/* extern p_lval  cmr_new_lval( SEQvalue list ); */
#define	cmr_new_sval(sid)	(Mkp_sval(sid))
#define	cmr_new_nval(val)	(Mkp_nval(val))
#define	cmr_new_dval(did)	(Mkp_dval(did))
#define	cmr_new_lval(list)	(Mkp_lval(list))

/*  Production subtrees: */
/*  extern p_alt    cmr_new_alt(SEQunit units1, units2);	 */
/*  extern p_opt    cmr_new_opt(SEQunit units);			 */
/*  extern p_plus   cmr_new_plus(SEQunit units);		 */
/*  extern p_star   cmr_new_star(SEQunit units);		 */
/*  extern p_delrep cmr_new_delrep(SEQunit units1, units2);	 */
/*  extern p_elunit cmr_new_elunit(DID elem);			 */
#define	cmr_new_alt(units1, units2)	(Mkp_alt(units1, units2))
#define	cmr_new_opt(units)		(Mkp_opt(units))
#define	cmr_new_plus(units)		(Mkp_plus(units))
#define	cmr_new_star(units)		(Mkp_star(units))
#define	cmr_new_delrep(units1, units2)	(Mkp_delrep(units1, units2))
#define	cmr_new_elunit(elem)		(Mkp_elunit(elem))



/* SECTION 5. Manipulating COMAR lists. */

/* Create new empty list */
#define	cmr_new_list()	(NULL)

/* Return TRUE iff list is empty */
#define	cmr_isempty_list(list)	((pGenList)(list) == NULL)

/* Add item to front of list */
#define	cmr_add_to_list(list, item)	( IDLListAddFront(list, item) )

/* Return concatenation of second list to end of first */
extern pGenList cmr_cat_lists( /* pGenList list1, list2; */ );

/* Return list2 with list1 spliced in after! item in list2 addressed by p */
extern pGenList cmr_splice_lists( /* pGenList list1, list2; Gen_class p */ );

/* splits list after item addressed by p and returns splitted list */
extern pGenList cmr_split_list( /* pGenList list, Gen_class p */ );

/* Return item at head of list */
#define	cmr_list_head(list)	( (Gen_class)IDLListRetrieveFirst(list) )

/* Return list less the item at its head */
#define	cmr_list_tail(list)	\
		( (pGenList)(list)? ((pGenList)(list))->next : (pGenList)NULL )

/* Return number of elements in list */
#define cmr_list_len(list)	(IDLListLength(list))

/* Return list with indicated item no longer in list */
#define	cmr_list_delitem(list, item)	( (pGenList)(list)? \
		IDLListRemoveCell((pGenList)list,item) : (pGenList)NULL )


/* SECTION 6. Deleting COMAR components and subtrees. */

/*  Symb_entry's cannot be deleted. Deletion of a subtree node causes deletion
 *  of all its descendants, as in the case of components of production or
 *  property subtrees. If they are to be saved for reattachment fetch
 *  and save the roots of subtrees and store null subtrees in their place 
 *  before destroying the component. All return the modified structure.
 */

/* Delete an arbitrary subtree . t may be of type 'value', 'unit' or
 * 'prop_entry'. For delete of sequences use cmr_del_list.
 */
extern p_comar cmr_del_subtr( /* Gen_class t; */ );

/* Delete all elements of the list and frees their memory .
 * list may be of any type of the form SEQxxx .
 */
extern p_comar cmr_del_list( /* pGenList  list; */ );

/* Delete def_entry or prod indicated by did. */
extern p_comar cmr_del_def_entry( /* p_comar c; DID did; */ );

/* Delete property identified by sid, held by did. */
extern p_comar cmr_del_prop_entry( /* p_comar c; DID did; SID sid; */ );


/* SECTION 7. Fetching COMAR attributes of COMAR components. */

/* The first group return CMR_UNKERR if the COMAR component
 *  indicated by the argument does not exist.
 */

/* return tag of any COMAR node */
extern int cmr_get_tag( /* Gen_class c; */ );


/* return COMAR attributes of symb_entry identified by SID */
extern String cmr_get_symb_entry_sym( /* p_comar c; SID sid; */ ); 
extern int cmr_get_symb_entry_tag( /* p_comar c; SID sid; */ ); 

/* return SID of symb_entry identified by SYM */
extern SID cmr_get_symb_entry_sid( /* p_comar c; String str; */ ); 

/* return COMAR attributes of def_entry or prod identified by DID */
extern SID cmr_get_def_entry_sid( /* p_comar c; DID did; */ ); 
extern int cmr_get_def_entry_tag( /* p_comar c; DID did; */ ); 
extern DID cmr_get_prod_lhs( /* p_comar c; DID did */ ); 

/* return elem did of elunit subtee nodes */
extern DID cmr_get_elunit_elem( /* unit elunit; */ ); 



/* return value of simple valued property */
/* extern SID cmr_get_sval_sid( value sval); */
/* extern int cmr_get_nval_val( value nval); */
/* extern DID cmr_get_dval_did( value dval); */
#define	cmr_get_sval_sid(sval)	\
		( (((p_sval)(sval))->IDLhidden.TypeID == P_SVAL) ? \
		((p_sval)(sval))->sid : CMR_UNKERR )
#define	cmr_get_nval_val(nval)	\
		( (((p_nval)(nval))->IDLhidden.TypeID == P_NVAL) ? \
		((p_nval)(nval))->val : CMR_UNKERR )

#define	cmr_get_dval_did(dval)	\
		( (((p_dval)(dval))->IDLhidden.TypeID == P_DVAL) ? \
		((p_dval)(dval))->did : CMR_UNKERR )



/*  The next group of functions return NULL if the COMAR 
 *  component does not exist.
 */

/* return list of list-valued property */
/* extern SEQvalue cmr_get_lval_list(value lval); */
#define	cmr_get_lval_list(lval)	\
		( (((p_lval)(lval))->IDLhidden.TypeID == P_LVAL) ? \
		((p_lval)(lval))->list : (SEQvalue)NULL )


/* return subtree of a production definition or subtree node */
extern SEQunit  cmr_get_prod_units( /* p_comar c; DID did */ );

/* extern SEQunit  cmr_get_alt_units1(Gen_unit alt); */
/* extern SEQunit  cmr_get_alt_units2(Gen_unit alt); */
/* extern SEQunit  cmr_get_opt_units(Gen_unit opt); */
/* extern SEQunit  cmr_get_plus_units(Gen_unit plus); */
/* extern SEQunit  cmr_get_star_units(Gen_unit star); */
/* extern SEQunit  cmr_get_delrep_units1(Gen_unit delrep); */
/* extern SEQunit  cmr_get_delrep_units2(Gen_unit delrep); */
#define	cmr_get_alt_units1(alt)	\
		( (((p_alt)(alt))->IDLhidden.TypeID == P_ALT) ? \
		((p_alt)(alt))->rhs1 : (SEQunit)NULL )

#define	cmr_get_alt_units2(alt)	\
		( (((p_alt)(alt))->IDLhidden.TypeID == P_ALT) ? \
		((p_alt)(alt))->rhs2 : (SEQunit)NULL )

#define	cmr_get_opt_units(opt)	\
		( (((p_opt)(opt))->IDLhidden.TypeID == P_OPT) ? \
		((p_opt)(opt))->rhs : (SEQunit)NULL )

#define	cmr_get_plus_units(plus)	\
		( (((p_plus)(plus))->IDLhidden.TypeID == P_PLUS) ? \
		((p_plus)(plus))->rhs : (SEQunit)NULL )

#define	cmr_get_star_units(star)	\
		( (((p_star)(star))->IDLhidden.TypeID == P_STAR) ? \
		((p_star)(star))->rhs : (SEQunit)NULL )

#define	cmr_get_delrep_units1(delrep)	\
		( (((p_delrep)(delrep))->IDLhidden.TypeID == P_DELREP) ? \
		((p_delrep)(delrep))->rhs : (SEQunit)NULL )

#define	cmr_get_delrep_units2(delrep)	\
		( (((p_delrep)(delrep))->IDLhidden.TypeID == P_DELREP) ? \
		((p_delrep)(delrep))->sep : (SEQunit)NULL )


/* return list of all properties of definition identified by did. */
extern SEQprop_entry cmr_get_prop_entries( /* p_comar c; DID did; */ );

/* return ptr. to 1st property of def; modifications will be permanent */
extern Gen_prop cmr_get_first_prop( /* p_comar c; DID did: */ );

/* return ptr. to next property of def; modifications will be permanent */
extern Gen_prop cmr_get_next_prop( /* p_comar c; DID did; Gen_prop p; */ );
 

/* the next four functions are taken from newcomar, but now implemented
 * with IDL.
 */

/*  Return the first instance found of the property named by SID, held by 
 *  the def_entry identified by DID (use cmr_get_prop_tag() on returned
 *  property to determine if it has a value subtree).
 */
extern Gen_prop cmr_get_prop( /* p_comar c; DID did; SID sid; */ );

/*  Return SID of property addressed by p. CMR_UNKERR ==> p is not a property. */
extern SID cmr_get_prop_sid(/* Gen_prop prop */);

/*  Return tag of property addressed by p. CMR_UNKERR ==> p is not a property. */
extern int cmr_get_prop_tag(/* Gen_prop prop */);

/*  Return value subtree of property addressed by p. Return of
 *  NULL indicates there is none (property is P_PROP not P_PROP_VAL).
 */
extern Gen_val cmr_get_prop_val(/* Gen_prop prop */);




/* SECTION 8. Storing new attribute values to COMAR attributes. */

/*  There are no store functions for symb_entry's since they cannot be
 *  modified. The following functions return NULL if the COMAR 
 *  component does not exist; otherwise they return their first argument
 *  after setting the specified attribute of the COMAR component.
 */

/* Set SID of def_entry identified by DID */
extern p_comar cmr_set_def_entry_sid( /* p_comar c; DID did; SID sid; */ ); 

/* Set components of a production definition */
extern p_comar cmr_set_prod_lhs( /* p_comar t; DID did, lhs; */ ); 
extern p_comar cmr_set_prod_units( /* p_comar t; DID did; SEQunit units; */ ); 

/* set subtree of subtree node of specified type */
extern p_alt  cmr_set_alt_units1( /* Gen_unit alt; SEQunit units1; */ ); 
extern p_alt  cmr_set_alt_units2( /* Gen_unit alt; SEQunit units2; */ ); 
extern p_opt  cmr_set_opt_units( /* Gen_unit opt; SEQunit units; */ ); 
extern p_plus  cmr_set_plus_units( /* Gen_unit plus; SEQunit units; */ ); 
extern p_star  cmr_set_star_units( /* Gen_unit star; SEQunit units; */ ); 
extern p_delrep  cmr_set_delrep_units1( /*Gen_unit delrep; SEQunit units1;*/ );
extern p_delrep  cmr_set_delrep_units2( /*Gen_unit delrep; SEQunit units2;*/ );
extern p_elunit  cmr_set_elunit_elem( /* Gen_unit elunit; DID elem; */ ); 

extern p_comar cmr_set_prop_val( /* p_comar c; DID did; SID sid; Gen_val val */);     
extern p_sval  cmr_set_sval_sid( /* Gen_val c; SID sid; */);     
extern p_nval  cmr_set_nval_val( /* Gen_val c; int val; */);     
extern p_dval  cmr_set_dval_did( /* Gen_val c; DID did; */);     
extern p_lval  cmr_set_lval_list( /* Gen_val c; SEQvalue list; */);     


#endif !ccomar_DEF
