static char RCSid[] = "$Id: op_chain.c,v 1.12 1992/07/26 23:23:11 waite Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */

/* This module contains routines to eliminate chain rules in an EBNF
 * concrete grammar structure as part of developing an abstract grammar.
 *
 * Routines:
 *      1) get_chainrule_list - Obtain a list of potential chain rules.
 *      2) delete_chainrule_list - Delete the list of chain rules.
 *      3) build_eq_list - Construct a list of all rules or identifiers
 *                         from the potential chain rule list.
 *      4) delete_eq_list - Delete the list of all rules or identifiers
 *                         from the potential chain rule list.
 *      5) build_eq_structure - Combine a rule and identifier eq_list
 *                              together into an equivalence structure.
 *      6) delete_eq_structure - Delete an existing equivalence structure.
 *      7) mark_eq_class - Place a common group number in the class field
 *                         of all identifiers in a given equivalence class.
 *      8) mark_eq_classes - Mark all equivalence classes in a given
 *                           equivalence structure.
 *      9) output_eq_class - Output the members of an equivalence class
 *                           to a curses window.
 *     10) get_eq_class_nth_node - Returns a pointer to the nth
 *                                 node in a given equivalence class.
 *     11) verify_chain_rules - Interatively determine a set of rules
 *                              from the list of potential rules that
 *                              the user wants considered as chain rules
 *                              and as such, eliminated from the grammar.
 *     12) verify_eq_structure - Iteratively determine a set of equivalence
 *                               classes based upon the current potential
 *                               chain rules that satisfy the user.
 *     13) rename_eq_classes - Given an approved equivalence class, rename
 *                             all members of the group to a common user
 *                             supplied name.
 *     14) delete_chain_rules - Delete all rules considered to be chain
 *                              rules from the EBNF grammar structure.
 *     15) process_chainrules - Interactively process chain rules in an
 *                              EBNF grammar structure with the goal of
 *                              eliminating non-essential chain rules.
*/



#include <stdio.h>
#include <stdlib.h>
#include <curses.h>
#include <math.h>
#include "cagt_config.h"
#include "cagt_usr_err.h"
#include "support.h"
#include "io_curses.h"
#include "gram.h"



typedef struct chain_rule CHAIN_RULE;
typedef struct chain_rule *CHAIN_RULE_PTR;
typedef struct eq_node EQ_NODE;
typedef struct eq_node *EQ_NODE_PTR;






/*
 * eq_node is used to build the equivalence class data structure.
*/
struct eq_node {
        char type;                                       /* RULE or NONTERM */
        short int class;                     /* Number of equivalence class */
        union {
              CHAIN_RULE_PTR cr;	   /* Points at chain rule list elt */
              SYMBOL_PTR_NODE_PTR nt;	    /* Pointer to pointer to SYMBOL */
              } x;
        EQ_NODE_PTR left,	              /* Rules with nonterm on left */
                    right;	             /* Rules with nonterm on right */
        EQ_NODE_PTR next;	/* Allows keeping list of rules or nonterms */
        };


/* Possible values for type field of eq_node struct */
#define RULE 1
#define NONTERM 2



/* chain_rule is used to keep track of possible chain rules */
struct chain_rule {
        char valid;       /* TRUE if rule should be considered a chain rule */
        LEFT_NODE_PTR rule;	     /* Left side of rule in EBNF structure */
        SYMBOL_PTR_NODE_PTR nt;	     /* Points to nonterminal text on right */
        CHAIN_RULE_PTR next;	                         /* Next chain rule */
        };








private CHAIN_RULE_PTR get_chainrule_list(start)
   LEFT_NODE_PTR start;
/*
 * On Entry:
 *    start points at the top of the list of rules in an existing EBNF
 *       grammar structure.
 *
 * On Exit:
 *    get_chainrule_list returns a list of chainrules. The valid fields
 *       of all rules in the list are set to TRUE.
*/
   {
   SYMBOL_PTR_NODE_PTR nt_symb;
   CHAIN_RULE_PTR top = (CHAIN_RULE_PTR)0,
                  cur = (CHAIN_RULE_PTR)0;

   while (start)
        {
        if (start->code != NULT)                            /* Deleted Rule */
           if (start->chain_rule)
                {
		(void) is_chainrule(start->right_side,FALSE,&nt_symb,FALSE);
                if (cur)
                        {
			GET_MEMORY(cur->next, CHAIN_RULE_PTR, 1, CHAIN_RULE,
				   "get_chainrule_list", 1)
                        cur = cur->next;
                        }
                   else
			GET_MEMORY(top =cur, CHAIN_RULE_PTR, 1, CHAIN_RULE,
				   "get_chainrule_list", 2)
                cur->valid = TRUE;
                cur->next = (CHAIN_RULE_PTR)0;
                cur->rule = start;
                cur->nt = nt_symb;
                }
        start = start->next_rule;
        }
   return(top);
   }







private void delete_chainrule_list(list)
   CHAIN_RULE_PTR *list;
/*
 * On Entry:
 *    *list points at the top of the list of rules in an existing EBNF
 *       grammar structure.
 *
 * On Exit:
 *    *list is NULL. All storage used by the chain rule list has been
 *       deallocated.
*/
   {
   CHAIN_RULE_PTR temp;

   while (*list)
        {
        temp = *list;
        *list = (*list)->next;
	FREE_MEMORY(temp, "delete_chainrule_list", 1)
        }

   }







private EQ_NODE_PTR build_eq_list(type,chain_rule_list)
   char type;
   CHAIN_RULE_PTR chain_rule_list;
/*
 * On Entry:
 *      type contains either RULE or NONTERM.
 *      chain_rule_list points at a valid chain rule list.
 *
 * On Exit:
 *      if (type == RULE)
 *         build_eq_list returns a linked list containing RULE eq_node(s)
 *            for each of the rules in chain_rule list.
 *      otherwise
 *         build_eq_list returns a linked list containing NONTERM eq_node(s)
 *            for each of the rules in chain_rule list.
 *      In either case:
 *              The type field is set to type
 *              The class field is set to NULL
 *              The left and right nodes point at the node itself
*/
   {
   EQ_NODE_PTR list = (EQ_NODE_PTR)0,
               list_top = (EQ_NODE_PTR)0,
               cur;
   char found[2];
   short int i;

   if (type == RULE)
         while (chain_rule_list)
            {
            if (chain_rule_list->valid)
               {
               if (list)
                     {
		     GET_MEMORY(list->next, EQ_NODE_PTR, 1, EQ_NODE,
				"build_eq_list", 1)
                     list = list->next;
                     }
                  else
		     GET_MEMORY(list_top = list, EQ_NODE_PTR, 1, EQ_NODE,
				"build_eq_list", 2)
               list->type = RULE;
               list->class = (short int)0;
               list->x.cr = chain_rule_list;
               list->left = list->right = list;
               list->next = (EQ_NODE_PTR)0;
               }
            chain_rule_list = chain_rule_list->next;
            }
      else
         while (chain_rule_list)
            {
            if (chain_rule_list->valid)
               {
               cur = list_top;
               found[0] = FALSE;
               found[1] = ( chain_rule_list->rule->text->symbol_ptr ==
                            chain_rule_list->nt->symbol_ptr );
               while ( cur && !(found[0] && found[1]) )
                  {
                  if (chain_rule_list->rule->text->symbol_ptr ==
                      cur->x.nt->symbol_ptr)
                        found[0] = TRUE;
                  if (chain_rule_list->nt->symbol_ptr ==cur->x.nt->symbol_ptr)
                        found[1] = TRUE;
                  cur = cur->next;
                  }
               for (i=0; i < 2; i++)
                  if (!found[i])
                     {
                     if (list)
                           {
			   GET_MEMORY(list->next, EQ_NODE_PTR, 1, EQ_NODE,
				      "build_eq_list", 3)
                           list = list->next;
                           }
                        else
			   GET_MEMORY(list_top=list, EQ_NODE_PTR, 1, EQ_NODE,
				      "build_eq_list", 4)
                                               
                     list->type = NONTERM;
                     list->class = (short int)0;
                     if (i == 0)
                           list->x.nt = chain_rule_list->rule->text;
                        else
                           list->x.nt = chain_rule_list->nt;
                     list->left = list->right = list;
                     list->next = (EQ_NODE_PTR)0;
                     }
                  }
            chain_rule_list = chain_rule_list->next;
            }
   return(list_top);
   }







private void delete_eq_list(list)
   EQ_NODE_PTR *list;
/*
 * On Entry:
 *      *list points at the top of the list of eq_nodes.
 *
 * On Exit:
 *    *list is NULL. All storage used by the eq_node list has been
 *       deallocated.
*/
   {
   EQ_NODE_PTR temp;

   while (*list)
        {
        temp = *list;
        *list = (*list)->next;
	FREE_MEMORY(temp, "delete_eq_list",1)
        }

   }







private void build_eq_structure(chain_rule_list,rules,nonterms)
   CHAIN_RULE_PTR chain_rule_list;
   EQ_NODE_PTR *rules,
               *nonterms;
/*
 * On Entry:
 *      chain_rule_list contains the current possible chain rules.
 *      *rules is NULL.
 *      *nonterms is NULL.
 *
 * On Exit:
 *      *rules points at a linked list of chain rules in the eq_structure
 *      *nonterms points at a linked list of non terminal nodes in the
 *         eq_structure.
 *      *rules and *nonterms have been combined into an equivalence structure.
*/
   {
   EQ_NODE_PTR cur_rules;
   EQ_NODE_PTR cur_nonterms;

   /* Get one eq_node for each rule and each non-terminal */
   *rules = build_eq_list(RULE,chain_rule_list);
   *nonterms = build_eq_list(NONTERM,chain_rule_list);

   /* Link the nodes together into an equivalence structure */
   cur_nonterms = *nonterms;
   while (cur_nonterms)
        {
        cur_rules = *rules;
        while (cur_rules)
           {
           if (cur_rules->x.cr->rule->text->symbol_ptr ==     /* Left sides */
               cur_nonterms->x.nt->symbol_ptr)
                 {
                 if (cur_nonterms->left)
                         cur_rules->left = cur_nonterms->left;
                     else
                         cur_rules->left = cur_nonterms;
                 cur_nonterms->left = cur_rules;
                 }
           if (cur_rules->x.cr->nt->symbol_ptr ==            /* Right sides */
               cur_nonterms->x.nt->symbol_ptr)
                 {
                 if (cur_nonterms->right)
                         cur_rules->right = cur_nonterms->right;
                     else
                         cur_rules->right = cur_nonterms;
                 cur_nonterms->right = cur_rules;
                 }
           cur_rules = cur_rules->next;
           }
        cur_nonterms = cur_nonterms->next;
        }
   
   }







private void delete_eq_structure(rules,nonterms)
   EQ_NODE_PTR *rules,
               *nonterms;
/*
 * On Entry:
 *      *rules points at the top of the list of eq_nodes.
 *      *nonterms  points at the top of the list of eq_nodes.
 *
 * On Exit:
 *      *rules is NULL.
 *      *nonterms is NULL.
 *      All storage used by the eq_structure has been deallocated.
*/
   {
   delete_eq_list(rules);
   delete_eq_list(nonterms);
   }







private void mark_eq_class(node,class_num)
   EQ_NODE_PTR node;
   int class_num;
/*
 * On entry:
 *      node points at an eq_node in the desired equivalence class.
 *      class_num contains the number of the class to be marked.
 *      The class fields of all non-terminals in the equivalence
 *         class have been set to NULL.
 *      
 * On exit:
 *      The class fields of all non-terminals in the equivalence
 *         class pointed at by node have been set to class_num.
*/
   {
   if (node->type == NONTERM)
        if (node->class)
                return;
           else
                node->class = class_num;
   mark_eq_class(node->left,class_num);
   mark_eq_class(node->right,class_num);
   }







private int mark_eq_classes(nonterms)
   EQ_NODE_PTR nonterms;
/*
 * On entry:
 *      nonterms points at equivalence list of nonterminals in a
 *         valid equivalence structure.
 *      The class fields of all non-terminals have been set to NULL.
 *      
 * On exit:
 *      The class fields of all non-terminals have been set to 
 *         the number of the equivalence class they belong to.
 *      mark_classes returns the number of equivalence classes encountered.
*/
   {
   int class_num = 0;
   EQ_NODE_PTR cur_nt = nonterms;

   while(cur_nt)
        {
        if (!cur_nt->class)
                {
                class_num++;
                mark_eq_class(cur_nt,class_num);
                }
        cur_nt = cur_nt->next;
        }
   return(class_num);
   }







private int output_eq_class(win,nonterms,class_num)
   WINDOW *win;
   EQ_NODE_PTR nonterms;
   int class_num;
/*
 * On entry:
 *      win points at a valid curses window
 *      nonterms points at equivalence list of nonterminals in a
 *         valid equivalence structure.
 *      class_num contains the number of the equivalence class to
 *         be output.
 *      
 * On exit:
 *      All nonterminals in the equivalence class have been output on
 *         win.
 *      output_eq_class returns the number of nonterminals displayed.
*/
   {
   int line;
   int col;
   int num = 0;                /* Number of non-terminal within given class */
   int num_len;                    /* Length of ASCII representation of num */

   getyx(win,line,col);
   while(nonterms)
        {
        if (nonterms->class == class_num)
           {
           num++;
           num_len = ((int) log10((double) num)) +3; /*undercounts & 2 for()*/
           if ((col + nonterms->x.nt->symbol_ptr->l +num_len + 2) > COLS)
                {
                line++;
                col = 3;
                (void) wmove(win,line,col);
                }
           (void) wprintw(win,"(%d)",num);
           col += c_prtsym(win,nonterms->x.nt->symbol_ptr) + 
                       num_len +5;
           (void) wmove(win,line,col);                /* Space out past end */
           }
        nonterms = nonterms->next;
        }
   return(num);
   }







private EQ_NODE_PTR get_eq_class_nth_node(class_num,nth_num,nonterms)
   int class_num;
   int nth_num;
   EQ_NODE_PTR nonterms;
/*
 * On entry:
 *      class_num contains the number of the equivalence class.
 *      nth_num contains the number of the nonterminal in the
 *         equivalence group desired.
 *      nonterms points at equivalence list of nonterminals in a
 *         valid equivalence structure.
 *      The class fields of all nonterm nodes in the eq. structure have
 *         been set by a prior call to mark_eq_classes.
 *      
 * On exit:
 *      If the current node is the desired one, get_eq_class_nth_node
 *         returns TRUE, otherwise FALSE.
 *      if the desired node is found get_eq_class_nth_node returns a
 *         pointer to it, otherwise NULL.
 *
*/
   {
   int num_seen = 0;

   while(nonterms)
        {
        if (nonterms->class == class_num)
                {
                num_seen++;
                if (num_seen == nth_num) return(nonterms);
                }
        nonterms = nonterms->next;
        }

   return((EQ_NODE_PTR)0);                                 /* Not found */
   }







private void verify_chain_rules(win,spoiler_ptr)
   WINDOW *win;
   EQ_NODE_PTR spoiler_ptr;
/*
 * On Entry:
 *      win points at a valid curses window.
 *      spoiler_ptr points at a non-terminal node in the equivalence
 *         structure that the user has specified as being an equivalence
 *         class 'spoiler'
 *
 * On Exit:
 *      The user has been asked to verify every chain rule that contains
 *         the non-terminal pointed at by spoiler_ptr. Any rules the user
 *         indicates shouldn't be treated as chain rules have been removed
 *         from chain. A new equivalence structure should be built and
 *         submitted to the user for approval.
*/
   {
   EQ_NODE_PTR cur = spoiler_ptr;	       /* Current pnt in eq. struct */
   int pass;
                
   int ini_line;
   int ini_col;
   int cur_line;
   int cur_col;


#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif

   getyx(win,ini_line,ini_col);                       /* Get initial values */
   for (pass=0; pass < 2; pass++)
      {
      if (pass)
            cur = spoiler_ptr->right;
         else
            cur = spoiler_ptr->left;
      while (cur != spoiler_ptr)
           {
           (void) wmove(win,ini_line,ini_col); /* Go to start pnt on screen */
           (void) wclrtobot(win);
           (void) wmove(win,ini_line,ini_col); /* Go to start pnt on screen */
           if (cur->type == RULE)
                 {
                 (void) wmove(win,ini_line, START_COL + 1);
                 c_output_left_node(win,cur->x.cr->rule);
                 (void) c_output_right_side(win,cur->x.cr->rule->right_side,
                                          FALSE,((RIGHT_NODE_PTR)0));
                 getyx(win,cur_line,cur_col);
                 mvwaddstr(win,cur_line + 3,START_COL,
              "Should the above rule be considered a chain rule? (y or n)> ");
                 box(win,'|','-');
                 (void) wrefresh(win);
                 if (!get_boolean_response(win,TRUE))
		    {
		    cur->x.cr->rule->chain_rule = FALSE; /* Perm. disable */
                    cur->x.cr->valid = FALSE;         /* Disable chain rule */
		    }
                 }
              else
		 cagt_msg(0,MSG_EXIT,(cagt_msg_text, msg_arr[-(CAGT_UNXRULNOD)],"verify_chain_rules",1))
           if (pass)
                 cur = cur->right;
              else
                 cur = cur->left;
           }
        }
   (void) wmove(win,ini_line,ini_col);
   (void) wclrtobot(win);
   (void) wmove(win,ini_line,ini_col);
   box(win,'|','-');
   (void) wrefresh(win);
   }







private int verify_eq_structure(win,class_num,nonterms)
   WINDOW *win;
   int class_num;
   EQ_NODE_PTR nonterms;
/*
 * On entry:
 *      win is a valid curses window.
 *      class_num contains the number of classes in the equivalence
 *         structure
 *      nonterms points at a linked list of non terminal nodes in the
 *         equivalence structure.
 *      The class fields of all nonterm nodes in the eq. structure have
 *         been set by a prior call to mark_eq_classes.
 *      
 * On exit:
 *      If verify_eq_structure returns TRUE
 *              All equivalence classes in the equivalence structure have been
 *                 approved by the user.
 *      Otherwise,
 *              The user has caused some rule(s) in the chain rule list to
 *                 be removed. A new equivalence structure must by
 *                 built and then re-verified via this routine.
*/
   {
   int cur_class;                         /* Class currently being verified */
   int nt_num;                    /* # of nonterminals in equivalence class */
   int spoiler_nt;               /* Number of problem non-term in eq. class */
   EQ_NODE_PTR spoiler_ptr;
   int ini_line;
   int ini_col;
   int cur_line;
   int cur_col;


   getyx(win,ini_line,ini_col);                       /* Get initial values */
   mvwaddstr(win,ini_line,START_COL,
                      "The following nonterminals are currently equivalent:");
   for (cur_class = 1; cur_class <= class_num; cur_class++)
        {
        (void) wmove(win,ini_line + 2, 3);
        (void) wclrtobot(win);
        (void) wmove(win,ini_line + 2, 3);
        nt_num = output_eq_class(win,nonterms,cur_class);
        getyx(win,cur_line,cur_col);
        mvwaddstr(win,cur_line + 2, START_COL,
                     "Is the above equivalence class acceptable? (y or n)> ");
        box(win,'|','-');
        (void) wrefresh(win);
        if (!get_boolean_response(win,TRUE))
                {
                mvwaddstr(win,cur_line + 3,START_COL,
                          "Enter number of a 'spoiler' nonterminal above > ");
                box(win,'|','-');
                (void) wrefresh(win);
                spoiler_nt = get_unsigned_int_response(win,TRUE,FALSE,
						       TRUE,1,nt_num);

                spoiler_ptr = get_eq_class_nth_node(cur_class,spoiler_nt,
                                                    nonterms);
                cur_line = ini_line;
                cur_col = ini_col;
                (void) wmove(win,cur_line,cur_col);
                (void) wclrtobot(win);
                (void) wmove(win,cur_line,cur_col);
                verify_chain_rules(win,spoiler_ptr);
                return(FALSE);                              /* Go try again */
                }
      }
   box(win,'|','-');
   (void) wrefresh(win);
   return(TRUE);
   }







private void rename_eq_classes(win,class_num,nonterms)
   WINDOW *win;
   int class_num;
   EQ_NODE_PTR nonterms;
/*
 * On entry:
 *      win is a valid curses window.
 *      A valid symbol table must be active.
 *      class_num contains the number of classes in the equivalence
 *         structure
 *      nonterms points at a linked list of non terminal nodes in the
 *         equivalence structure.
 *      The class fields of all nonterm nodes in the eq. structure have
 *         been set by a prior call to mark_eq_classes.
 *      All equivalence classes in the equivalence structure have
 *         been confirmed.
 *      
 * On exit:
 *      The user has been shown every equivalence class and has supplied
 *         a name to be used for all elements in each class. The nonterminals
 *         in each class have been renamed to this new name.
*/
   {
   int cur_class;                         /* Class currently being verified */
   EQ_NODE_PTR cur_nt;
   SYMBOL new_sym;
   int cur_line;
   int cur_col;
   int ini_line;
   int ini_col;
   char non_term = TRUE;
              

#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif

   getyx(win,ini_line,ini_col);
   mvwaddstr(win,ini_line,ini_col,
     "The following nonterminals have been confirmed as equivalent:");
   for (cur_class = 1; cur_class <= class_num; cur_class++)
        {
        (void) wmove(win,ini_line + 2,3);
        (void) wclrtobot(win);
        (void) wmove(win,ini_line + 2,3);
        (void) output_eq_class(win,nonterms,cur_class);
        getyx(win,cur_line,cur_col);
        mvwaddstr(win,cur_line + 2, START_COL,
                          "Enter a name to be used to describe the above > ");
        box(win,'|','-');
        (void) wrefresh(win);
        new_sym = get_identifier(win,TRUE,FALSE,((SYMBOL)0),&non_term);
        cur_nt = nonterms;
        while (cur_nt)                   /* Rename identifiers in the class */
           {
           if (cur_nt->class == cur_class)
                rename_identifier(cur_nt->x.nt->symbol_ptr,new_sym);
           cur_nt = cur_nt->next;
           }
        }
   box(win,'|','-');
   (void) wrefresh(win);
   }







private void delete_chain_rules(chain)
   CHAIN_RULE_PTR *chain;
/* On Entry:
 *      *chain points at a valid chain rule list.
 *
 * On Exit:
 *      Any valid chain rules in *chain have been removed from the
 *         the EBNF grammar structure.
 *      The storage used by the chain rule list has been returned
 *         to the operating system.
*/
   {
   CHAIN_RULE_PTR temp;

   while (temp = *chain)
        {
        *chain = (*chain)->next;
        if (temp->valid) NULT_rule(temp->rule);
	FREE_MEMORY(temp, "delete_chain_rules", 1)
        }

   }








public void process_chainrules(grammar)
   LEFT_NODE_PTR grammar;
/*
 * On Entry:
 *      grammar points at a list of left sides of an EBNF grammar structure.
 *	A valid symbol table for grammar must be active.
 *
 *
 * On Exit:
 *      All possible chain rules have been removed under user control.
*/

   {
   WINDOW *chain_win;
   CHAIN_RULE_PTR chain;
   EQ_NODE_PTR rules;
   EQ_NODE_PTR nonterms;
   int status;
   int class_num;                          /* Number of equivalence classes */


   /* Get a chain rule processing window for curses */
   chain_win = newwin(0,0,0,0);
   box(chain_win,'|','-');
   output_centered(chain_win,1,TRUE,"Chain Rule Elimination");
   mvwaddstr(chain_win,2,1,LINE);

  if (!(chain = get_chainrule_list(grammar)))
        message_and_pause(chain_win,10,"No Chain Rules Found.");
     else
        {
        do {
           build_eq_structure(chain,&rules,&nonterms);
           class_num = mark_eq_classes(nonterms);

           (void) wmove(chain_win,4,START_COL);
           status = verify_eq_structure(chain_win,class_num,nonterms);
           if (!status) delete_eq_structure(&rules,&nonterms);

           } while (!status);

         (void) wmove(chain_win,4,1);
         rename_eq_classes(chain_win,class_num,nonterms);

         delete_eq_structure(&rules,&nonterms);
         delete_chain_rules(&chain);
         (void) NULT_redundant_rules(grammar);
	 delete_chainrule_list(&chain);
         }
   
   delwin(chain_win);
   }
