static char RCSid[] = "$Id: gram_tools.c,v 1.14 1992/07/26 23:18:36 waite Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */

/*    This module contains routines that support the construction and
 * manipulation of CAGT grammar structures. It represents the
 * combination of the gram_build and gram_edit modules, and superceeds
 * both of them.
 *
 * Routines:
 *			- C R E A T I O N -
 *	1) create_left_node - returns a pointer to a new left node
 *	2) create_right_node - returns a pointer to a new right node
 *	3) create_infix_node - returns a pointer to a new right node
 *			with an infix nest attached
 *	4) add_rel_chain - adds the initial relationship chains to
 *			an existing EBNF grammar.
 *
 *			- D E L E T I O N -
 *      1) delete_right_side - Returns the storage used by an existing r. s.
 *	2) delete_rule - Returns the storage used by an existing rule.
 *	3) delete_redundant_rules - Returns the storage used by rules
 *			that duplicate an earlier existing rule.
 *
 *			- D U P L I C A T I O N -
 *      1) dup_left_side - Returns a duplicate of an existing left side
 *			including the relationship chain.
 *      2) dup_right_side - Returns a duplicate of an existing right side.
 *      3) dup_rel_chain - Returns a duplicate of an existing relationship
 *			chain.
 *
 *			- R E M O V A L -
 *      1) NULT_rule - "Deletes" rule by replacing it by a NULT left side
 *			and hanging rule from undo pointer.
 *      2) NULT_redundant_rules - Passes through the EBNF grammar structure
 *			and "deletes" extra copies of rules that appear.
 *
 *			- E D I T I N G -
 *	1) remove_alternation - removes alternation from an existing EBNF
 *			rule.
 *      2) trim_nesting - Removes unneeded parenthesis.
 *      3) change_idnt_or_litt - Changes an identifier or literal node
 *			on the right side of a rule into a different
 *			identifier or literal.
 *      4) add_comb_gen_rule - Adds a rule generated by the combination
 *			of two similar rules (Differing only in one IDNT
 *			or LITT). Such rules consist of a left side with
 *			a single IDNT or LITT on the right.
 *      5) combine_redundant_rules - Given two identical rules, this
 *			routine combines them into one rule with their
 *			relationship pointers combined.
 *      6) remove_connections - Removes all connection points from a
 *                      given right side.
 *	7) remove_pgs_nodes - Removes all PGS related nodes
 *			from a grammar structure.
 *
 *			- S E A R C H I N G -
 *      1) count_nonterminals - Count nonterminals in a rule right side.
 *	2) first_ptr_addr - returns the address of the pointer
 *			to the the first occurrance of a node in an existing
 *			right side.
 *	3) find_unnecessary_LPNT_ptr_addr - returns a FIFO queue containing
 *			the addresses of pointers to any unnecessary '('
 *			nodes in an existing right side.
 *	4) find_token_ptr_addr - returns a FIFO queue containing the
 *			addresses of pointers to any nodes of specified
 *			token type in an existing right side.
 *	5) traversal_ptr_addr - Returns a FIFO queue containing the
 *			addresses of pointers to any nodes in a
 *			depth first left-to-right traversal of an
 *			existing right hand side (excluding the
 *			addresses of the pointers to the first
 *			node.
 *
 *			- D E C I S I O N -
 *	1) LPNT_unnecessary - determines whether a given parenthesis is
 *			really necessary.
 *      2) litt_cr_spoiler - decide if a given literal is a chain
 *                           rule "spoiler ".
 *      3) is_chainrule - Decide whether a given rule is a potential
 *                        chain rule.
 *
 *			- C O M P A R I S O N -
 *      1) same_left_side - Compares the left sides of two rules and
 *			returns TRUE if they are the same.
 *      2) same_right_side - Compares the right sides of two rules and
 *			returns TRUE if they are the same.
 *      3) same_rule - Compares two rules and returns TRUE if they are
 *			the same.
 *
 *			- I N D E X E D   A C C E S S -
 *      1) init_indexed_access-Adds constant time access front end to grammar.
 *      2) rule_by_index - Allows access to rules in constant time by index
 *      3) delete_indexed_access - Deletes the constant time front end from
 *			the grammar.
 *
 *
 *
 * General EBNF structure form:
 * ----------------------------
 *
 *
 *
 *                     | <--(rule chain)      :---(Right side of rule)
 *                     |                      V
 *                     V                         
 *	:---:    :-----------:    :------------:
 *	|   |--->| left node |--->| right node |--->...
 *	|---|    |___________|    |____________|
 *	|   |-         |                |
 *	|---| |        |                |
 *	|   | |        V                V
 *	|---| |  :-----------:          .
 *	|   | |->| left node |          .
 *	|---|    |___________|          .
 *	|   |          |
 *	|---|          |
 *	|   |          V
 *	|---|          .
 *	|   |          .
 *	|---|          .
 *	|   |
 *	|---|
 *	|   |
 *	|---|
 *	|   | <---(Indexed rule access)
 *	|---|
 *        .
 *        .
 *        .
 *
 *    As can be seen above, an EBNF grammar structure is a linked list
 * of rule left sides. Each left side is represented by an instantiation
 * of the left_node struct. If needed, indexed access to the rules
 * by rule number can be had by adding an array of pointers to the rules.
 * This array is shown above, but is not strictly necessary. It allows
 * access of rules in constant time. Each left side has an associated
 * right side. The right side is a prefix ordered singly linked tree
 * structure. Once such a structure is built, alternation is removed, and
 * unnecessary parenthesis are eliminated to obtain a 'normalized grammar'.
 *
 * Example:  (xExpr)+ | (xExpr xOp xExpr)*.
 *
 *    Structure built by parser:
 *    --------------------------
 *
 *                :---:   :---:
 *            --->| | |-->| . |
 *                |___|   |___|
 *                 | |
 *          _______| |________________
 *          |                        |
 *          V                        V
 *        :---:                    :---:
 *        | + |                    | * |
 *        |___|                    |___|
 *	    |                        |
 *	    V                        V
 *	  :---:                    :---:
 *        | ( |                    | ( |
 *        |___|                    |___|
 *          |                        |
 *          V                        V
 *	:-------:   :---:        :-------:   :-----:   :-------:   :---:
 *      | xExpr |-->| ) |        | xExpr |-->| xOp |-->| xExpr |-->| ) |
 *      |_______|   |___|        |_______|   |_____|   |_______|   |___|
 *
 *
 *    Removing Alternation:
 *    ---------------------
 *
 *                :---:   :---:
 *       -------->| + |-->| . |
 *                |___|   |___|
 *                  |
 *                  V
 *                :---:
 *                | ( |
 *                |___|
 *                  |
 *                  V
 *              :-------:   :---:
 *              | xExpr |-->| ) |
 *              |_______|   |___|
 *           
 *           
 *                :---:   :---:
 *          ----->| * |-->| . |
 *                |___|   |___|
 *                  |
 *                  V
 *	          :---:
 *                | ( |
 *                |___|
 *                  |
 *                  V
 *	        :-------:   :-----:   :-------:   :---:
 *              | xExpr |-->| xOp |-->| xExpr |-->| ) |
 *              |_______|   |_____|   |_______|   |___|
 *
 *
 *    Removing unneeded parenthesis:
 *    ------------------------------
 *
 *                :---:   :---:
 *       -------->| + |-->| . |
 *                |___|   |___|
 *                  |
 *                  V
 *              :-------:
 *              | xExpr |
 *              |_______|
 *           
 *           
 *                :---:   :---:
 *          ----->| * |-->| . |
 *                |___|   |___|
 *                  |
 *                  V
 *	          :---:
 *                | ( |
 *                |___|
 *                  |
 *                  V
 *	        :-------:   :-----:   :-------:   :---:
 *              | xExpr |-->| xOp |-->| xExpr |-->| ) |
 *              |_______|   |_____|   |_______|   |___|
 *
*/



#include <stdio.h>
#include <stdlib.h>
#include "cagt_config.h"
#include "cagt_usr_err.h"
#include "support.h"
#include "gram.h"
#include "queue.h"







/* --------------- C R E A T I O N -----------------------------------------*/

public LEFT_NODE_PTR create_left_node(code,sym)
   int code;
   SYMBOL sym;
/*
 * On Entry:
 *	code contains a valid lexical token code.
 *	sym contains a valid SYMBOL.
 * On Exit:
 * 	create_left_node returns a pointer to a new left_node.
 *	   The fields of this node are initialized as follows:
 *
 *		code is given the value of the code parameter
 *		text is given a pointer to a symbol_pointer_node that
 *		         contains the value of the text parameter
 *		chain_rule is FALSE
 *		rel_ptr is NULL
 *		right_side is NULL
 *		next_rule is NULL
 *		undo is NULL
 *		redo is NULL
*/
   {
   LEFT_NODE_PTR tmp;

   GET_MEMORY(tmp,LEFT_NODE_PTR, 1, LEFT_NODE,"create_left_node",1)
   tmp->code = code;
   tmp->text = first_symbol_ptr(sym);
   tmp->chain_rule = FALSE;
   tmp->rel_ptr = (REL_CHAIN_ELT_PTR) 0;
   tmp->right_side = (RIGHT_NODE_PTR) 0;
   tmp->next_rule = tmp->undo = tmp->redo = (LEFT_NODE_PTR) 0;

   return(tmp);
   }







public RIGHT_NODE_PTR create_right_node(code)
   int code;
/*
 * On Entry:
 *	code contains a valid lexical token code
 * On Exit:
 * 	create_right_node returns a pointer to a new right_node.
 *	   The fields of this node are initialized as follows:
 *
 *		code is given the value of the code parameter
 *		next is NULL
 *		undo is NULL
 *		redo is NULL
 *		x is NULL
*/
   {
   RIGHT_NODE_PTR tmp;

   GET_MEMORY(tmp, RIGHT_NODE_PTR, 1, RIGHT_NODE, "create_right_node", 1)
   tmp->code   = code;
   tmp->next = tmp->undo = tmp->redo = (RIGHT_NODE_PTR) 0;
   /*
    * The x (extend) union is set to NULL by assigning NULL to one of
    * it's members. This is based on the assumption that the compiler
    * will make all three fields the same length and overlay all three
    * fields.
   */
   tmp->x.nest  = (RIGHT_NODE_PTR) 0;

   return(tmp);
   }







public RIGHT_NODE_PTR create_infix_node(code,left,right)
   int code;
   RIGHT_NODE_PTR left,
		  right;
/*
 * create_infix node creates a right_node with an infix_nest and attaches
 * the left and right branches to it.
 *
 * On Entry:
 *	code contains the code to use for the new node (Should
 *	   be BART or SEPT).
 *	left points at the left sub-rule to be attached.
 *	right points at the right sub-rule to be attached.
 * On Exit:
 *	create_infix_node returns a pointer to the new node.
*/
   {
   RIGHT_NODE_PTR result;

   result = create_right_node(code);
   GET_MEMORY(result->x.infix, INFIX_NEST_PTR, 1, INFIX_NEST,
	      "create_infix_node", 1)
   result->x.infix->left = left;
   result->x.infix->right = right;

   return result;
   }







public void add_rel_chain(grammar)
   LEFT_NODE_PTR grammar;
/*
 *    Each rule in an abstract grammar represents one or more rules
 * in the corresponding concrete grammar. It is desirable to be
 * able to keep track of which abstract rules represent which
 * concrete rule(s). This is done by attaching relationship
 * chains to the left side of each rule in the grammar before
 * performing any concrete to abstract transformations. When two
 * rules are merged during processing, their relationship chains
 * are merged. Thus, a given rule represents the concrete rules
 * whose rule numbers are held in it's rel_chain.
 *
 * On Entry:
 *	grammar points at an existing EBNF grammar structure.
 *	The grammar is unprocessed (still equivalent to concrete)
 *	The grammar rules do not have rel_chains attached.
 *
 * On Exit:
 *	Each rule has a one-element rel_chain that contains it's
 *	   own rule number (Every rule in a concrete grammar
 *	   represents itself).
*/
   {
   int rule = 0;

   while (grammar)
	{
	GET_MEMORY(grammar->rel_ptr, REL_CHAIN_ELT_PTR, 1, REL_CHAIN_ELT,
		   "add_rel_chain", 1)
	grammar->rel_ptr->rule_num = ++rule;
	grammar->rel_ptr->next = (REL_CHAIN_ELT_PTR) 0;
	grammar = grammar->next_rule;
	}

   }







/* --------------- D E L E T I O N -----------------------------------------*/

public void delete_right_side(node)
   RIGHT_NODE_PTR node;
/*
 * On Entry:
 *	node contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 * On Exit:
 *	The right side of the rule, starting from the node pointed
 *	   at by node has been deleted.
*/
   {
   if (node == (RIGHT_NODE_PTR) 0) return;

   switch (node->code) {
	case NULT :
	case  IST :
	case GTRT : 
	case LSST :
	case LITT :
	case IDNT:
		{
		delete_right_side(node->next);
		FREE_MEMORY(node, "delete_right_side", 1)
		break;
		}
	case BART : 
	case SEPT : 
		{
		delete_right_side(node->x.infix->left);
		delete_right_side(node->x.infix->right);
		delete_right_side(node->next);
		FREE_MEMORY(node->x.infix, "delete_right_side", 2)
		FREE_MEMORY(node, "delete_right_side", 3)
		break;
		}
	case AMPT : 
	case  ATT :
	case DOLT :
	case ASTT : 
	case PLST : 
	case LPNT :
	case LBKT :
		{
		delete_right_side(node->x.nest);
		delete_right_side(node->next);
		FREE_MEMORY(node, "delete_right_side", 4)
		break;
		}
	case DOTT :
	case RPNT:
	case RBKT:
		{
		FREE_MEMORY(node, "delete_right_side", 5)
		break;
		}
	}

   }







public void delete_rule(rule)
   LEFT_NODE_PTR rule;
/*
 * On Entry:
 *	rule points at an existing rule left hand side.
 *
 * On Exit:
 *	The storage used by rule has been returned to the operating
 *	   system
*/
   {
   REL_CHAIN_ELT_PTR temp;

   delete_right_side(rule->right_side);			    /* Right side */
   
   while (rule->rel_ptr)			    /* Relationship chain */
	{
	temp = rule->rel_ptr;
	rule->rel_ptr = rule->rel_ptr->next;
	FREE_MEMORY(temp, "delete_rule", 1)
	}

   FREE_MEMORY(rule, "delete_rule", 2)			     /* Left side */
   }







public int delete_redundant_rules(grammar)
   LEFT_NODE_PTR grammar;
/*
 * On Entry:
 *	grammar points at the linked list of EBNF rule left hand sides.
 *
 * On Exit:
 *	Only one copy of any rule exists in the grammar structure. Duplicate
 *	   rules have been completely eliminated. Any storage used by
 *	   such rules has been returned to the operating system. The
 *	   relationship chains of such rules have been deleted as well.
 *	delete_redundant_rules returns the number of redundant rules
 *	   encountered.
*/
   {
   int num = 0;
   LEFT_NODE_PTR inner;
   LEFT_NODE_PTR dead_rule;

   while (grammar)
   	{
        if (grammar->code != NULT)		/* NULT rules can't combine */
	      {
              inner = grammar;
	      while (inner->next_rule)
	            {
	    	    while (inner->next_rule->code == NULT)
			inner = inner->next_rule;
		    if (inner->next_rule)
	       	          if (same_rule(grammar,inner->next_rule))
				{
				dead_rule = inner->next_rule;
				inner->next_rule=inner->next_rule->next_rule;
				delete_rule(dead_rule);
				num++;
				}
			     else
	            		inner = inner->next_rule;
	            }
	      }
       grammar = grammar->next_rule;
       }

   return(num);
   }








/* --------------- D U P L I C A T I O N -----------------------------------*/

public LEFT_NODE_PTR dup_left_side(old_rule)
   LEFT_NODE_PTR old_rule;
/*
 * On Entry:
 *	old_rule contains a pointer to the left side of an existing rule
 *	   in the internal EBNF grammar structure.
 * On Exit:
 *	dup_left_side returns a pointer to a new left_node with
 *	the same code and text and chain_rule values as old_rule. The rel_ptr
 *	field points to a copy of the linked list pointed at by
 *	old_rule->rel_ptr. All other fields are set to NULL.
 *	   
*/
   {
   LEFT_NODE_PTR left;				     	   /* New left side */

   left = create_left_node(old_rule->code,old_rule->text->symbol_ptr);
   left->chain_rule = old_rule->chain_rule;
   left->rel_ptr = dup_rel_chain(old_rule->rel_ptr);

   return(left);
   }







public RIGHT_NODE_PTR dup_right_side(old_rule)
   RIGHT_NODE_PTR old_rule;
/*
 * On Entry:
 *	old_rule contains a pointer to the right side of an existing rule
 *	   in the internal EBNF grammar structure.
 * On Exit:
 *	dup_right_side returns a pointer to a copy of the old rule
 *	   starting from the node pointed at by old_rule.
*/
   {
   RIGHT_NODE_PTR node;

   if (!old_rule) return((RIGHT_NODE_PTR) 0);

   /* Get new node */
   node = create_right_node(old_rule->code);

   switch (node->code) {
	case NULT :
	case  IST :
	case SLHT :
	case GTRT :
	case LSST :
		{
		node->next = dup_right_side(old_rule->next);
		break;
		};
	case AMPT :
	case  ATT :
	case DOLT :
	case ASTT :
	case PLST :
	case LPNT :
	case LBKT :
		{
		node->x.nest = dup_right_side(old_rule->x.nest);
		node->next = dup_right_side(old_rule->next);
		break;
		}
	case BART :
	case SEPT :
		{
		GET_MEMORY(node->x.infix, INFIX_NEST_PTR, 1, INFIX_NEST,
			   "dup_right_side", 1)
		node->x.infix->left = dup_right_side(old_rule->x.infix->left);
		node->x.infix->right =dup_right_side(old_rule->x.infix->right);
		node->next = dup_right_side(old_rule->next);
		break;
		}
	case IDNT:
	case LITT:
		{
		node->x.text = old_rule->x.text;
		node->next = dup_right_side(old_rule->next);
		break;
		}
	}

   return(node);
   }







public REL_CHAIN_ELT_PTR dup_rel_chain(old_chain)
   REL_CHAIN_ELT_PTR old_chain;
/*
 * On Entry:
 *	old_chain points at a valid rel_chain_elt.
 *
 * On Exit:
 *	dup_rel_chain returns a pointer to a duplicate of the relationship
 *	   chain starting at old_chain.
*/
   {
   REL_CHAIN_ELT_PTR top = (REL_CHAIN_ELT_PTR) 0,
		     cur = (REL_CHAIN_ELT_PTR) 0;

   if (old_chain == (REL_CHAIN_ELT_PTR) 0) return (REL_CHAIN_ELT_PTR) 0;

   while (old_chain)
	{
	if (cur)
	      {
	      GET_MEMORY(cur->next, REL_CHAIN_ELT_PTR, 1, REL_CHAIN_ELT,
			 "dup_rel_chain", 1)
	      cur = cur->next;
	      }
	   else
	      GET_MEMORY(top = cur, REL_CHAIN_ELT_PTR, 1, REL_CHAIN_ELT,
			 "dup_rel_chain", 2)

	cur->rule_num = old_chain->rule_num;
	cur->next = (REL_CHAIN_ELT_PTR) 0;

	old_chain = old_chain->next;
        }

   return(top);
   }






/* --------------- R E M O V A L -------------------------------------------*/

public void NULT_rule(left)
   LEFT_NODE_PTR left;
/*
 * On Entry:
 *	left points at the rule that is to be 'deleted'.
 *
 * On Exit:
 *	The rule has been replaced by a NULT left_node, and the old rule
 *	   has been nested on the undo field of the new node. 
*/
   /* 
    * Strategy:
    * Since the left sides are linked downwards, the rule above this
    * one isn't accessible (to change it's next pointer). Therefore,
    * we duplicate the existing one, put the duplicate on the undo
    * node, and change the original to a NULT.
    *
    * NOTE: UNDO_POINT - The actions taken by this routine must be
    *	    remembered by any future UNDO system that might be added
    *	    to CAGT. In particular, a pointer to the new NULT left side
    *	     must be kept, and the pointer changes must be remembered.
   */

   {
   LEFT_NODE_PTR undo= left->undo;		 /* Save current undo chain */

   /* Get new left side and hang on undo pointer of current node */
   left->undo = create_left_node(left->code,left->text->symbol_ptr);

   /* Transfer the state of current node to copy on undo pointer */
   left->undo->right_side = left->right_side;
   left->undo->next_rule = (LEFT_NODE_PTR) 0;
   left->undo->undo = undo;
   if (undo) undo->redo = left->undo;
   left->undo->redo = left;
   left->undo->rel_ptr = left->rel_ptr;

   /* Turn the original left side into a NULT */
   left->code = NULT;				       /* Token code = NULT */
   left->text = (SYMBOL_PTR_NODE_PTR) 0;		   /* No identifier */
   left->right_side = (RIGHT_NODE_PTR) 0;		   /* No right side */

   }







public int NULT_redundant_rules(grammar)
   LEFT_NODE_PTR grammar;
/*
 * On Entry:
 *	grammar points at the linked list of EBNF rule left hand sides.
 *
 * On Exit:
 *	Only one copy of any rule exists in the grammar structure. Duplicate
 *	   rules have had their relationship pointers added to the surviving
 *	   copy (i.e. The surviving rule stands for all the duplicates).
 *	In addition to removing extra copies of rules, rules of the form
 *		A ::= A.
 *	   are considered redundant and are removed.
 *	NULT_redundant_rules returns the number of redundant rules
 *	   encountered.
*/
   {
   int num = 0;
   LEFT_NODE_PTR inner;
   char trivial;

   while (grammar)
   	{
	trivial = FALSE;			  /* Rule is not "A ::= A." */
        if (grammar->code != NULT)		/* NULT rules can't combine */
	      {
	      /* Is current rule "trivial" (i.e. A ::= A.)? */
	      if (grammar->right_side->next->code == IDNT)
		 if (grammar->right_side->next->next->code == DOTT)
		    if (grammar->right_side->next->x.text->symbol_ptr ==
		        grammar->text->symbol_ptr)
				{
				trivial = TRUE;
				NULT_rule(grammar);
				num++;
				}
	      if (!trivial)
		 {
                 inner = grammar->next_rule;
	         while (inner)
	            {
	    	    if (inner->code != NULT)
	       	          if (same_rule(grammar,inner))
				{
		                combine_redundant_rules(grammar,inner);
				num++;
				}
	            inner = inner->next_rule;
	            }
	         }
	      }
       grammar = grammar->next_rule;
       }

   return(num);
   }







/* --------------- E D I T I N G -------------------------------------------*/

public void remove_alternation(rule)
   LEFT_NODE_PTR rule;
/*
 * remove_alternation splits rules containing alternation into separate rules
 * with the alternation removed. All alternation in the original rule
 * is removed, but subsequent calls to remove_alternation on the generated
 * rules are needed to completely remove all alternation.
 *
 * Example:
 *
 *                                 :---:
 *     	                  before-->| | |-->after
 *                                 |___|
 *                                  | |
 *	               _____________| |_________
 *                     |                       |
 * 	               V                       V
 *                 :--------:                :---:
 *                 | node 1 |                | ( |
 *                 |________|                |___|
 *                                             |
 *                                             V
 *                        	             :---:   :---:
 *                                           | | |-->| ) |
 *                                           |___|   |___|
 *                                            | |
 *                                      ______| |______
 *                                      |             |
 *                                      V             V
 *                                  :-------:     :-------:
 *                                  | node2 |     | node3 |
 *                                  |_______|     |_______|
 *
 * Is replaced by 2 new rules:
 *
 *                                 :-------:
 *     	                  before-->| node1 |-->after
 *                                 |_______|
 *
 *
 *                                 :---:
 *     	                  before-->| ( |-->after
 *                                 |___|
 *                                   |
 *                                   V
 *            	                   :---:   :---:
 *                                 | | |-->| ) |
 *                                 |___|   |___|
 *                                  | |
 *                            ______| |______
 *                            |             |
 *                            V             V
 *                        :-------:     :-------:
 *                        | node2 |     | node3 |
 *                        |_______|     |_______|
 *
 * The second rule is inserted into the chain of rule left hand
 * sides directly below the original rule. In the above, the second
 * rule still has an instance of alternation remaining. This is
 * because only the first rule is processed recursivly until all
 * alternation is removed. Rules generated in this process must
 * have remaining alternation removed in a subsequent call. This
 * routine is intended to be called in a mannar similar to the
 * following:
 *
 *		start = pointer to first rule in chain of Left Hand Sides
 *		while (start)
 *			{
 *			remove_recursion(start);
 *			start = start->next_rule;
 *			}
 *
 * On Entry:
 *   rule points at the left side of an EBNF rule.
 *
 * On Exit:
 *	remove_alternation removes alternation from rule by generating
 *	   new rules which describe the other alternatives.
 *	   These new rules are inserted directly beneath rule.
*/
   {
   RIGHT_NODE_PTR *alt_ptr_addr;	      /* Address of pointer to BART */
   RIGHT_NODE_PTR alt_node;		 /* Pointer to BART (*alt_ptr_addr) */
   LEFT_NODE_PTR dup;				/* Points at duplicate rule */
   LEFT_NODE_PTR next_rule;			       /* Used to add rules */
   RIGHT_NODE_PTR temp;			/* Used to find tail of alternative */


   alt_ptr_addr = first_ptr_addr(BART,rule->right_side);       /* Find BART */
   if (!alt_ptr_addr)
	  return;				 /* rule had no alternation */
      else
	  {
	  /* Duplicate the rule */
	  next_rule = rule->next_rule;
	  dup = rule->next_rule = dup_left_side(rule);
	  dup->right_side = dup_right_side(rule->right_side);
	  dup->next_rule = next_rule;

          /* Keep the left alterative in the original rule */
	  alt_node = *alt_ptr_addr;
	  temp = alt_node->x.infix->left;
	  if (temp)				  /* Left side is not empty */
		{
	        while (temp->next) temp = temp->next;
	        temp->next = alt_node->next;
		*alt_ptr_addr = alt_node->x.infix->left;
		}
	     else				      /* Left side is empty */
		if (alt_node->next)
		      *alt_ptr_addr = alt_node->next;
		   else
		      *alt_ptr_addr = (RIGHT_NODE_PTR) 0;
		   
	  alt_node->next = (RIGHT_NODE_PTR) 0;    /* Dump the right side */
	  alt_node->x.infix->left = (RIGHT_NODE_PTR) 0;
	  delete_right_side(alt_node);
	  
          /* Keep the right alterative in the duplicate rule */
	  alt_ptr_addr = first_ptr_addr(BART,dup->right_side);
	  alt_node = *alt_ptr_addr;
	  temp = alt_node->x.infix->right;
	  if (temp)				 /* Right side is not empty */
		{
	        while (temp->next) temp = temp->next;
	        temp->next = alt_node->next;
		*alt_ptr_addr = alt_node->x.infix->right;
		}
	     else				     /* Right side is empty */
		if (alt_node->next)
		      *alt_ptr_addr = alt_node->next;
		   else
		      *alt_ptr_addr = (RIGHT_NODE_PTR) 0;

	  alt_node->next = (RIGHT_NODE_PTR) 0;     /* Dump the left side */
	  alt_node->x.infix->right = (RIGHT_NODE_PTR) 0;
	  delete_right_side(alt_node);


	  /* Remove any remaining alternation in original rule */
	  remove_alternation(rule);
          }

   }







public void trim_nesting(node)
   RIGHT_NODE_PTR node;
/*
 * nesting sweeps through the right side of an EBNF rule.
 * Any occurrence of an unnecessary parenthesis nesting
 * is removed from the subtree beginning at the node pointed at by node.
 *
 * On Entry:
 *	node contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 * On Exit:
 *	All unnecessary nesting in the grammar has been removed.
*/
   {
   QUEUE_PTR queue;
   RIGHT_NODE_PTR *cur;			      /* Address of pointer to LPNT */
   RIGHT_NODE_PTR temp;		      /* Used to delete LPNT and RPNT nodes */

   queue = init_queue();

   /* Get the locations of unneeded LPNT nodes */
   find_unnecessary_LPNT_ptr_addr(queue,node);

   /* Process them in the reverse order they were found */
   reverse_queue(queue);
   while (!(empty(queue)))
      {
      cur = (RIGHT_NODE_PTR *) dequeue(queue);
      if ((*cur)->x.nest->code == RPNT)			    /* Empty Parens */
            {
	    temp = *cur;
	    *cur = (*cur)->next;
	    temp->next = (RIGHT_NODE_PTR) 0;
	    delete_right_side(temp);
	    }
	 else
	    {
	    temp = (*cur)->x.nest;
	    while(temp->next->code != RPNT) temp = temp->next;
	    delete_right_side(temp->next);		 /* Delete the RPNT */
	    temp->next = (*cur)->next;
	    temp = *cur;
	    *cur = (*cur)->x.nest;
	    temp->x.nest = (RIGHT_NODE_PTR) 0;
	    temp->next = (RIGHT_NODE_PTR) 0;
	    delete_right_side(temp);			 /* Delete the LPNT */
	    }
      }
   delete_queue(queue);
   }







public void change_idnt_or_litt(node,new_nt,new_type)
   RIGHT_NODE_PTR node;
   SYMBOL new_nt;
   int new_type;
/*
 * On Entry:
 *	node points at the right_node that is to be 'changed'.
 *	new_nt is a new SYMBOL.
 *	new_type is the type of new_nt (IDNT or LITT).
 *
 * On Exit:
 *	A new node has replaced the old node with the new identifier
 *	   or literal, and the old node has been nested on the undo field
 *	   of the new node. 
*/
   /*
    * NOTE: UNDO_POINT - The actions taken by this routine must be
    *	    remembered by any future UNDO system that might be added
    *	    to CAGT. In particular, a pointer to the new right node
    *	     must be kept, and the pointer changes must be remembered.
   */

   {
   RIGHT_NODE_PTR undo = node->undo;		 /* Save current undo chain */


   /* Get new node and hang on undo pointer of current node */
   node->undo = create_right_node(node->code);

   /* Transfer the state of current node to copy on undo pointer */
   node->undo->next = node->next;
   node->undo->undo = undo;
   if (undo) undo->redo = node->undo;
   node->undo->redo = node;
   node->undo->x.text = node->x.text;

   /* Add the new identifier */
   node->code = new_type;
   node->x.text = first_symbol_ptr(new_nt);

   }







public void add_comb_gen_rule(prior_rule,left_idnt,orig_right_node)
   LEFT_NODE_PTR prior_rule;
   SYMBOL left_idnt;
   RIGHT_NODE_PTR orig_right_node;
/*
 * On Entry:
 *	prior_rule points to the rule in the grammar directly above where
 *	   the new rule should be placed in the EBNF grammar structure.
 *	left_idnt contains a SYMBOL to be used on the left side of the
 *	   new rule.
 *	orig_right_node points at an existing idnt or litt node that
 *	   should be duplicated as the right side of the new rule.
 *
 * On Exit:
 *	A new rule has been added to the grammar. This rule has the form:
 *
 *      :------:   :-----:   :-------:   :---:
 *      | left |-->| ::= |-->| right |-->| . |
 *      |______|   |_____|   |_______|   |___|
*/
   {
   LEFT_NODE_PTR rest = prior_rule->next_rule;
   LEFT_NODE_PTR new_rule;
   RIGHT_NODE_PTR cur;
   SYMBOL_PTR_NODE_PTR dummy;		      /* used to call is_chainrule */

   /* Install new left side */
   new_rule = prior_rule->next_rule = create_left_node(IDNT,left_idnt);
   new_rule->next_rule = rest;

   /* Add IST node */
   cur = new_rule->right_side = create_right_node(IST);

   /* Identifier or Literal */
   cur->next = create_right_node(orig_right_node->code);
   cur = cur->next;
   cur->x.text = orig_right_node->x.text;

  /* Last node */
   cur->next = create_right_node(DOTT);

   /* Set chain rule status */
   new_rule->chain_rule = is_chainrule(new_rule->right_side, FALSE, &dummy,
				       TRUE);
   }
     






public void combine_redundant_rules(original,copy)
   LEFT_NODE_PTR original,
		 copy;
/*
 * On Entry:
 *	original contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	copy contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	Neither left side has it's code field set to NULT.
 *	The two rules have been verified as being identical.
 * On Exit:
 *	The rule pointed at by copy has been "deleted" It's relationship
 *	   pointers have been added to those for original.
*/
   {
   REL_CHAIN_ELT_PTR temp = original->rel_ptr;

   /*
    * NOTE: UNDO_POINT - The actions taken by this routine must be
    *	    remembered by any future UNDO system that might be added
    *	    to CAGT. In particular, The relationship chain of the
    *	    original rule is extended, and the duplicate rule
    *	    is 'deleted'.
   */
   
   if (temp)
	 {
	 while (temp->next) temp = temp->next;	       /* Find end of chain */
         temp->next = dup_rel_chain(copy->rel_ptr);	/* Add new pointers */
	 }
      else						/* Shouldn't happen */
	 original->rel_ptr = dup_rel_chain(copy->rel_ptr);

   NULT_rule(copy);
   }







public void remove_connections(node)
   RIGHT_NODE_PTR node;
/*
 * Intended Application:
 *    The decorated abstract grammar that comes back from attribute
 * grammar generation (possibly via GAG) has the same rules as were
 * generated by cagt in the forward phase. However, they may differ
 * in 3 ways: (1) The decorated grammar has connection points, (2)
 * It may have extra parenthesis forced by the addition of these
 * connection points, and (3) The user may have reordered the rules.
 * Removing connections in a copy of the decorated rules means that the
 * copy can be compared to the original abstract grammar.
 *
 * On Entry:
 *      node contains a pointer to a node on the right side of a
 *         rule in the EBNF internal structure.
 *
 * On Exit:
 *      All connection points in the grammar have been removed.
*/
   {
   QUEUE_PTR queue;
   RIGHT_NODE_PTR *cur;
   RIGHT_NODE_PTR temp;
   static short toklist[NUM_TOKENS];	/* Is initialized to zero
					   by compiler */

   queue = init_queue();

   /* Get the locations of AMPT nodes */
   toklist[AMPT] = TRUE;
   find_token_ptr_addr(toklist,queue,node);

   /* Remove all connections in the reverse order found */
   reverse_queue(queue);
   while (!(empty(queue)))
        {
        cur = (RIGHT_NODE_PTR *) dequeue(queue);
	temp = *cur;
	*cur = (*cur)->next;
        temp->next = (RIGHT_NODE_PTR) 0;
        delete_right_side(temp);
        }
   delete_queue(queue);
   }








public void remove_pgs_nodes(grammar)
   LEFT_NODE_PTR grammar;
/*
 * Intended Application:
 *    The concrete grammar supplied by the user may contain PGS
 * tokens ("$" or "@") that are used to give PGS "hints" about how
 * to handle certain ambiguous situations and connection points (&)
 * for building the parse tree.
 * This routine removes such nodes from the grammar structure.
 *
 * On Entry:
 *      grammar contains a pointer to the first rule in an ebnf
 *	   grammar structure.
 *
 * On Exit:
 *      All PGS nodes in the rule have been removed.
*/
   {
   QUEUE_PTR queue;
   RIGHT_NODE_PTR *cur;
   RIGHT_NODE_PTR temp;
   LEFT_NODE_PTR rule = grammar;
   static short toklist[NUM_TOKENS];	/* Is initialized to zero
					   by compiler */

   /* Indicate the PGS modification nodes */
   toklist[ATT] = toklist[DOLT] = toklist[AMPT] = TRUE;

   queue = init_queue();

   while (rule)
      {
      /* Get the locations of PGS modification nodes */
      find_token_ptr_addr(toklist,queue,rule->right_side);

      /* Remove all connections in the reverse order found */
      reverse_queue(queue);
      while (!(empty(queue)))
           {
           cur = (RIGHT_NODE_PTR *) dequeue(queue);
	   temp = *cur;
	   *cur = (*cur)->next;
           temp->next = (RIGHT_NODE_PTR) 0;
           delete_right_side(temp);
           }
      rule = rule->next_rule;
      }
   /* Weed out duplicate rules - BUT DON'T DELETE THEM*/
   (void) NULT_redundant_rules(grammar);
   delete_queue(queue);
   }








/* --------------- S E A R C H I N G ---------------------------------------*/

public int count_nonterminals(node)
   RIGHT_NODE_PTR node;
/*
 * On Entry:
 *      node contains a pointer to a node on the right side of a
 *         rule in the EBNF internal structure.
 * On Exit:
 *      count_nonterminals returns a count of the number of non_terminals
 *         in the subtree pointed at by node.
 *
*/
   {
   int count = 0;                       /* Counts number in current subtree */

   if (!node) return(0);

   switch (node->code) {

        /* Nodes that have 3 continuations */
        case BART :
        case SEPT :
                {
                count += ( count_nonterminals(node->x.infix->left)  +
                           count_nonterminals(node->x.infix->right) +
                           count_nonterminals(node->next) );
                break;
                }

        /* Nodes that have 2 continuations */
        case AMPT :
	case  ATT :
	case DOLT :
        case ASTT :
        case PLST :
        case LPNT :
        case LBKT :
                {
                count += ( count_nonterminals(node->x.nest) +
                           count_nonterminals(node->next) );
                break;
                }

        /* Nodes that have 1 continuation */
        case IDNT : if (node->x.text->symbol_ptr->non_term) count++;

        case NULT :
        case  IST :
        case GTRT :
        case LSST :
        case LITT : count += count_nonterminals(node->next);

      };

   return(count);
   }







public RIGHT_NODE_PTR *first_ptr_addr(code,node)
   int code;
   RIGHT_NODE_PTR node;
/*
 * first_ptr_addr sweeps through the right side of an EBNF rule searching
 * for a node that has a next or nested pointer to a node with the token
 * type specified by code.
 *
 * On Entry:
 *	code contains the token code of the desired node.
 *	node contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 * On Exit:
 *	if a node of type code is found
 *	      The address of the pointer to that node is returned.
 *	   otherwise
 *	      NULL is returned.
*/
   {
   RIGHT_NODE_PTR *temp;

   if (!node) return ((RIGHT_NODE_PTR *) 0);

   switch (node->code) {
	/* Types that have the next continuation only */
	case  IST :
	case GTRT :
	case LSST :
	case AMPT :
	case  ATT :
	case DOLT :
	case LITT :
	case IDNT :
		{
		if (node->next)
			if (node->next->code == code)
				return (&(node->next));
			    else
				return (first_ptr_addr(code,node->next));
		break;
		}

	/* Types that have nest and next continuations */
   	case ASTT :
	case PLST :
	case LPNT :
	case LBKT :
		{
		if (node->x.nest)
		   {
		   if (node->x.nest->code == code) return (&(node->x.nest));
		   if (temp = first_ptr_addr(code,node->x.nest))
			return (temp);
		   }
		if (node->next)
		   {
		   if (node->next->code == code) return (&(node->next));
		   return (first_ptr_addr(code,node->next));
		   }
		break;
		}

	/* Types that have two nested and a next continuation */
	case BART :
	case SEPT :
		{
		if (node->x.infix->left)
		   {
		   if (node->x.infix->left->code == code)
			return(&(node->x.infix->left));
		   if (temp = first_ptr_addr(code,node->x.infix->left))
			return (temp);
		   }
		if (node->x.infix->right)
		   {
		   if (node->x.infix->right->code == code)
			return(&(node->x.infix->right));
		   if (temp = first_ptr_addr(code,node->x.infix->right))
			return (temp);
		   }
		if (node->next)
		   {
		   if (node->next->code == code) return (&(node->next));
		   return (first_ptr_addr(code,node->next));
		   }
		break;
		}
      };

   return ((RIGHT_NODE_PTR*) 0);			       /* Catch all */
   }







public void find_unnecessary_LPNT_ptr_addr(queue,node)
   QUEUE_PTR queue;
   RIGHT_NODE_PTR node;
/*
 * find_unnecessary_LPNT_ptr_addr sweeps through the right side of an EBNF
 * rule searching for unnecessary parenthesis. When any such node is found,
 * the address of the pointer in the node that points at the LPNT
 * node is entered into queue.
 *
 * On Entry:
 *	node contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 * On Exit:
 *	A pointer to any occurrence of an unnecessary LPNT is saved in queue.
 *
*/
   {
   if (!node) return;

   switch (node->code) {

	/* Nodes that have 1 continuation */
	case IST :
		{
		if (node->next)
		    if (node->next->code == LPNT)
			if (LPNT_unnecessary(node->next,FALSE,TRUE))
				enqueue(queue,((void *) (&(node->next))) );
		find_unnecessary_LPNT_ptr_addr(queue,node->next);
		break;
		}

	case AMPT :	       /* Don't care about connection point nesting */
	case  ATT :	       		/* Don't care about PGS "@" nesting */
	case DOLT :	       		/* Don't care about PGS "$" nesting */
	case IDNT :
	case NULT :
	case GTRT :
	case LSST :
	case LITT :
		{
		if (node->next)
		    if (node->next->code == LPNT)
			if (LPNT_unnecessary(node->next,FALSE,FALSE))
				enqueue(queue,((void *) (&(node->next))) );
		find_unnecessary_LPNT_ptr_addr(queue,node->next);
		break;
		}

	/* Nodes that have 2 continuations */
	case LPNT :
	case LBKT :
		{
		if (node->x.nest)
		    if (node->x.nest->code == LPNT)
			if (LPNT_unnecessary(node->x.nest,FALSE,FALSE))
				enqueue(queue,((void *) (&(node->x.nest))) );
		find_unnecessary_LPNT_ptr_addr(queue,node->x.nest);
		if (node->next)
		    if (node->next->code == LPNT)
			if (LPNT_unnecessary(node->next,FALSE,FALSE))
				enqueue(queue,((void *) (&(node->next))) );
		find_unnecessary_LPNT_ptr_addr(queue,node->next);
		break;
		}

   	case ASTT :
	case PLST :
		{
		if (node->x.nest)
		    if (node->x.nest->code == LPNT)
			if (LPNT_unnecessary(node->x.nest,TRUE,FALSE))
				enqueue(queue,((void *) (&(node->x.nest))) );
		find_unnecessary_LPNT_ptr_addr(queue,node->x.nest);
		if (node->next)
		    if (node->next->code == LPNT)
			if (LPNT_unnecessary(node->next,TRUE,FALSE))
				enqueue(queue,((void *) (&(node->next))) );
		find_unnecessary_LPNT_ptr_addr(queue,node->next);
		break;
		}

	/* Nodes that have 3 continuations */
	case BART :
	case SEPT :
		{
		if (node->x.infix->left)
		    if (node->x.infix->left->code == LPNT)
			if (LPNT_unnecessary(node->x.infix->left,TRUE,FALSE))
			   enqueue(queue,((void *) (&(node->x.infix->left))));
		find_unnecessary_LPNT_ptr_addr(queue,node->x.infix->left);
		if (node->x.infix->right)
		    if (node->x.infix->right->code == LPNT)
			if (LPNT_unnecessary(node->x.infix->right,TRUE,FALSE))
			   enqueue(queue,((void *)(&(node->x.infix->right))));
		find_unnecessary_LPNT_ptr_addr(queue,node->x.infix->right);
		if (node->next)
		    if (node->next->code == LPNT)
			if (LPNT_unnecessary(node->next,TRUE,FALSE))
				enqueue(queue,((void *) (&(node->next))) );
		find_unnecessary_LPNT_ptr_addr(queue,node->next);
		break;
		}

	};

   }







public void find_token_ptr_addr(toklist,queue,node)
   short *toklist;
   QUEUE_PTR queue;
   RIGHT_NODE_PTR node;
/*
 * On Entry:
 *	toklist points at an array of NUM_TOKENS elements. If a token is
 *		to be matched, the entry indexed by its code will
 *		be TRUE, otherwise FALSE.
 *      node contains a pointer to a node on the right side of a
 *         rule in the EBNF internal structure.
 * On Exit:
 *      The address of the pointer to any occurrence of a node 
 *	with a type in toklist has been placed in queue. The order is
 *	based on a left to right depth first search.
 *
*/
   {
   if (!node) return;
   

   switch (node->code) {
        /* Types that have the next continuation only */
	case  ATT :			/* Don't care about PGS "@" nesting */
	case DOLT :			/* Don't care about PGS "$" nesting */
        case  IST :
        case GTRT :
        case LSST :
        case LITT :
        case IDNT :
                {
		if (node->next)
			{
			if (toklist[node->next->code])
				enqueue(queue,((void *) (&(node->next))) );
                	find_token_ptr_addr(toklist,queue,node->next);
			}
                break;
                }

        /* Types that have nest and next continuations */
        case AMPT :
        case ASTT :
        case PLST :
        case LPNT :
        case LBKT :
                {
		if (node->x.nest)
			{
			if (toklist[node->x.nest->code])
				enqueue(queue,((void *) (&(node->x.nest))) );
	                find_token_ptr_addr(toklist,queue,node->x.nest);
			}
		if (node->next)
			{
			if (toklist[node->next->code])
				enqueue(queue,((void *) (&(node->next))) );
	                find_token_ptr_addr(toklist,queue,node->next);
			}
                break;
                }

        /* Types that have 2 nested and a next continuation */
        case BART :
        case SEPT :
                {
		if (node->x.infix->left)
			{
			if (toklist[node->x.infix->left->code])
			    enqueue(queue,((void *)(&(node->x.infix->left))));
	                find_token_ptr_addr(toklist, queue,
					    node->x.infix->left);
			}
		if (node->x.infix->right)
			{
			if (toklist[node->x.infix->right->code])
			   enqueue(queue,((void *)(&(node->x.infix->right))));
	                find_token_ptr_addr(toklist, queue,
					    node->x.infix->right);
			}
		if (node->next)
			{
			if (toklist[node->next->code])
				enqueue(queue,((void *) (&(node->next))) );
	                find_token_ptr_addr(toklist,queue,node->next);
			}
                break;
                }
      }

   }







public void traversal_ptr_addr(queue,node)
   QUEUE_PTR queue;
   RIGHT_NODE_PTR node;
/*
 * On Entry:
 *      node contains a pointer to a node on the right side of a
 *         rule in the EBNF internal structure.
 * On Exit:
 *      The address of the pointer to every node in the right side
 *	(excluding the first node) has been placed in queue in
 *	traversal order.
 *
*/
   {
   if (!node) return;

   switch (node->code) {
        /* Types that have the next continuation only */
	case  ATT :			/* Don't care about PGS "@" nesting */
	case DOLT :			/*  "     "    "     "  "$" nesting */
        case AMPT :			/*  "     "    " connection nesting */
        case  IST :
        case GTRT :
        case LSST :
        case LITT :
        case IDNT :
                {
		if (node->next)
			{
			enqueue(queue,((void *) (&(node->next))) );
                	traversal_ptr_addr(queue,node->next);
			}
                break;
                }

        /* Types that have nest and next continuations */
        case ASTT :
        case PLST :
        case LPNT :
        case LBKT :
                {
		if (node->x.nest)
			{
			enqueue(queue,((void *) (&(node->x.nest))) );
	                traversal_ptr_addr(queue,node->x.nest);
			}
		if (node->next)
			{
			enqueue(queue,((void *) (&(node->next))) );
	                traversal_ptr_addr(queue,node->next);
			}
                break;
                }

        /* Types that have 2 nested and a next continuation */
        case BART :
        case SEPT :
                {
		if (node->x.infix->left)
			{
			enqueue(queue,((void *)(&(node->x.infix->left))));
	                traversal_ptr_addr(queue,node->x.infix->left);
			}
		if (node->x.infix->right)
			{
			enqueue(queue,((void *)(&(node->x.infix->right))));
	                traversal_ptr_addr(queue,node->x.infix->right);
			}
		if (node->next)
			{
			enqueue(queue,((void *) (&(node->next))) );
	                traversal_ptr_addr(queue,node->next);
			}
                break;
                }
      }

   }








/* --------------- D E C I S I O N -----------------------------------------*/

public int LPNT_unnecessary(node,ASTT_or_PLST_decended,previous_IST)
   RIGHT_NODE_PTR node;
   char ASTT_or_PLST_decended;
   char previous_IST;
/*
 * LPNT_unnecessary is used to determine whether given parenthesis are
 * really necessary in a given spot in a grammar structure.
 *
 * On Entry:
 *	node points at a LPNT node in a grammar structure.
 *	ASTT_or_PLST_decended is TRUE of the LPNT is a direct decendedent
 *	    of an ASTT or PLST node, FALSE otherwise.
 *	previous_IST is TRUE if the node pointing at the LPNT is an IST.
 *
 * On Exit:
 *	LPNT_unnecessary returns FALSE if the parenthesis should remain
 *	    in place, TRUE if it is OK to remove them.
 *
 * LPNT returns TRUE (parenthesis not needed) if:
 *
 *
 *	(previous_IST is TRUE) and
 *	(The node to the right of the LPNT is DOTT)
 *
 *				OR
 *
 *	(ASTT_or_PLST_decended is FALSE) and
 *	(The direct decendent from the LPNT is not a BART or SEPT node)
 *
 *				OR
 *
 *	(ASTT_or_PLST_decended is TRUE) and
 *	(The direct decendent from the LPNT is not BART,SEPT,ASTT,or PLST) and
 *	(The level directly decended from the LPNT is one node long
 *	      i.e. The only decendent is the direct decendent)
 *
 *	Example 1:
 *
 *         :-----:   :---:   :---:
 *	-->| ::= |-->| ( |-->| . |			::= (B | C) .
 *         |_____|   |___|   |___|
 *                     |
 *                     V
 *                   :---:   :---:
 *                   | | |-->| ) |
 *                   |___|   |___|
 *                    | |
 *                ____| |____
 *                |         |
 *                V         V
 *              :---:     :---:
 *              | B |     | C |
 *              |___|     |___|
 *
 *		becomes
 *
 *         :-----:   :---:   :---:
 *	-->| ::= |-->| | |-->| . |			::= B | C .
 *         |_____|   |___|   |___|
 *                    | |
 *                ____| |____
 *                |         |
 *                V         V
 *              :---:     :---:
 *              | B |     | C |
 *              |___|     |___|
 *
 *	Example 2:
 *
 *         :---:   :---:   :---:   :---:
 *	-->| A |-->| ( |-->| D |-->| . |		A (B C) D .
 *         |___|   |___|   |___|   |___|
 *                   |
 *                   |
 *                 :---:   :---:   :---:
 *                 | B |-->| C |-->| ) |
 *                 |___|   |___|   |___|
 *
 *		becomes
 *
 *         :---:   :---:   :---:   :---:   :---:
 *	-->| A |-->| B |-->| C |-->| D |-->| . |	A B C D .
 *         |___|   |___|   |___|   |___|   |___|
 *
 *	Example 3:
 *
 *         :---:   :---:   :---:   :---:
 *	-->| A |-->| * |-->| C |-->| . |		A (B)* C .
 *	   |___|   |___|   |___|   |___|
 *                   |
 *                   |
 *                 :---:
 *                 | ( |
 *                 |___|
 *                   |
 *                   |
 *                 :---:   :---:
 *                 | B |-->| ) |
 *		   |___|   |___|
 *
 *		becomes
 *
 *         :---:   :---:   :---:   :---:
 *	-->| A |-->| * |-->| C |-->| . |		A B* C .
 *         |___|   |___|   |___|   |___|
 *                   |
 *                   |
 *                 :---:
 *		   | B |
 *		   |___|
*/
   {
   int conclusion = FALSE;			      /* Assume it's needed */

   if (!node) return (FALSE);
   if (node->code != LPNT) return (FALSE);

   if (previous_IST)
	if (node->next)
		if (node->next->code == DOTT) return (TRUE);

   if (ASTT_or_PLST_decended)
	switch (node->x.nest->code) {	     /* LPNT always has a decendent */
		case BART :
		case SEPT :
		case ASTT :
		case PLST : break;			 /* Can't remove it */

		default :				 /* Decendent is OK */
			if (node->x.nest->next)
			    if (node->x.nest->next->code == RPNT)
				conclusion = TRUE;	     /* Length == 1 */
		}
      else
	 if ( (node->x.nest->code != BART) && (node->x.nest->code != SEPT) )
		conclusion = TRUE;

   return (conclusion);
   }







public int litt_cr_spoiler(text)
   SYMBOL_PTR_NODE_PTR text;
/*
 *    A literal in a BNF or EBNF rule can disqualify that rule as a
 * chainrule. For instance,
 *
 *              xProgram ::= '(' xBlock ')' .           is a chain rule
 *
 *              xJump ::= 'goto' xIdentifier.           is not a chain rule
 *
 *   It is not really clear that the only literals that don't disqualify
 *   a chain rule are '(' and ')', however, this routine assumes that
 *   this is indeed the case.
 *
 * On Entry:
 *      text points at a valid SYMBOL for a literal.
 *
 * On Exit:
 *      If the symbol pointed at by text disqualifies ("spoils") a
 *         rule from being a chain rule, litt_cr_spoiler returns
 *         TRUE, otherwise FALSE.
*/
   {
   char c[BUFSIZ];
   int i;

   (void)strncpy(c,string[text->symbol_ptr->str],text->symbol_ptr->l);
   c[text->symbol_ptr->l] = '\0';
   for (i=0; i < text->symbol_ptr->l; i++)
        if ( (c[i] != ' ') && (c[i] != '(') && (c[i] != ')') ) return(TRUE);
   return(FALSE);
   }







public int is_chainrule(node,nt_seen,nt_symb,cl)
   RIGHT_NODE_PTR node;
   int nt_seen;
   SYMBOL_PTR_NODE_PTR *nt_symb;
   int cl;
/*
 * On Entry:
 *      Alternation has been removed from the rule.
 *      node contains a pointer to a node on the right side of a
 *         rule in the EBNF internal structure.
 *      nt_seen is true if a non_terminal has already been seen in the
 *         right side.
 *	cl is TRUE if the contents of literal nodes in the
 *	   right hand side should be checked for 'spoilers', FALSE
 *	   if the contents of literals is not considered important.
 *
 * On Exit:
 *      *nt_symb contains the symbol_ptr_node containing the SYMBOL
 *         for the first non-terminal in the rule.
 *      is_chainrule returns TRUE if the rule is a chain rule according
 *         to the definition given above, otherwise FALSE.
*/
   {
   int result;

   if (!node) return(TRUE);

   switch (node->code) {

        /* "Spoiler" nodes */
        case BART :                         /* See interface specifications */
        case SEPT :
        case ASTT :
        case PLST :
        case LBKT :
        case RBKT :
                {
                result = FALSE;
                break;
                }

        /* Nodes that have 2 continuations */
        case LPNT :
                {
                result = ( is_chainrule(node->x.nest,nt_seen,nt_symb,cl) &&
                           is_chainrule(node->next,nt_seen,nt_symb,cl) );
                break;
                }

        /* Identifiers */
        case IDNT :
                if ( !(node->x.text->symbol_ptr->non_term) || nt_seen)
                      result = FALSE;
                   else
                      {
		      *nt_symb = node->x.text;
                      result = is_chainrule(node->next,TRUE,nt_symb,cl);
                      }
		break;

        /* Literals */
        case LITT :
		if (cl && litt_cr_spoiler(node->x.text)) /*Short circ. eval*/
			result = FALSE;
		    else
	                result = is_chainrule(node->next,nt_seen,nt_symb,cl);
                break;

        /* Nodes that have 1 continuation */
        case AMPT :                   /* Ignore the nested connection point */
	case  ATT :				  /* Ignore PGS "@" nesting */
	case DOLT :				  /* Ignore PGS "$" nesting */
        case NULT :
        case  IST :
        case GTRT :
        case LSST :
                {
                result = is_chainrule(node->next,nt_seen,nt_symb,cl);
                break;
                }

        /* Nodes that indicate a dead end */
        case RPNT :
        case DOTT : result = nt_seen;                  /* This branch is OK */

      };

   return(result);
   }







/* --------------- C O M P A R I S O N -------------------------------------*/

public int same_left_side(rule1,rule2)
   LEFT_NODE_PTR rule1,
		 rule2;
/*
 * On Entry:
 *	rule1 contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	rule2 contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	Neither left side has it's code field set to NULT.
 * On Exit:
 *	same_left_side returns TRUE if the left side of the two rules
 *         are the same.
*/
   {
   /* If it isn't NULT, then it is identifier, by definition */
   return( rule1->text->symbol_ptr == rule2->text->symbol_ptr );

   }







public int same_right_side(rule1,rule2)
   RIGHT_NODE_PTR rule1,
		  rule2;
/*
 * On Entry:
 *	rule1 contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 *	rule2 contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 * On Exit:
 *	same_right_side returns TRUE if the right side of the two rules,
 *         starting from the nodes pointed at by rule1 and rule2 represent
 *	   identical EBNF subtrees.
 *
 * NOTE: This routine does not take NULT nodes into consideration. Thus,
 *	 it is possible that if future CAGT routines can cause such
 *	 nodes to occur in a rule, this routine will flag seemingly
 *	 identical rules as being different. If that becomes a problem,
 *	 this routine should be extended to cover such nodes.
 *
*/
   {
   int result;

   if ((rule1 == (RIGHT_NODE_PTR) 0) || (rule2 == (RIGHT_NODE_PTR) 0))
	if ((rule1==(RIGHT_NODE_PTR) 0)&&(rule2==(RIGHT_NODE_PTR) 0))
		return(TRUE);
	   else
		 return(FALSE);

   if (rule1->code != rule2->code) return(FALSE);

   switch (rule1->code) {
	/* Instantly positive */
	case DOTT :
	case RPNT:
	case RBKT:
		{
		result = TRUE;
		break;
		};

	/* Single continuation */
	case NULT :					    /* Deleted node */
	case  IST :
	case GTRT : 
	case LSST :
		{
		result = same_right_side(rule1->next,rule2->next);
		break;
		}

	/* Literals and Identifiers */
	case LITT :
	case IDNT:
		{
		if (rule1->x.text->symbol_ptr != rule2->x.text->symbol_ptr)
			result = FALSE;
		   else
			result = same_right_side(rule1->next,rule2->next);
		break;
		}

	/* Two continuations */
	case AMPT : 
	case  ATT :
	case DOLT :
	case ASTT : 
	case PLST : 
	case LPNT :
	case LBKT :
		{
		if (result = same_right_side(rule1->x.nest,rule2->x.nest))
			result = same_right_side(rule1->next,rule2->next);
		break;
		}

	/* Three continuations */
	case BART : 
	case SEPT : 
		{
		if(result = same_right_side(rule1->x.infix->left,
					    rule2->x.infix->left) )
		   if (result = same_right_side(rule1->x.infix->right,
						rule2->x.infix->right) )
		      result = same_right_side(rule1->next,rule2->next);
		break;
		}
	}

   return(result);
   }







public int same_rule(rule1,rule2)
   LEFT_NODE_PTR rule1,
		 rule2;
/*
 * On Entry:
 *	rule1 contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	rule2 contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	Neither left side has it's code field set to NULT.
 * On Exit:
 *	same_rule returns TRUE if the two rules are the same.
*/

   {
   int result;

   if (result = same_left_side(rule1,rule2))
	result = same_right_side(rule1->right_side,rule2->right_side);

   return(result);
   }







/* --------------- I N D E X E D   A C C E S S ------------------------------*/

public void init_indexed_access(grammar,access_blk_ptr)
   LEFT_NODE_PTR grammar;
   struct indexed_access_block *access_blk_ptr;
/* 
 * On Entry:
 *	grammar contains a pointer to an existing EBNF grammar structure
 *	access_blk_ptr points to an uninitialized access_block.
 *
 * On Exit:
 *	The indexed_access_block pointed at by access_blk_ptr has
 *	   been initialized to represent the grammar pointed at
 *	   by grammar. A pointer to any rule can then be had in
 *	   constant time via a call to rule_by_index.
*/
   {
   LEFT_NODE_PTR temp;			      /* Used to move through rules */
   int cur_rule;				  /* Number of current rule */

   access_blk_ptr->rule_cnt = 0;
   temp = grammar;
   while (temp)
	{
	access_blk_ptr->rule_cnt++;
	temp = temp->next_rule;
	}

   GET_MEMORY(access_blk_ptr->block, LEFT_NODE_PTR *,
	      access_blk_ptr->rule_cnt, LEFT_NODE_PTR,
	      "init_indexed_access", 1)
   temp = grammar;
   for (cur_rule = 0; cur_rule < access_blk_ptr->rule_cnt; cur_rule++)
	{
	*(access_blk_ptr->block + cur_rule)  = temp;
	temp = temp->next_rule;
	}

   }







public LEFT_NODE_PTR rule_by_index(access_blk_ptr,rule_num)
   struct indexed_access_block *access_blk_ptr;
   int rule_num;
/* 
 * On Entry:
 *	access_blk_ptr points to a valid indexed_access_block.
 *	rule_num is the number of the desired rule (1<= rule_num)
 *
 * On Exit:
 *	rule_by_index returns a pointer to the desired rule.
*/
   {
   if ((rule_num > access_blk_ptr->rule_cnt) || (rule_num < 1))
	cagt_msg(0, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CNTINXRUL)], rule_num))
   return(*(access_blk_ptr->block + rule_num -1));
   }







public void delete_indexed_access(access_blk_ptr)
   struct indexed_access_block *access_blk_ptr;
/* 
 * On Entry:
 *	access_blk_ptr points to a valid access_block.
 *
 * On Exit:
 *	The storage used by the constant time access front end has
 *	   been returned to the operating system.
*/
   {
   access_blk_ptr->rule_cnt = 0;

   FREE_MEMORY(access_blk_ptr->block, "delete_indexed_access", 1)

   }
