static char RCSid[] = "$Id: cagt.c,v 1.19 1993/01/22 23:38:47 kadhim Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */

/* Abstract:
 *    	CAGT is designed to aid the compiler writer in the task of producing
 * 	an abstract grammar from a given concrete grammar, and then later,
 * 	once the abstract grammar has been augmented with connection points,
 * 	to automatically generate a parsing grammar (The original concrete
 * 	grammar with the connection points inserted).
 *
 * Routines:
 *	1) cagt_getopt  -   Public domain version of getopt(3) modified for
 *			    cagt.
 *      2) cagt_cmd_args -  Get command line arguments,
 *			    set global variables accordingly.
 *      3) cagt_forward -   Implements the cagt forward processing phase
 *                          (Concrete Grammar to Abstract Grammar).
 *	4) cagt_write_fwd - Write results of forward pass to appropriate files.
 *      5) cagt_reverse -   Implements the cagt reverse processing phase
 *                          (Decorated Abstract Grammar to Parsing Grammar).
 *      6) main - 	    Entry point for the cagt tool.
 */

#include <stdio.h>
#include <errno.h>
#include <sys/file.h>		/* needed for open call */
#include <sys/param.h>		/* needed for MAXPATHLEN */
#include <sys/dir.h>		/* needed for MAXNAMLEN */
#include <string.h>


extern int errno;


#include "cagt_config.h"
#include "cagt.h"
#include "cagt_usr.h"
#include "cagt_usr_err.h"


/* global information required by various top-level routines. Generally set by
 * cagt_cmd_args() and accessed by other top-level routines.
*/

public char *cmd_name;		/* pointer to invocation name (argv[0]) */
public char *grammar;		/* grammar name from command line */

public char con_fname[MAXNAMLEN];	/* concrete grammar - input file */
public char abs_fname[MAXNAMLEN];	/* abstract grammar */
public char dec_fname[MAXNAMLEN];	/* decorated abstract grammar */
public char rel_fname[MAXNAMLEN];	/* relationship file */
public char sym_fname[MAXNAMLEN];	/* Symbol equivalence file */
public char cull_fname[MAXNAMLEN];	/* Rule cull file */
public char par_fname[MAXNAMLEN];	/* parsing grammar */
public char lis_fprefix[MAXNAMLEN];	/* listing file w/o numeric suffix */
public char err_fname[MAXNAMLEN];	/* error listing file */

#define CAGT_FORWARD  0	  /* produce abstract grammar from concrete grammar */
#define CAGT_CONTINUE 1	  /* continue forward pass transformations */
#define CAGT_REVERSE  2	  /* produce parsing grammar */

/* CAGT mode is FORWARD unless tool invoked with "-r" or "-c" options */
private int cagt_mode = CAGT_FORWARD;

/* This flag is TRUE for batch mode operation, and FALSE otherwise */
private int cagt_batch = FALSE;

/* This flag is TRUE if cagt should be verbose about what it's doing,
 * and FALSE otherwise.
*/
public int cagt_verbose = FALSE;


/* common data structures referenced by cagt_forward and cagt_reverse. */
public LEFT_NODE_PTR cagt_gram;


/**********************************/
#define ERR(s, c)	if(opterr){\
	extern int write();\
	char errbuf[2];\
	errbuf[0] = c; errbuf[1] = '\n';\
	(void) write(2, argv[0], strlen(argv[0]));\
	(void) write(2, s, strlen(s));\
	(void) write(2, errbuf, 2);}

/* Global variables referenced by cagt_getopt. See the getopt(3) manpage
 * for a discussion of what these mean.
*/
public int opterr = 1;
public int optind = 1;
public int optopt;
public char *optarg;

public int cagt_getopt(argc, argv, opts)
     int argc;
     char **argv, *opts;
{
  static int sp = 1;
  register int c;
  register char *cp;
  
  if(sp == 1)
    if(optind >= argc || argv[optind][0] != '-' || argv[optind][1] == '\0')
      return(EOF);
    else if(!strcmp(argv[optind], "--")) {
      optind++;
      return(EOF);
    }
  optopt = c = argv[optind][sp];
  if(c == ':' || (cp=strchr(opts, c)) == (char *) 0) {
    ERR(": illegal option -- ", c);
    if(argv[optind][++sp] == '\0') {
      optind++;
      sp = 1;
    }
    return('?');
  }
  if(*++cp == ':') {
    if(argv[optind][sp+1] != '\0')
      optarg = &argv[optind++][sp+1];
    else if(++optind >= argc) {
      ERR(": option requires an argument -- ", c);
      sp = 1;
      return('?');
    } else
      optarg = argv[optind++];
    sp = 1;
  } else {
    if(argv[optind][++sp] == '\0') {
      sp = 1;
      optind++;
    }
    optarg = NULL;
  }
  return(c);
}







private int cagt_cmd_args(argc, argv)
     int argc;
     char *argv[];
/*
 * cagt_cmd_args - retrieve command line arguments 
 *
 * entry:
 *	argc, argv - command line args passed to main()
 *
 * exit:
 *	Global variables have been set based on the command arguments.
 *	The following flags are processed:
 *
 *		-c  	Indicates this is to be a continuation of a previous
 *		    	forward pass and the abstract grammar and relationship
 *		    	pointer files should be available from that pass.
 *		-r  	Indicates this is the reverse (abstract-to-parsing 
 *			grammar) pass. The original concrete grammar, the
 *			corresponding cagt-constructed absolute grammar and
 *			relative pointer files, plus a decorated abstract 
 *			grammar file should exist before this pass is invoked.
 *		-b	Batch mode, forward pass without interactive step
 *		-v	Verbose. Tells cagt to tell what its doing during
 *			non-interactive stages by issuing messages to stderr.
 *		    
 */
{
  int i;
  
  cmd_name = argv[0];
  while ( (i = cagt_getopt(argc,argv,"bcrv")) != EOF)
    switch (i) {
    case 'b' :
      cagt_batch = TRUE;
      break;
    case 'c' :
      if (cagt_mode == CAGT_REVERSE) return(SYS_ERR);
      cagt_mode = CAGT_CONTINUE;
      break;
    case 'r' :
      if (cagt_mode == CAGT_CONTINUE) return(SYS_ERR);
      cagt_mode = CAGT_REVERSE;
      break;
    case 'v' :
      cagt_verbose = TRUE;
      break;
      default :
	return(SYS_ERR);
    }
  
  /* Is there a grammar to work on? There should only be a single arg left */
  if (optind != (argc-1)) return(SYS_ERR);
  
  grammar = argv[optind];  /* grammar is last on command line */
  
  /* use the grammar name for file prefixes */
  (void) strcpy(con_fname, grammar);
  (void) strcpy(abs_fname, grammar);
  (void) strcpy(dec_fname, grammar);
  (void) strcpy(rel_fname, grammar);
  (void) strcpy(sym_fname, grammar);
  (void) strcpy(cull_fname, grammar);
  (void) strcpy(par_fname, grammar);
  (void) strcpy(err_fname, grammar);
  (void) strcpy(lis_fprefix, grammar);
  
  /* concatenate suffixes */
  (void) strcat(con_fname, ".con");
  (void) strcat(abs_fname, ".abs");
  (void) strcat(dec_fname, ".dec");
  (void) strcat(rel_fname, ".rel");
  (void) strcat(sym_fname, ".sym");
  (void) strcat(cull_fname, ".cull");
  (void) strcat(par_fname, ".par");
  (void) strcat(err_fname, ".err");
  (void) strcat(lis_fprefix, ".lis");
  
  return(0);
}







private cagt_write_fwd()
/*
 * Write results of forward pass to appropriate files.
 *
 * entry:
 *	cagt_cmd_args has returned successfully.
 *	internal grammar representation cagt_gram exists.
 *
 * exit:
 *	The files have been written.
 */
{
  FILE *abs_outf, *rel_outf, *sym_outf;
  
  /* open output files */
  if ((abs_outf = fopen(abs_fname, "w")) == 0)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			       "Abstract Grammar", abs_fname, "output"))
  if ((rel_outf = fopen(rel_fname, "w")) == 0)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			       "Relationship Pointer", rel_fname, "output"))
  if ((sym_outf = fopen(sym_fname, "w")) == 0)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			       "Symbol Equivalence", sym_fname, "output"))
	      
	      
  /* write results */
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Writing abstract grammar."))
  output_ebnf_grammar(abs_outf, cagt_gram);
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Writing relationship file."))
  output_rel_file(rel_outf, cagt_gram);
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Writing symbol equivalence file."))
  output_symbol_assoc(sym_outf);
  
  /* close output files */
  if (fclose(abs_outf) == EOF)
    cagt_msg(errno,MSG_EXIT,(cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			     "Abstract Grammar",abs_fname))
  if (fclose(rel_outf) == EOF)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			       "Relationship Pointer", rel_fname))
  if (fclose(sym_outf) == EOF)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			       "Symbol Equivalence", sym_fname))
}







private void cagt_forward()
/*
 * The cagt forward pass, producing an abstract grammar from the
 * concrete grammar.
 *
 * exit:
 *	If the user has altered the concrete grammar and indicated
 *	a desire to save his work then the new abstract grammar
 *	reflecting those manipulations has been written to file
 *	abs_fname, the relationship pointers have been written to
 *	the file rel_fname, and symbol equivalences
 *	have been written to sym_fname - all in the user's current
 *	directory.
*/
{
  int  confd;			  /* Concrete Grammar input file descriptor */
  FILE *relf;				       /* Relationship file pointer */
  int retstat;
  int write_output;

  /* either use abs grammar and rel chains as concrete... */
  if (cagt_mode == CAGT_CONTINUE) 
    {
      if ((confd = open(abs_fname, O_RDONLY)) <= SYS_ERR)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
				   "Abstract Grammar", abs_fname, "input"))
      if ((relf = fopen(rel_fname, "r")) == 0)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text,msg_arr[-(CAGT_OPNERR)],
				   "Relationship Pointer", rel_fname, "input"))

      /* read abstract grammar, attach stored relationship chains */
      if (cagt_verbose)
	cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			      "Reading existing abstract grammar."))
      cagt_gram = get_ebnf_grammar(abs_fname, confd, TRUE);
      if (cagt_verbose)
	cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			      "Reading existing relationship file."))
      attach_stored_rel_chains(relf,cagt_gram);
      if (close(confd) <= SYS_ERR)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
				   "Abstract Grammar", abs_fname))
      if (fclose(relf) == EOF)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
				   "Relationship Pointer", rel_fname))
    } else {    /* or start things off with new concrete grammar */
      if ((confd = open(con_fname, O_RDONLY)) <= SYS_ERR)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
				   "Concrete Grammar", con_fname, "input"))
      if (cagt_verbose)
	cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			      "Reading concrete grammar."))
      cagt_gram = get_ebnf_grammar(con_fname, confd, TRUE);

      /* NOTE: In the next 2 statements, order is important */
      add_rel_chain(cagt_gram);
      remove_pgs_nodes(cagt_gram); /*Remove PGS related nodes*/

      if (close(confd) <= SYS_ERR)
	cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
				   "Concrete Grammar", con_fname))
    }
    
  /* Do the renaming of identifiers specified in sym_fname */
  process_sym_file(cagt_gram);

  if (cagt_batch)
    {			/* Just write the current grammar */
      write_output = TRUE;
    } else {			/* Allow the normal interactive pass */
      if ((retstat = (* cagt_interface.init)()) <= SYS_ERR)
	cagt_msg(0, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_INITERR)], 0))

      if ((retstat = (* cagt_interface.forward)()) <= SYS_ERR)
	{
	  if ((* cagt_interface.shutdown)() <= SYS_ERR)
	    cagt_msg(0,MSG_EXIT,(cagt_msg_text, msg_arr[-(CAGT_FATLABRT)], 0))
	  else
	    cagt_msg(0,MSG_EXIT,(cagt_msg_text, msg_arr[-(CAGT_FATLSHUT)], 0))
	} else {
	  write_output = (retstat == USR_SAVFWD);
	}
	
      if ((* cagt_interface.shutdown)() <= SYS_ERR)
	cagt_msg(0, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_FATLABRT)], 0))
    }
    
  /* Remove any rules specified in cull_fname file from the abs grammar */
  process_cull_file(cagt_gram);

  /* Write the output files if desired */
  if (write_output) cagt_write_fwd();

}







private void cagt_reverse()
/*
 * Reverse pass of cagt
 *
 * exit:
 *	A parsing grammar derived from the concrete, abstract, and 
 *	decorated abstract grammars and stored relationship pointers
 *	has been written to the parsing grammar file, or else an error
 *	listing has been provided.
 */
{
  int confd;			  /* Concrete Grammar input file descriptor */
  int absfd;			  /* Abstract Grammar input file descriptor */
  int decfd;		       /* Decorated Abs Gram. input file descriptor */
  LEFT_NODE_PTR abs_gram;	                        /* Abstract Grammar */
  LEFT_NODE_PTR dec_gram;	              /* Decorated Abstract Grammar */
  FILE *relf;			         /* Relationship file input pointer */
  FILE *parf;			     /* Parsing Grammar output file pointer */
  struct indexed_access_block con_access_blk;    /* Allows Indexed Access */

  /* Open input files */
  if ((confd = open(con_fname, O_RDONLY)) <= SYS_ERR)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			       "Concrete Grammar", con_fname, "input"))
  if ((absfd = open(abs_fname, O_RDONLY)) <= SYS_ERR)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			       "Abstract Grammar", abs_fname, "input"))
  if ((decfd = open(dec_fname, O_RDONLY)) <= SYS_ERR)
    cagt_msg(errno,MSG_EXIT,(cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			     "Decorated Abstract Grammar",dec_fname,"input"))
  if ((relf = fopen(rel_fname,"r")) == 0)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			       "Relationship Pointer", rel_fname, "input"))

  /* Read in the abstract grammar */
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Writing  abstract grammar."))
  abs_gram = get_ebnf_grammar(abs_fname, absfd, TRUE);
      
  /* Read in the decorated abstract grammar */
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Writing  decorated abstract grammar."))
  dec_gram = get_ebnf_grammar(dec_fname, decfd, FALSE);
      
  /* Replace the original abstract grammar by the decorated rules in the
   * right order. The decorated abstract grammar structure is destroyed
   * in the process.
   */
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Reordering decorated abstract grammar."))
  (void) reorder_dec_to_abs(abs_gram, dec_gram);

  /* Attach the saved relationship chains to the new abstract grammar */  
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Reading relationship information."))
  attach_stored_rel_chains(relf,abs_gram);

  /* Read in concrete grammar and create parsing grammar */
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Reading concrete grammar."))
  cagt_gram = get_ebnf_grammar(con_fname, confd, FALSE);
  init_indexed_access(cagt_gram, &con_access_blk);
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Performing abstract to parsing operation."))
  abstract_to_parsing(&con_access_blk, abs_gram);

  /* save the parsing grammar */
  if ((parf = fopen(par_fname, "w")) == 0)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_OPNERR)],
			       "Parsing Grammar", par_fname, "output"))
  if (cagt_verbose)
    cagt_msg(0, MSG_RET, (cagt_msg_text, msg_arr[-(CAGT_VERBOSE)],
			  "Writing parsing grammar."))
  output_ebnf_grammar(parf, cagt_gram);

  /* Close files before exiting */
  if (close(confd) <= SYS_ERR)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			       "Concrete Grammar",con_fname))
  if (close(absfd) <= SYS_ERR)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			       "Absolute Grammar",abs_fname))
  if (close(decfd) <= SYS_ERR)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			       "Decorated Abstract Grammar", dec_fname))
  if (fclose(relf) == EOF)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			       "Relationship Pointer", rel_fname))
  if (fclose(parf) == EOF)
    cagt_msg(errno, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_CLOSERR)],
			       "Parsing Grammar",par_fname))
}







main(argc,argv)
int argc; 
char *argv[];
{
  if (cagt_cmd_args(argc, argv) <= SYS_ERR)
    cagt_msg(0, MSG_EXIT, (cagt_msg_text, msg_arr[-(CAGT_USERR)], cmd_name))

  if (cagt_mode == CAGT_REVERSE)
    cagt_reverse();
  else
    cagt_forward();

  return(0);
}

