/*
 * NAME
 *     Room.c -- Room handling functions
 *
 * AUTHOR
 *     Ken MacLeod
 */

#ifndef lint
static char sccsId [] = "@(#) Room.c  1.3 11/11/90 07:40:45\n\t";
#endif

#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include "ErrorLog.h"
#include "defs.h"
#include "Unidel.h"
#include "User.h"
#include "Message.h"
#include "Room.h"

Article *articles = NULL;
int numArticles;
void LoadArticles ();

EnterRoom ()

{
	FILE *mailPipe;
	char roomDescription[_bufSize], roomName[_bufSize], commandString[_bufSize];
	int private, fileRoom;

	(void) printf ("What is the new room's name? ");
	GetString (roomName, _lowerCase);
	(void) printf ("Give a short description of the room:\n: ");
	GetString (roomDescription, _normal);
	private = AnswerYesNo ("Is this a private room (Y/[N])? ", _defaultAnswer);
	if (private == -1)
		return;
	fileRoom = AnswerYesNo ("Is this a file room (Y/[N])? ", _defaultAnswer);
	if (fileRoom == -1)
		return;
	(void) printf ("\nDue to a small problem, new rooms cannot be automatically created,\n");
	(void) printf ("the sysop will be told about the new room and will mail you when it has\n");
	(void) printf ("been created.\n");

	(void) sprintf (commandString, "/bin/mail sysop");
	mailPipe = popen (commandString, "w");
	if (mailPipe == NULL) {
		(void) printf ("*** ERROR ***  Unable to mail to sysop, new room message NOT sent.");
		ErrorLog ("EnterRoom:001", _error, "Cannot open pipe to 'mail'");
		return;
	}
	(void) fprintf (mailPipe, "To: sysop@%s\n", _siteName);
	(void) fprintf (mailPipe, "From: %s@%s\n", logname, _siteName);
	(void) fprintf (mailPipe, "Sender: unidel@%s\n", _siteName);
	(void) fprintf (mailPipe, "Subject: Request for room creation.\n");
	(void) fprintf (mailPipe, "\n");
	(void) fprintf (mailPipe, "#ROOM %s\n", roomName);
	(void) fprintf (mailPipe, "#DESC %s\n", roomDescription);
	(void) fprintf (mailPipe, "#PRIV %s\n", private ? "Yes" : "No");
	(void) fprintf (mailPipe, "#FILE %s\n", fileRoom ? "Yes" : "No");
	(void) pclose (mailPipe);
}

char *
RoomNameString (room)

unsigned int room;

{
	static char roomName[_bufSize];	/* XXX Can we allocate this? */

	if (rooms[room].description && (strlen (rooms[room].description) < _aliasLength))
	  (void)strcpy (roomName, rooms[room].description);
	else
	  (void)strcpy (roomName, rooms[room].name);

	(void)strcat (roomName, (rooms[room].fileDir != NULL) ? "]" : ">");
	
	return (roomName);
}

PrintForgottenRooms ()

{
	unsigned int room, floor;

	(void) printf ("*** Forgotten public rooms:\n");
	Columnize ((char *) NULL);
	for (room = 0; rooms[room].name != NULL; room ++) {
		if ((rooms[room].forgotten && !rooms[room].private)
		  && (!floorMode || rooms[room].floor == currentFloor)
		  && (rooms[room].inviteOnly == rooms[room].invited)) {
			Columnize (RoomNameString (room));
		}
	}
	(void) printf ("\n");
	if (floorMode) {
		(void) printf ("*** Forgotten public floors:\n");
		Columnize ((char *) NULL);
		for (floor = 0; floors[floor].name != NULL; floor ++) {
			if (floors[floor].forgotten && !floors[floor].private
			  && (floors[floor].inviteOnly == floors[floor].invited)) {
				if (floors[floor].description)
					Columnize (floors[floor].description);
				else
					Columnize (floors[floor].name);
			}
		}
		(void) printf ("\n");
	}
}

KnownRooms ()

{
	unsigned int i;

	PrintRoomsWithUnreadMessages ();
	(void) printf ("You have read all messages in these rooms:\n");
	Columnize ((char *) NULL);
	for (i = 0; rooms[i].name != NULL; i ++) {
		if (!rooms[i].forgotten && !floors[rooms[i].floor].forgotten
		  && !rooms[i].newArticles && (!floorMode || rooms[i].floor == currentFloor)
		  && (rooms[i].inviteOnly == rooms[i].invited)) {
/*			(void) printf ("%s ", RoomNameString (i)); */
			Columnize (RoomNameString (i));
		}
	}
	(void) printf ("\n");
}

PrintRoomsWithUnreadMessages ()

{
	unsigned int room, floor;

	(void) printf ("There are unread messages in these rooms:\n");
	Columnize ((char *) NULL);
	for (room = 0; rooms[room].name != NULL; room ++) {
		if (!rooms[room].forgotten && !floors[rooms[room].floor].forgotten
		  && rooms[room].newArticles
		  && (!floorMode || rooms[room].floor == currentFloor)
		  && (rooms[room].inviteOnly == rooms[room].invited)) {
			Columnize (RoomNameString (room));
		}
	}
	(void) printf ("\n");
	if (floorMode) {
		(void) printf ("There are unread messages on these floors:\n");
		Columnize ((char *) NULL);
		for (floor = 0; floors[floor].name != NULL; floor ++) {
			if (!floors[floor].forgotten && floor != currentFloor) {
				for (room = 0; rooms[room].name != NULL; room ++) {
					if (!rooms[room].forgotten && rooms[room].newArticles
					  && rooms[room].floor == floor
					  && (rooms[room].inviteOnly == rooms[room].invited)) {
						if (floors[floor].description != NULL)
							Columnize (floors[floor].description);
						else
							Columnize (floors[floor].name);
						break;
					}
				}
			}
		}
	}
	(void) printf ("\n");
}

MarkAllRead ()

{
	int i;
	char articlesRead[512];

	for (i = 0; i < numArticles; i ++)
		articles[i].read = 1;
	(void) sprintf (articlesRead, "1-%d", i ? articles[i-1].number : rooms[currentRoom].highest);
	rooms[currentRoom].newArticles = 0;
	AllocString ("MarkAllRead:001", &rooms[currentRoom].articlesRead, articlesRead);
}

MarkRead ()

{
	/* XXX individual articles should be noted as being read */
}

FindRoomCommand (type)

long type;

{
	char roomName[_bufSize];
	unsigned int room;

	GetString (roomName, _lowerCase);
	if (roomName[0]) {
		if ((room = FindRoom (type, roomName)) == -1)
			(void) printf ("No room matching '%s'.\n", roomName);
		else
			(void) printf ("%s\n", RoomNameString (room));
	}
}

int
FindRoom (type, roomName)

long type;
char *roomName;

{
	register int room;
	char copyOfRoomName[_bufSize], copyOfRoomsName[_bufSize];
	char *ptr;
	int roomNameLength, i;

	if (rooms[0].name == NULL)	/* Dirty, lazy fix for ReadNewsFile */
		return (-1);

	/* convert room name to lower case */
	for (i = 0; roomName[i]; i ++) {
		if (isupper (roomName[i]))
			roomName[i] = tolower (roomName[i]);
	}

	/* search for exact match first */
	room = currentRoom;
	do {
		room ++;
		if (rooms[room].name == NULL)
			room = 0;
		if (strcmp (rooms[room].name, roomName) == 0)
			return (room);
	} while (room != currentRoom);
	if (type == _findExact)
		return (-1);

	/* then search for a partial match if we didn't find an exact match */
	(void)strcpy (copyOfRoomName, roomName);	/* convert to lower case */
	for (i = 0; copyOfRoomName[i]; i ++) {
		if (isupper (copyOfRoomName[i]))
			copyOfRoomName[i] = tolower (copyOfRoomName[i]);
	}
	roomNameLength = strlen (roomName);
	room = currentRoom;
	do {
		room ++;
		if (rooms[room].name == NULL)
			room = 0;
		if (rooms[room].description && (strlen (rooms[room].description) < _aliasLength)) {
		  (void)strcpy (copyOfRoomsName, rooms[room].description);	/* convert to lower case */
			for (i = 0; copyOfRoomsName[i]; i ++) {
				if (isupper (copyOfRoomsName[i]))
					copyOfRoomsName[i] = tolower (copyOfRoomsName[i]);
			}
			ptr = copyOfRoomsName - 1;	/* we prime this for the 'strchr' call */
			while ((ptr = strchr (ptr + 1, copyOfRoomName[0])) != NULL) {
				if (strncmp (ptr, copyOfRoomName, roomNameLength) == 0)
					return (room);
			}
		}
		(void)strcpy (copyOfRoomsName, rooms[room].name);	/* convert to lower case */
		for (i = 0; copyOfRoomsName[i]; i ++) {
			if (isupper (copyOfRoomsName[i]))
				copyOfRoomsName[i] = tolower (copyOfRoomsName[i]);
		}
		ptr = copyOfRoomsName - 1;	/* we prime this for the 'strchr' call */
		while ((ptr = strchr (ptr + 1, copyOfRoomName[0])) != NULL) {
			if (strncmp (ptr, copyOfRoomName, roomNameLength) == 0)
				return (room);
		}
	} while (room != currentRoom);
	return (-1);
}

GotoCommand (type)

long type;

{
	char roomName[_bufSize];
	int room;

	switch ((int) type) {
	case 0:	/* Yawn, mark everything read an let's get out of here */
		MarkAllRead ();
		GotoNew ();
		break;
	case 1:	/* Mark only what we have read */
		MarkRead ();
		GotoNew ();
		break;
	case 2:	/* Don't mark anything read, as if we never came here */
		GotoNew ();
		break;
	case 3:	/* Forget this room, mark everything read on the way out */
		rooms[currentRoom].forgotten = _true;
		MarkAllRead ();
		GotoNew ();
		break;
	case 4:	/* Skip this room, go directly to 'room' */
	case 5:	/* Ungoto this room, go directly to 'room' */
	case 6:	/* Go directly to room */
		if (type != 6l)
			(void) printf (" %s goto ", RoomNameString (currentRoom));
		GetString (roomName, _lowerCase);
		room = FindRoom (_findPartial, roomName);
		if (room == -1) {
			(void) printf ("No room matching '%s'.\n", roomName);
		} else {
			if (rooms[room].inviteOnly != rooms[room].invited)
				(void) printf ("No room matching '%s'.\n", roomName);
			else {
				if (type != 5l)
					MarkRead ();
				Goto (room);
			}
		}
		break;
	}
}

GotoNew ()

{
	int floor, room, done;

	floor = currentFloor;
	done = _false;
	do {
		room = currentRoom;
		do {
			/* if floor and room room are not forgotten, and there are new articles, and we're on the
			   right floor (or not in floor mode) then we're done */
			if ((!rooms[room].forgotten && !floors[rooms[room].floor].forgotten)
			  && rooms[room].newArticles
			  && (!floorMode || rooms[room].floor == floor)
			  && (rooms[room].inviteOnly == rooms[room].invited))
				done = _true;
			else
				room ++;
			if (rooms[room].name == NULL)
				room = 0;
		} while (!done && room != currentRoom);
		if (!done && floorMode) {
			do {
				floor ++;
				if (floors[floor].name == NULL)
					floor = 0;
			} while (floors[floor].forgotten);
		}
	} while (!done && floorMode && floor != currentFloor);
	if (room == currentRoom)
		room = 0;
	Goto (room);
}

Goto (room)

int room;

{
	int newArticles, i;
	int previousFloor;

	if (rooms[room].inviteOnly != rooms[room].invited)
		return;

	previousRoom = currentRoom;
	previousFloor = currentFloor;
	currentRoom = room;
	currentFloor = rooms[currentRoom].floor;
	rooms[currentRoom].forgotten = _false;
	floors[currentFloor].forgotten = _false;
	LoadArticles ();
	MarkBits (rooms[currentRoom].articlesRead);
	(void) printf ("%s", RoomNameString (currentRoom));
	if (rooms[currentRoom].description && !(strlen (rooms[currentRoom].description) < _aliasLength))
		(void) printf ("\t%s", rooms[currentRoom].description);
	(void) printf ("\n");
	if (floorMode && (previousFloor != currentFloor)) {
/*		(void) printf ("[%s", floors[currentFloor].name);
		if (floors[currentFloor].description) {
			(void) printf ("%s", floors[currentFloor].description);
		}
		(void) printf ("]\n");
*/
		if (floors[currentFloor].description)
			(void) printf ("[%s]\n", floors[currentFloor].description);
		else
			(void) printf ("[%s]\n", floors[currentFloor].name);
	}
	(void) printf ("  %d messages\n", numArticles);

	for (newArticles = i = 0; i < numArticles; i ++)
		if (!articles[i].read)
			newArticles ++;
	if (newArticles != 0) {
		(void) printf ("  %d new\n", newArticles);
	}
}

static int
IntCompare ();

void
LoadArticles ()
/* load article numbers into articles, and count in numArticles. */

{
	void qsort ();
	long atol ();
	static int maxArticles = 0;
	DIR *tempDir;
	struct dirent *dirEnt;
	int n;

	numArticles = 0;
	if ((tempDir = opendir (RoomPath (rooms[currentRoom].name))) == NULL) {
		return;
	}
	/* slow way to count number of articles in dir */
	for (n = 0; (dirEnt = readdir (tempDir)) != NULL;) {
		if ((dirEnt -> d_name[0] >= '0') && (dirEnt -> d_name[0] <= '9')) {
			if (n == maxArticles) {
				/* XXX Test numArticles overflow */
				maxArticles += 50;
				if (articles == NULL)
					articles = (Article *) malloc ((size_t) maxArticles * sizeof (Article));
				else
					articles = (Article *) realloc ((malloc_t) articles, (size_t) maxArticles * sizeof (Article));
				if (articles == NULL) {
					(void) sprintf (errorString, "Out of memory (too many articles, %d, in %s).", maxArticles, rooms[currentRoom].name);
					ErrorLog ("CountArticles", _fatalError, errorString);
				}
			}
			articles[n].number = atol (dirEnt -> d_name);
			articles[n].checked = 0;
			articles[n++].read = 0;
		}
	}
	(void)closedir (tempDir);
	numArticles = n;
	if (n == 0)
		return;
	qsort ((void *) articles, (size_t) n, sizeof (*articles), IntCompare);
	return;
}

static int
IntCompare (item1, item2)

long *item1, *item2;

{
	/* XXX Better way to return an int indicator for the two longs? */
	long result;

	result = *item1 - *item2;
	if (result != 0l)
		return (result < 0l ? -1 : 1);
	else
		return (0);
}

char *
RoomPath (string)

char *string;

{
	/* XXX Can we allocate this? */
	static char returnString[_bufSize];
	register char *ptr;

	if (rooms[currentRoom].type == _mailType) {
		(void) sprintf (returnString, "%s/mail/%s", homeDir, string);
	} else {
		(void) sprintf (returnString, "%s/%s", _spoolDir, string);
	}
	for (ptr = returnString; *ptr; ptr ++) {
		if (*ptr == '.')
			*ptr = '/';
	}
	return (returnString);
}

/* articles on line are already marked, now we'll
   clear the ones we've read, leaving unread articles marked */
MarkBits (string)

char *string;

{
	register int i;
	long next, atol ();
	int dashed;

	i = 0;
	dashed = 0;
	while (*string && i < numArticles) {
		while (*string && !isdigit (*string))
			string++;
		if (!*string)
			break;
		next = atol (string);
		while (i < numArticles && articles[i].number < next)
			articles[i++].read = dashed;
		if (i < numArticles && articles[i].number == next);
			articles[i++].read = 1;
		while (!ispunct (*string) && *string)
			string++;
		dashed = *string == '-';
	}
}
