/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* Some simple RCS related procedures. */

#include <stdio.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <strings.h>
#include "extfil.h"

#include <stdio.h>
#if SYS_V == 1
#include <string.h>
#else
#include <strings.h>
#endif

#if SYS_V == 1
#define index  strchr
#define rindex strrchr
#endif

#define TRUE 1
#define FALSE 0

/*
 * RcsClean
 * ********
 * Execute rcsclean on a file.
 * First this program will check whether the file itself exists in
 * the first place or not. If the file does NOT exist, it will
 * return TRUE ("nothing to clean").
 * If the file does exist, an rcsclean with the provided revision number
 * (if it is not empty) will be attempted. If it succeeded, the function will return TRUE.
 * Next an rcsclean with NO version number will be attempted. If it
 * succeeds, the function will return TRUE.
 * Otherwise it will return false.
 *
 * fn: file name of this file.
 * rev: revision number provided.
 * RET: TRUE if the file does not exist, or the rcsclean operation
 *      worked and the file was removed.
 *      FALSE if the file could not be rcsclean-ed.
 */
int
RcsClean(fn, rev)
     char *fn;
     char *rev;
{
  struct stat f_status; /* File status buffer. */
  char system_call[256];
  char *p;

  if (Strlen(fn) == 0)
    Fatal ("RcsClean(): empty file name");

  /* Check whether file exists. */
  if (access(fn, R_OK))
    return (TRUE); /* File does not exist. */

  /* Get status of the file, error if it's not a regular file. */
  if (stat(fn, &f_status))
    Fatal2 ("RcsClean(): cannot stat file \"%s\".", fn);
  if (! f_status.st_mode & S_IFREG)
    Fatal2 ("RcsClean(): file \"%s\" is not a regular file (stat call).", fn);

  /* Now try rcsclean, first with the revision number which was provided,
     if there is such. */
  if (Strlen(rev) != 0) {
    sprintf (system_call, "rcsclean -r%s %s", rev, fn);
    (void) system  (system_call); /* Ignore status returned. */
    if (access(fn, R_OK))
      return (TRUE); /* File went away. */
  }

  /* Now try a bare rcsclean. */
  sprintf (system_call, "rcsclean %s", fn);
  (void) system  (system_call); /* Ignore status returned. */
  if (access(fn, R_OK))
    return (TRUE); /* File went away. */
  
  /* Bad luck: file still around. */
  return (FALSE);
}

/*
 * FileHasRcsFile
 * **************
 * Check whether file fn has an RCS file associated with it.
 *
 * fn: file name (an ordinary source code file name).
 * RET: TRUE if there is a file fn,v or RCS/fn,v.
 */
int
FileHasRcsFile(fn)
     char *fn;
{
  char buffer[256];
  char *p;

  if (Strlen(fn) == 0)
    Fatal ("FileHasRcsFile(): file name empty.");
  sprintf (buffer, "%s,v", fn);
  if (! access(buffer, R_OK))
    return (TRUE);

  /* Now we have to test the RCS subdirectory for an RCS file. */
  if ((p=rindex(buffer, '/')) == 0)
    /* No directory part of this file name! */
    sprintf(buffer, "RCS/%s,v", fn);
  else {
    /* Take the directory part, put an "RCS/" inbetween and then
       append the remainder of the file name. */
    strncpy(buffer, fn, fn - p + 1); /* Also copies the / */
    strcat (buffer, "RCS/");
    strcat (buffer, p+1); /* Copy remainder of file name. */
    strcat (buffer, ",v");
  }
  fprintf (stderr, "FileHasRcsFile(): RCS file name: \"%s\"\n", buffer);
  return (!access(buffer, R_OK));
}
