/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* Some simple directory manipulation functions. */

#include <stdio.h>
#include <sys/param.h>

/* External declarations. */
extern char *getwd();
extern char *StrcpyAlloc();
extern int FileBusinessVerbose;

/* Forward declarations. */
char * GetWd();

/*
 * ChDir
 * *****
 * Change directory to the specified directory.
 * Generate a fatal error if you can't.
 *
 * dir_name: directory to change to.
 */
void
ChDir(dir_name)
     char *dir_name;
{
  char tmp[MAXPATHLEN];
  char *p;

  if (Strlen(dir_name) == 0)
    Fatal ("ChDir(): directory name is empty.");
  if (chdir(dir_name))
    Fatal2 ("ChDir(): cannot change directory to \"%s\".", dir_name);
  if (FileBusinessVerbose)
    fprintf (stderr, "ChDir(): path name after chdir: \"%s\"\n", GetWd());
}

/*
 * GetWd
 * *****
 * Return a pointer to a string containing the current
 * directory. Generate a fatal error if this is not possible.
 */
char *
GetWd()
{
  char tmp[MAXPATHLEN];
  char *p;

  if ((p=getwd(tmp)) == NULL)
    Fatal ("GetWd(): cannot getwd().");
  return(StrcpyAlloc(p));
}

/* The absolute path name of the home directory of the user. */
char *HomeDirAbsolutePathName = NULL;

/*
 * GetWdCshSyntax
 * ***************
 * Return a pointer to a string containing the current
 * directory. In this case "~" is used at the beginning
 * in case the file name is specified with respect to the home directory
 * of the user who executes the program.
 */
char *
GetWdCshSyntax()
{
  char tmp[MAXPATHLEN];
  char *p;

  if (Strlen(HomeDirAbsolutePathName) == 0) {
    /* The absolute path name is not yet specified! */
  }

  p = GetWd();

  if (strncmp(p, HomeDirAbsolutePathName, strlen(HomeDirAbsolutePathName)) == 0) {
    ReplaceASubString(p, strlen(HomeDirAbsolutePathName), "~");
  }

  return(StrcpyAlloc(p));
}

/*
 * RmDir
 * ******
 * Remove a directory and generate a fatal error if it failed.
 *
 * dirn: file name of file to RmDir.
 */
void
RmDir(dirn)
    char *dirn;
{  
  if (Strlen(dirn) == 0)
    Fatal ("RmDir(): empty directory name.");
  if (FileBusinessVerbose)
    fprintf (stderr, "RmDir(): directory \"%s\"\n", dirn);
  if (rmdir(dirn) == -1)
    Fatal2 ("RmDir(): RmDir of \"%s\" failed.", dirn);
}
