/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


#include <stdio.h>
#include "defs.h"
#include "dvi-com.h"
#include "units.h"
#include "dvitps.h"
#include "extfil.h"
#include "fontd.h"
#include "emit.h"

extern EX_FILES ExDvi;
extern EX_FILES Ex_PsOutput;
extern int PrevPP;
extern int Count[10];
extern int Verbose;
extern int NoEject;
extern int CurFontNumber;
extern FE_P Fonts[MAX_FONTS];
extern int DviFileFontMap[MAX_FONTS];
extern int CountPages;
extern void ChangeFontAndSetup();
extern int PssColumn;
extern PXLU AccDeltaH;
extern PXLU AccDeltaV;
extern int StderrOutputPageNumber;
extern int StderrOutputPageNumberMod;
extern int ConformingDocument;
extern int FromPageNumber;
extern int ToPageNumber;
extern int command_offset;

/* Positions, all in DVIU. */
DVIU h; /* Current horizontal position according to dvi file. */
DVIU v;
DVIU hh = 0; /* Position on device: dividing by factor gives. */
DVIU vv = 0; /* Integral pixel position. */

extern char SpecialString[SPECIAL_LENGTH];
extern int HConv;
extern int VConv;

/*
 * Pass1
 * *****
 * Execute pass 1 of the driver. The dvi file is NOT repositioned
 * after the page is processed.  More precisely it leaves off at the
 * EOP of the current page.
 *
 * Sets as side effect PrevPP so if necessary the previous file can be accessed.
 *
 * RET: PASS1_POSTAMBLE_FOUND (the PostAmble was found)
 *      PASS1_EOP
 */
int
Pass1()
{
  int command; /* Command read in from dvi file. */
  int val, val2; /* Parameters of dvi commands. */

  int i; /* For loops. */
  int k;

  DVIU w; /* Horizontal spacing from dvi file. */
  DVIU x;
  DVIU y; /* Vertical spacing from dvi file. */
  DVIU z;

  /* Stack pointer and stack of the dvi file. */
  int sp;
  struct dvi_file_stack {
    DVIU s_h, s_v, s_hh, s_vv, s_w, s_x, s_y, s_z;};
  struct dvi_file_stack stack[MAX_STACK_SIZE];

  PXLU diff; /* Temporary diff values. */
  int output_this_page; /* Should this page be written out? */

  PssChangeFormat(PSS_FOR_CONT);
  SwitchPsOutput(2);

  /* Get the very first command in the dvi file now. */
  switch (command = NoSignExtend (EX_FP(ExDvi), 1)) {
    case DVI_POST:
      return (PASS1_POSTAMBLE_FOUND);
    case DVI_BOP:
      break; /* ok */
    default:
      Fatal2 ("Pass1(): illegal command %d", command);
      break;
  }
  FExSeek (&ExDvi, -1, FSEEK_REL); /* Position backwards to read DVI_BOP again. */

  /* Big loop to read in the commands from the dvi file.  */
  while (TRUE) {
    command_offset = FExTell(&ExDvi);
    command = NoSignExtend (EX_FP(ExDvi), 1);
#ifdef DEBUG
    fprintf (stderr, "%% Pass1: command = %d\n", command);
#endif
    switch (command)    {
      /* Set a character, moves the reference point */
      case DVI_SET1: case DVI_SET2: case DVI_SET3: case DVI_SET4:
        val = NoSignExtend (EX_FP(ExDvi), command - DVI_SET1+1);
	if (output_this_page)
	  SetPutChar (val, 0);
	break;

      /* Put (print) a character without moving the reference point */
      case DVI_PUT1: case DVI_PUT2: case DVI_PUT3: case DVI_PUT4:
	if ((val = NoSignExtend(EX_FP(ExDvi), command - DVI_PUT1+1)) >= 256)
	  Fatal ("Pass1(): PUT error.");
	if (! output_this_page)
	  break;
	PssFlushStringAndPos();
	SetPutChar (val, 1);
	break;
      
      case DVI_PUT_RULE:
      case DVI_SET_RULE:
	val = NoSignExtend (EX_FP(ExDvi), 4);
	val2= NoSignExtend (EX_FP(ExDvi), 4);
	if (output_this_page)
	  SetRule(val, val2, command);
	break;

      case DVI_NOP:
	break;

      case DVI_BOP:
	for (i=0; i<=9; i++)
	  Count[i] = NoSignExtend(EX_FP(ExDvi), 4);
	PrevPP = NoSignExtend(EX_FP(ExDvi), 4);
	output_this_page = (FromPageNumber <= Count[0] &&
			    Count[0] <= ToPageNumber);
	if (! output_this_page)
	  break;
      
	h = v = w = x = y = z = 0;
	hh = vv = 0;
	sp = 0;
	PssSendInit(); /* Various initializations before positions and
			  strings can be sent out to the printer */
	/* For the following check the arguments of @bop1 in texpre.pro */
	PssSendInteger (0);
	PssSendInteger (0);
	PssSendInteger (Count[0]);
	PssSendCommand ("@bop1");

	switch (Verbose) {
          case V_QUIET:
	    break;
          case V_SOME:
	    if (ConformingDocument) {
	      fprintf (stderr, "[%d", Count[0]);
	      StderrOutputPageNumber++;
	    }
	    break;
	  case V_A_LOT:
	    fprintf(stderr, "[P-1: %d", Count[0]);
	    fflush(stderr);
	    StderrOutputPageNumber++;
	    break;
	  }
	break;

      case DVI_EOP:
	if (! output_this_page)
	  return (PASS1_EOP_FOUND);
	PssFlushStringAndPos();
	EMIT_NEW_LINE;
	PssSendInteger (Count[0]);
	PssSendCommand ("@eop1");
	EMIT_NEW_LINE;
	switch (Verbose) {
          case V_QUIET:
	    break;
	  case V_SOME:
	    fprintf(stderr, "] ");
	    if (StderrOutputPageNumber >= StderrOutputPageNumberMod) {
	      fprintf(stderr, "\n");
	      StderrOutputPageNumber = 0;
	    }
	    fflush(stderr);
	    break;
	  case V_A_LOT:
	    fprintf(stderr, "] ");
	    if (StderrOutputPageNumber >= StderrOutputPageNumberMod) {
	      fprintf(stderr, "\n");
	      StderrOutputPageNumber = 0;
	    }
	    fflush(stderr);
	    break;
	  }
	return (PASS1_EOP_FOUND);
	
      case DVI_PUSH:
	if (! output_this_page)
	  break;
#ifdef POS_CHECK
	fprintf (stderr, "%% PUSH[%d] h= %6d, hh= %6d, diff(hh-h)= %6d\n",
		 sp, h, hh, hh-h);
#endif
	if (sp >= MAX_STACK_SIZE)
	  Fatal("Pass1(): stack overflow.");
	stack[sp].s_h = h;
	stack[sp].s_v = v;
	stack[sp].s_hh= hh;
	stack[sp].s_vv= vv;
	stack[sp].s_w = w;
	stack[sp].s_x = x;
	stack[sp].s_y = y;
	stack[sp].s_z = z;
	sp++;
	break;

      case DVI_POP:
	if (! output_this_page)
	  break;
#ifdef POS_CHECK
	fprintf (stderr, "%% POP\n");
#endif
	--sp;
	if (sp < 0)
	  Fatal("Pass1(): stack underflow.");
	h = stack[sp].s_h;
	v = stack[sp].s_v;
	w = stack[sp].s_w;
	x = stack[sp].s_x;
	y = stack[sp].s_y;
	z = stack[sp].s_z;
	/* Move the printer back. */
	diff = PixRound(stack[sp].s_hh - hh, HConv);
#ifdef POS_CHECK
	fprintf (stderr, "%% POP[%d]: h = %7d, hh = %7d, stack-hh:\
 %7d, diff = %d[pxl]\n",
				  sp,      h,         hh,
 stack[sp].s_hh, diff);
#endif
	AccDeltaH += diff;
	hh += HConv * diff;
#ifdef POS_CHECK
	fprintf (stderr, "%% POP[%d]: Updated hh = %7d,diff = %6d[%3d pxl]\n",
	                          sp,               hh,        h-hh,
							 PixRound(h-hh, HConv));
#endif
	diff = PixRound(stack[sp].s_vv - vv, VConv);
	AccDeltaV += diff;
	vv += VConv * diff;
	break;

      case DVI_RIGHT1: case DVI_RIGHT2: case DVI_RIGHT3: case DVI_RIGHT4:
	val = SignExtend(EX_FP(ExDvi), command - DVI_RIGHT1+1);
	if (output_this_page)
	  MoveH(val);
	break;

      case DVI_W0:
	if (output_this_page)
	  MoveH(w);
	break;

      case DVI_W1: case DVI_W2: case DVI_W3: case DVI_W4:
	w = SignExtend(EX_FP(ExDvi), command - DVI_W1+1);
	if (output_this_page)
	  MoveH(w);
	break;

      case DVI_X0:
	if (output_this_page)
	  MoveH(x);
	break;
	
      case DVI_X1: case DVI_X2: case DVI_X3: case DVI_X4:
        x = SignExtend(EX_FP(ExDvi), command - DVI_X1+1);
	if (output_this_page)
	  MoveH(x);
	break;
	
      case DVI_DOWN1: case DVI_DOWN2: case DVI_DOWN3: case DVI_DOWN4:
	val = SignExtend(EX_FP(ExDvi), command - DVI_DOWN1+1);
	if (output_this_page)
	  MoveV(val);
	break;

      case DVI_Y0:
	if (output_this_page)
	  MoveV(y);
	break;

      case DVI_Y1: case DVI_Y2: case DVI_Y3: case DVI_Y4:
	y = SignExtend(EX_FP(ExDvi), command - DVI_Y1+1);
	if (output_this_page)
	  MoveV(y);
	break;

      case DVI_Z0:
	if (output_this_page)
	  MoveV(z);
	break;

      case DVI_Z1: case DVI_Z2: case DVI_Z3: case DVI_Z4:
	z = SignExtend(EX_FP(ExDvi), command - DVI_Z1+1);
	if (output_this_page)
	  MoveV(z);
	break;

      /* Change font instructions. */
      case DVI_FNT1: case DVI_FNT2: case DVI_FNT3: case DVI_FNT4:
	k = NoSignExtend(EX_FP(ExDvi), command - DVI_FNT1+1);
	if (! output_this_page)
	  break;
	CurFontNumber = DviFileFontMap[k];
	ChangeFontAndSetup();
	break;

      /* \special{...} */
      case DVI_XXX1: case DVI_XXX2: case DVI_XXX3: case DVI_XXX4:
	DoSpecial (command, 1, output_this_page);
	break;

      /* Font definitions: already taken care of (postamble). */
      case DVI_FNT_DEF1: case DVI_FNT_DEF2: case DVI_FNT_DEF3: case DVI_FNT_DEF4:
	k = NoSignExtend(EX_FP(ExDvi), command - DVI_FNT_DEF1+1);
	k = DviFileFontMap[k]; 
	FExSeek (&ExDvi, Fonts[k]->f_fnt_def_length, FSEEK_REL);
	break;
	
      case DVI_PRE:
      case DVI_POST:
      case DVI_POST_POST:
	Fatal("Pass1(): DVI_PRE/POST/POST_POST");
	break;

      /*
       * Three remaining possibilities of what command is in the dvi file:
       *	(a) set a character
       *	(b) change font
       *	(c) illegal code
       */
      default:
	if (!output_this_page)
	  break;
	if (command >= DVI_SETC_000 && command <= DVI_SETC_127) {
	  SetString(command);
	  break;
	}
	if (command >= DVI_FONT_00 && command <= DVI_FONT_63)	{
	  k = command - DVI_FONT_00;
	  CurFontNumber = DviFileFontMap[k];
	  ChangeFontAndSetup();
	  break;
	}
	Fatal2 ("Pass1(): command %d undefined", command);
	break;
      } /* switch */
  } /* while */
}
