/*
 *	Author:  Alan Rollow, EIS/CXO, Digital Equipment Corp.
 *	File:	 swap.c
 *	Date:	 8/21/90
 *	Version: 1.15
 *
 *	swap.c - Collect and display data on swap space usage.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)swap.c	1.15 (monitor) 8/21/90" ;
#endif

/*
 * Modification history.
 *
 * Jan. 15, 1989 -- arr
 *
 *	Check the namelist in the setup function.
 *
 * Jan. 16, 1989 -- arr
 *
 *	Added code for the display function.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Use the function version of pgtok() and btok() instead
 *	of the macros.
 *
 * Dec. 25, 1989 -- arr
 *
 *	Setup private namelist.
 *
 * Dec. 26, 1989 -- arr
 *
 *	Add code to handle V4.0 differences.  This may all get backed out
 *	by Engineering, but works at the moment.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vm.h>

#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/param.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/text.h>
#include <sys/map.h>
#include <sys/conf.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include <net/if.h>
#include <netinet/in.h>

#include <machine/pte.h>

#include "monitor.h"
#include "options.h"
#include "include.h"
#include "extern.h"

/*
 *	Functions that don't return (int).
 */
char	*calloc() ;
void	free(), nlist() ;
long	rmalloc() ;

/*
 *	Things that live elsewhere, but are not of general
 *	interest.
 */
extern	int	lines ;
extern	WINDOW	*wp ;

/*
 *	Architecture dependent macros.
 */
#ifndef	V4_ULTRIX
#	ifdef	vax
#		define	MON_IPC_MASK	(IPC_ALLOC)
#		define	CLRND(x) 	(((x) + (sys_clsize-1)) & ~(sys_clsize-1))
#		define	STACK_MACRO(p)	\
			ctod(CLRND(UPAGES + (CLRND(ctopt((p)->p_ssize + HIGHPAGES)))))
#		define	DATA_MACRO(p) 	\
			ctod(CLRND(CLRND(ctopt((p)->p_tsize+(p)->p_dsize)) - \
			(p)->p_tsize/NPTEPG))
#	elif	mips
#		define	MON_IPC_MASK	(IPC_ALLOC|IPC_SYSTEM)
#		define	CLRND(x) 	(x)
#		define	STACK_MACRO(p)	ctod(CLRND(UPAGES) + CLRND(ctopt((p)->p_ssize)))
#		define	DATA_MACRO(p)	ctod(CLRND(ctopt((p)->p_dsize)))
#	endif
#endif

/*
 *	Passed to the error message routines.
 */
static	char	*module = "swap" ;

/*
 *	Things only used locally.  These will be initialized
 *	in setup_swap().  Some of them aren't needed in V4 so
 *	we'll #ifdef around them.
 */
#ifdef	V4_ULTRIX
static	struct swapu_t	swapu ;
#else
static	struct proc	*proc,    *p_addr ;
static	struct text	*xtext,   *t_addr ;
static	struct smem 	*smem;
static	struct sminfo 	sminfo;

static	int	nproc ;
static	int	ntext ;
static	int	start, end ;	  /* loop counters for the display function */
static	int	sys_clsize ;
#endif

/*
 *	These are needed for all versions.
 */
static	struct map	*swapmap, *smp_addr ;
static	struct swdevt	*swdevt,  *swp_addr ;

static	int	nswapmap ;
static	int	nswdev ;
static	int	dmmax ;		 /* this will double as swapfrag from V4.0 */
static	int	swap_ticks = 0 ;

#ifdef	V4_ULTRIX
static	int	swapfrag ;	/* this will be set when dmmax is */
#else
static	int	dmmin ;		/* this won't be used for pre-V4.0 */
#endif

/*
 *	After little consideration, it looks like the easiest
 *	way to handle the namelist differents between V4 and
 *	pre-V4 are two different name lists and sets of constants.
 *
 *	This has the advantage that the V4 namelist is smaller
 *	and therefore nlist(3) will be quicker.  It also makes
 *	checking for all the required namelist enties easier
 *	because we can just walk down the array.
 */
#ifdef	V4_ULTRIX
/*
 *	This is the local namelist for V4.0.
 */
struct nlist swap_nm[] = {
	{ "_swapmap" },		/* SWAP_NM_SWAPMAP */
	{ "_nswapmap" },	/* SWAP_NM_NSWAPMAP */
	{ "_swdevt" },		/* SWAP_NM_SWDEVT */
	{ "_nswdev" },		/* SWAP_NM_NSWDEV */
	{ "_swapfrag" },	/* SWAP_NM_SWAPFRAG */
	{ "_swapu" },		/* SWAP_NM_SWAPU */
	{ 0 }
};

/*
 *	Constants for local namelist.
 */
#define	SWAP_NM_SWAPMAP 	(0)
#define	SWAP_NM_NSWAPMAP 	(1)
#define	SWAP_NM_SWDEVT 		(2)
#define	SWAP_NM_NSWDEV 		(3)
#define	SWAP_NM_SWAPFRAG	(4)
#define	SWAP_NM_SWAPU		(5)

#else
/*
 *	This is the pre-V4.0 namelist.
 */
struct nlist swap_nm[] = {
	{ "_dmmin" },		/* SWAP_NM_DMMIN */
	{ "_dmmax" },		/* SWAP_NM_DMMAX */
	{ "_proc" },		/* SWAP_NM_PROC */
	{ "_nproc" },		/* SWAP_NM_NPROC */
	{ "_text" },		/* SWAP_NM_TEXT */
	{ "_ntext" },		/* SWAP_NM_NTEXT */
	{ "_swapmap" },		/* SWAP_NM_SWAPMAP */
	{ "_nswapmap" },	/* SWAP_NM_NSWAPMAP */
	{ "_swdevt" },		/* SWAP_NM_SWDEVT */
	{ "_nswdev" },		/* SWAP_NM_NSWDEV */
	{ "_smem" },		/* SWAP_NM_SMEM */
	{ "_sminfo" },		/* SWAP_NM_SMINFO */
	{ 0 }
};

/*
 *	Constants for local namelist.
 */
#define	SWAP_NM_DMMIN 		(0)
#define	SWAP_NM_DMMAX 		(1)
#define	SWAP_NM_PROC 		(2)
#define	SWAP_NM_NPROC 		(3)
#define	SWAP_NM_TEXT 		(4)
#define	SWAP_NM_NTEXT 		(5)
#define	SWAP_NM_SWAPMAP 	(6)
#define	SWAP_NM_NSWAPMAP 	(7)
#define	SWAP_NM_SWDEVT 		(8)
#define	SWAP_NM_NSWDEV 		(9)
#define	SWAP_NM_SMEM 		(10)
#define	SWAP_NM_SMINFO 		(11)

#endif

/*
 *	Free the space allocated tables for the swap information.
 */
free_swap()
{
	if( swapmap )
		free((char *)swapmap) ;

	if( swdevt )
		free((char *)swdevt) ;

#ifndef	V4_ULTRIX
	if( proc )
		free((char *)proc) ;

	if( xtext )
		free((char *)xtext) ;
#endif
}

/*
 *	Allocate space for the various data structures we're
 *	to need.  Reading them as a block is VERY much faster
 *	than one at time.  If Ultrix ever makes these data
 *	structures lists instead of arrays, then the code will
 *	have to change.  Hopefully by then there will be an
 *	ioctl(2) or getsysinfo(2) call to get information.
 */
setup_swap(op)
OPTION	*op ;
{
	register i ;

	(void)nlist(op->opt_kernel, swap_nm) ;

        /*
         *      This just walks through the array to see if all the
	 *	needed symbols are there.
         */
        for(i = 0; i < sizeof(swap_nm)/sizeof(swap_nm[0]) - 1; i++) {
		if( swap_nm[i].n_value == 0 ) {
			info("Swap space data is not available: %s.\n",
				swap_nm[i].n_name, module) ;
			op->opt_swap = 0 ;
		}
	}

	/*
	 *	Allocate space for the swap maps.
	 */
	nswapmap = get_word((long)swap_nm[SWAP_NM_NSWAPMAP].n_value);
	smp_addr = (struct map *)get_word((long)swap_nm[SWAP_NM_SWAPMAP].n_value) ;

	if((swapmap = (struct map *)calloc((unsigned)nswapmap,
	    sizeof(struct map))) == (struct map *) NULL) {
		fatal("Can't allocate space for swap map: %s.\n", module) ;
	}

	/*
	 *	Allocate space for the swap device table.
	 */
	nswdev   = get_word((long)swap_nm[SWAP_NM_NSWDEV].n_value);
	swp_addr = (struct swdevt *)swap_nm[SWAP_NM_SWDEVT].n_value ;

	if((swdevt = (struct swdevt *)calloc((unsigned)nswdev,
	    sizeof (struct swdevt))) == (struct swdevt *) NULL) {
		fatal("Can't allocate space for swapdev table: %s.\n", module) ;
	}

	/*
	 *	The version specific setup for V4 is very simple.
	 */
#ifdef	V4_ULTRIX
	/*
	 *	Swapfrag is the interesting size in V4.0, but to
	 *	make some of the other code simpiler we'll overload
	 *	dmmax to hold swapfrag.
	 */
	dmmax = swapfrag = get_word(swap_nm[SWAP_NM_SWAPFRAG].n_value) ;

	mon_swap.mon_dmmax = swapfrag ;
	mon_swap.mon_dmmin = 0 ;
#else
	/*
	 *	Initialize dmmin and dmmax, if there are there.  The
	 *	namelist check will ensure they are there if they
	 *	need to be.  The space for dmmax will used for swapfrag
	 *	for V4.0.
	 */
	dmmin = mon_swap.mon_dmmin = get_word((long)swap_nm[SWAP_NM_DMMIN].n_value) ;
	dmmax = mon_swap.mon_dmmax = get_word((long)swap_nm[SWAP_NM_DMMAX].n_value) ;
	
	/*
	 *	The system CLICK size, used by a macro later on.
	 *
	 *	512 is clicks, not pages
	 */
	sys_clsize = getpagesize() / 512;
	/*
	 *	Allocate space for the proc table for future use.
	 */
	nproc  = get_word((long)swap_nm[SWAP_NM_NPROC].n_value);
	p_addr = (struct proc *)get_word((long)swap_nm[SWAP_NM_PROC].n_value) ;

	if((proc = (struct proc *)calloc((unsigned)nproc,
	    sizeof(struct proc))) == (struct proc *) NULL){
		fatal("Can't allocate space for proc table: %s.\n", module) ;
	}

	/*
	 *	Allocate space for the text table for future use.
	 */
	ntext  = get_word((long)swap_nm[SWAP_NM_NTEXT].n_value);
	t_addr = (struct text *)get_word((long)swap_nm[SWAP_NM_TEXT].n_value) ;

	if((xtext = (struct text *)calloc((unsigned)ntext,
	    sizeof(struct text))) == (struct text *) NULL) {
		fatal("Can't allocate space for text table: %s.\n", module) ;
	}

	/*
	 *	Allocate space for the shared memory info.
	 */
	readk((long)swap_nm[SWAP_NM_SMINFO].n_value, (char *)&sminfo,
		sizeof(struct sminfo)) ;

	if((smem = (struct smem *)calloc((unsigned)sminfo.smmni,
	    sizeof(struct smem))) == (struct smem *) NULL) {
		fatal("Can't allocate space for shared memory table: %s.\n", module) ;
	}
#endif
}

/*
 *	Fill a mon_swap structure.  This is closely derived from
 *	the source of pstat.c.
 */
collect_swap()
{
	register leftover ;
	register struct mapent *me;
	register struct swdevt *sw;
#ifndef	V4_ULTRIX
	register struct text   *xp;
	register struct proc   *pp;
	register struct smem   *sp ;
#endif

	/*
	 *	Read the swap device table.
	 */
	readk((long)swp_addr, (char *)swdevt, nswdev * sizeof(struct swdevt)) ;

	/*
	 *	Read the swap maps.
	 */
	readk((long)smp_addr, (char *)swapmap, nswapmap * sizeof(struct map)) ;

	swapmap->m_name = "swap";
	swapmap->m_limit = (struct mapent *)&swapmap[nswapmap];

#ifdef	V4_ULTRIX
	/*
	 *      Read the swap usage structure.
	 */
	readk((long)swap_nm[SWAP_NM_SWAPU].n_value, (caddr_t)&swapu,
		sizeof(struct swapu_t)) ;
#else
	/*
	 *	Read the proc table.
	 */
	readk((long)p_addr, (char *)proc, nproc * sizeof(struct proc)) ;

	/*
	 *	Read the text table.
	 */
	readk((long)t_addr, (char *)xtext, ntext * sizeof(struct text)) ;

	/*
	 *	Read the shared memory stuff.
	 */
	readk((long)swap_nm[SWAP_NM_SMEM].n_value, (char *)smem,
		sminfo.smmni * sizeof(struct smem)) ;
#endif

	/*
	 *	Initialize the members of the mon_swap structure.
	 */
	mon_swap.mon_total  = 0 ;
	mon_swap.mon_free   = 0 ;
	mon_swap.mon_text   = 0 ;
	mon_swap.mon_smem   = 0 ;
	mon_swap.mon_wasted = 0 ;

	/*
	 *	Add up the amount of swap space.
	 */
	for (sw = swdevt; sw < &swdevt[nswdev]; sw++)
		mon_swap.mon_total += sw->sw_nblks;

	/*
	 *	This is where the two different version of counting
	 *	swap swap space get ugly.  V4 provides a data structure
	 *	where the kernel keeps track of this information.  Before
	 *	that we had to wander through all sorts of other kernel
	 *	structure to count it up.
	 */
#ifdef	V4_ULTRIX

	/*
	 *	The subtractions are because V4 appears to reserve
	 *	two CLICKS.  Doing this makes the numbers match with
	 *	those of pstat(8).
	 */
	mon_swap.mon_total = mon_swap.mon_total - ctod(CLSIZE) - ctod(CLSIZE) ;

	mon_swap.mon_text   = swapu.txt ;
	mon_swap.mon_smem   = swapu.smem ;
	mon_swap.mon_used   = swapu.total_used ;
	mon_swap.mon_wasted = swapu.wasted ;
	mon_swap.mon_free   = mon_swap.mon_total - mon_swap.mon_used ;
#else
	/*
	 *	Count up the amount of free space.
	 */
	for (me = (struct mapent *)(swapmap+1);
	    me < (struct mapent *)&swapmap[nswapmap]; me++)
		mon_swap.mon_free += me->m_size;

	/*
	 *	Count up the space allocated to text.
	 */
	for (xp = xtext; xp < &xtext[ntext]; xp++)
		if((xp->x_flag & XFREE) == 0)
			swap_text(&mon_swap, xp) ;

	/*
	 *	Count up the space allocated to shared memory.
	 */
	for (sp = smem; sp < &smem[sminfo.smmni]; sp++) {
		if ((sp->sm_perm.mode & MON_IPC_MASK) == IPC_ALLOC)
			swap_smem(&mon_swap, sp) ;
	}

	mon_swap.mon_used = mon_swap.mon_text + mon_swap.mon_smem ;

	/*
	 *	Walk the proc table.  The order of the checks
	 *	for an interesting process probably matters.
	 *	all the useful works ends up being done in
	 *	the appropriate version of swap_proc().
	 */
	for (pp = proc; pp < &proc[nproc]; pp++) {
		/*
		 *	Ignore used and zombie processes.
		 */
		if( pp->p_stat == 0 || pp->p_stat == SZOMB )
			continue;

		if( pp->p_flag & SSYS )
			continue;

		swap_proc(&mon_swap, pp) ;
	}
#endif

	/*
	 *	This will calculate how the swap space is
	 *	divided up.  There are two versions.
	 */
	swap_available(&mon_swap, swapmap) ;

	/*
	 *	Add up whatever is left over.
	 */
	leftover = 0 ;

	for (me = (struct mapent *)(swapmap+1);
	    me < (struct mapent *)&swapmap[nswapmap]; me++)
		leftover += me->m_size;

	mon_swap.mon_avail[0] = btok(leftover) ;
}

/*
 * Allocate 'size' units from the given
 * map. Return the base of the allocated space.
 * In a map, the addresses are increasing and the
 * list is terminated by a 0 size.
 *
 * Algorithm is first-fit.
 *
 * This routine knows about the interleaving of the swapmap
 * and handles that.
 */
long
rmalloc(mp, size)
register struct map *mp;
register size;
{
	register struct mapent *ep = (struct mapent *)(mp+1);
	register int addr;
	register struct mapent *bufp;
	swblk_t sw_first, rest;

	if (size <= 0 || size > dmmax)
		return (0);
	/*
	 * Search for a piece of the resource map which has enough
	 * free space to accomodate the request.
	 */
	for (bufp = ep; bufp->m_size; bufp++) {
		if (bufp->m_size >= size) {
			/*
			 * If allocating from swapmap,
			 * then have to respect interleaving
			 * boundaries.
			 */
			if (nswdev > 1 &&
			    (sw_first = dmmax - bufp->m_addr%dmmax) < bufp->m_size) {
				if (bufp->m_size - sw_first < size)
					continue;
				addr = bufp->m_addr + sw_first;
				rest = bufp->m_size - sw_first - size;
				bufp->m_size = sw_first;
				if (rest)
					rmfree(mp, rest, addr+size);
				return (addr);
			}
			/*
			 * Allocate from the map.
			 * If there is no space left of the piece
			 * we allocated from, move the rest of
			 * the pieces to the left.
			 */
			addr = bufp->m_addr;
			bufp->m_addr += size;

			if((bufp->m_size -= size) == 0 ) {
				do {
					bufp++;
					(bufp-1)->m_addr = bufp->m_addr;
				} while ((bufp-1)->m_size = bufp->m_size);
			}

			if (addr % CLSIZE)
				return (0);

			return (addr);
		}
	}
	return (0);
}

/*
 * Free the previously allocated space at addr
 * of size units into the specified map.
 * Sort addr into map and combine on
 * one or both ends if possible.
 */
rmfree(mp, size, addr)
struct map *mp;
long	size ;
int	addr;
{
	register struct mapent *firstbp;
	register struct mapent *bufp;
	register int t;

	/*
	 * Both address and size must be
	 * positive, or the protocol has broken down.
	 */
	if (addr <= 0 || size <= 0)
		goto badrmfree;
	/*
	 * Locate the piece of the map which starts after the
	 * returned space (or the end of the map).
	 */
	firstbp = bufp = (struct mapent *)(mp + 1);
	for (; bufp->m_addr <= addr && bufp->m_size != 0; bufp++)
		continue;
	/*
	 * If the piece on the left abuts us,
	 * then we should combine with it.
	 */
	if (bufp > firstbp && (bufp-1)->m_addr+(bufp-1)->m_size >= addr) {
		/*
		 * Check no overlap (internal error).
		 */
		if ((bufp-1)->m_addr+(bufp-1)->m_size > addr)
			goto badrmfree;
		/*
		 * Add into piece on the left by increasing its size.
		 */
		(bufp-1)->m_size += size;
		/*
		 * If the combined piece abuts the piece on
		 * the right now, compress it in also,
		 * by shifting the remaining pieces of the map over.
		 */
		if (bufp->m_addr && addr+size >= bufp->m_addr) {
			if (addr+size > bufp->m_addr)
				goto badrmfree;
			(bufp-1)->m_size += bufp->m_size;
			while (bufp->m_size) {
				bufp++;
				(bufp-1)->m_addr = bufp->m_addr;
				(bufp-1)->m_size = bufp->m_size;
			}
		}
		goto done;
	}
	/*
	 * Don't abut on the left, check for abutting on
	 * the right.
	 */
	if (addr+size >= bufp->m_addr && bufp->m_size) {
		if (addr+size > bufp->m_addr)
			goto badrmfree;
		bufp->m_addr -= size;
		bufp->m_size += size;
		goto done;
	}
	/*
	 * Don't abut at all.  Make a new entry
	 * and check for map overflow.
	 */
	do {
		t = bufp->m_addr;
		bufp->m_addr = addr;
		addr = t;
		t = bufp->m_size;
		bufp->m_size = size;
		bufp++;
	} while (size = t);
	/*
	 * Segment at bufp is to be the delimiter;
	 * If there is not room for it
	 * then the table is too full
	 * and we must discard something.
	 */
	if (bufp+1 > mp->m_limit) {
		/*
		 * Back bufp up to last available segment.
		 * which contains a segment already and must
		 * be made into the delimiter.
		 * Discard second to last entry,
		 * since it is presumably smaller than the last
		 * and move the last entry back one.
		 */
		bufp--;
		bufp[-1] = bufp[0];
		bufp[0].m_size = bufp[0].m_addr = 0;
	}
done:
	return;
badrmfree:
	panic("Manipulation of swapmap failed!\n", "map");
}


/*
 *	The following functions are used to do specialized
 *	counting of swap space usage.  Only swap_available()
 *	is used in both the V4 and pre-V4 versions.
 */
#ifdef	V4_ULTRIX
/*
 *	V4.0 makes this simpler.  The swap space consists of
 *	of swapfrag size pieces.  Those that are free are
 *	available.
 */
swap_available(sp, map)
struct mon_swap *sp ;
struct map *map ;
{
	register i = 0;

        while( rmalloc(map, swapfrag) != 0 )
                i++;

	sp->mon_avail[ffs(swapfrag)-2] = i ;
}
#else
/*
 *	Before V4.0 the swap space consisted of dmmin to dmmax
 *	sized pieces.  It counts them working down from dmmax
 *	to dmmin.
 */
swap_available(sp, map)
register struct mon_swap *sp ;
register struct map *map ;
{
	register i, j ;

	/*
	 *	Find out what sized pieces the swap space is
	 *	available in.  This is done by using the swap
	 *	space algorithm to allocate pieces from DMMAX
	 *	down to DMMIN.
	 */
	for (i = dmmax; i >= dmmin; i /= 2) {
		j = 0;

		while( rmalloc(map, i) != 0 )
			j++;

		sp->mon_avail[ffs(i)-2] = j ;
	}
}

/*
 *	This calculates how much swap space is allocated to text.
 */
swap_text(sp, xp)
register struct mon_swap *sp ;
register struct text *xp ;
{
	sp->mon_text += ctod(xp->x_size);
	if( xp->x_flag & XPAGI )
		sp->mon_text += ctod(CLRND(ctopt(xp->x_size)));
}

/*
 *	This calculates how much swap space is allocated to shared
 *	memory.
 */
swap_smem(sp, sm)
register struct mon_swap *sp ;
register struct smem *sm ;
{
	register smsize ;

	smsize = CLRND(btoc(sm->sm_size));
	sp->mon_smem += ctod(smsize);
	sp->mon_smem += ctod(CLRND(ctopt(smsize)));
}

/*
 *	This calculates how much swap space is allocated to
 *	a process.
 */
swap_proc(sp, p)
register struct mon_swap *sp ;
register struct proc *p ;
{
	register db, sb ;

	/*
	 *	Get the size allocated to data.
	 */
	db = ctod(p->p_dsize), sb = up(db);
	sp->mon_used   += sb;
	sp->mon_wasted += sb - db;

	/*
	 *	Get the size allocated to stack.
	 */
	db = ctod(p->p_ssize), sb = up(db);
	sp->mon_used   += sb;
	sp->mon_wasted += sb - db;

	/*
	 *	Get the size allocated to the swapped _u area.
	 */
	if((p->p_flag & SLOAD) == 0 )
		sp->mon_used += vusize(p);
}

up(size)
register int size ;
{
	register int i, block;

	i = 0;
	block = dmmin;
	while (i < size) {
		i += block;
		if (block < dmmax)
			block *= 2;
	}
	return (i);
}

/*
 * Compute number of pages to be allocated to the u. area
 * and data and stack area page tables, which are stored on the
 * disk immediately after the u. area.
 */
vusize(p)
	register struct proc *p;
{
	register int tsz = p->p_tsize / NPTEPG;

	/*
	 * We do not need page table space on the disk for page
	 * table pages wholly containing text.
	 */
	return (clrnd(UPAGES +
	    clrnd(ctopt(p->p_tsize+p->p_dsize+p->p_ssize+UPAGES)) - tsz));
}
#endif

/*
 *	Constants for placement of SWAP data.
 */
#define	SWAP_LINES	(17)
#define	SWAP_TOTAL_X	(11)
#define	SWAP_TOTAL_Y	(5)
#define	SWAP_X		(15)
#define	SWAP_Y		(3)
#define	SWAP_DATA	(SWAP_Y + 4)
#define	SWAP_AVAIL_Y	(15)
#define	SWAP_AVAIL_X	(0)

/*
 *	Setup the static screen for displaying swap space
 *	information.
 *
 *	ARGSUSED
 */
open_swap(op)
OPTION	*op ;
{
#ifndef	V4_ULTRIX
	register i, printed ;
#endif

	sample_header() ;

	lines = SWAP_LINES ;

	wmove(wp, SWAP_Y, 0) ;

	wprintw(wp, "dmmin:     %10u KB.\n", btok(mon_swap.mon_dmmin)) ;
#ifdef	V4_ULTRIX
	wprintw(wp, "swapfrag:  %10u KB.\n", btok(mon_swap.mon_dmmax)) ;
#else
	wprintw(wp, "dmmax:     %10u KB.\n", btok(mon_swap.mon_dmmax)) ;
#endif
	wprintw(wp, "Total:     %10u KB.\n\n", 0) ;

	wprintw(wp, "Text:          %6.2f %%\n",   0.0) ;
	wprintw(wp, "Free:          %6.2f %%\n",   0.0) ;
	wprintw(wp, "Used:          %6.2f %%\n",   0.0) ;
	wprintw(wp, "Wasted:        %6.2f %%\n",   0.0) ;
	wprintw(wp, "Missing:       %6.2f %%\n",   0.0) ;
	wprintw(wp, "Shared Memory: %6.2f %%\n\n", 0.0) ;

	wprintw(wp, "Usage of available space:") ;

#ifdef	V4_ULTRIX
	lines += 2 ;
#else
	start = ffs(btok(mon_swap.mon_dmmin)) - 1;
	end   = ffs(btok(mon_swap.mon_dmmax)) ;

	for(i = start, printed = 1; i < end; i++) {
		printed++ ;

		if((printed % 3) == 0 )
			lines++ ;
	}
#endif
}

/*
 *	Display the dynamic data of the swap space information.
 *
 *	ARGSUSED
 */
magnify_swap(op)
OPTION	*op ;
{
	register total, missing, i ;
#ifndef	V4_ULTRIX
	register printed ;
#endif

	sample_body((double)(sample.mon_ticks - swap_ticks)/first.mon_hz) ;
	swap_ticks = sample.mon_ticks ;

	total   = mon_swap.mon_total ;
	missing = total - (mon_swap.mon_used + mon_swap.mon_free) ;

	wmove(wp, SWAP_TOTAL_Y, SWAP_TOTAL_X) ;
	wprintw(wp, "%10u",  btok(total)) ;

	wmove(wp, SWAP_DATA, SWAP_X) ;
	wprintw(wp, "%6.2f", (double)(100.0 * mon_swap.mon_text)/total) ;

	wmove(wp, SWAP_DATA + 1, SWAP_X) ;
	wprintw(wp, "%6.2f", (double)(100.0 * mon_swap.mon_free)/total) ;

	wmove(wp, SWAP_DATA + 2, SWAP_X) ;
	wprintw(wp, "%6.2f", (double)(100.0 * mon_swap.mon_used)/total) ;

	wmove(wp, SWAP_DATA + 3, SWAP_X) ;
	wprintw(wp, "%6.2f", (double)(100.0 * mon_swap.mon_wasted)/total) ;

	wmove(wp, SWAP_DATA + 4, SWAP_X) ;
	wprintw(wp, "%6.2f", (double)(100.0 * missing)/total) ;

	wmove(wp, SWAP_DATA + 5, SWAP_X) ;
	wprintw(wp, "%6.2f", (double)(100.0 * mon_swap.mon_smem)/total) ;

	wmove(wp, SWAP_AVAIL_Y, SWAP_AVAIL_X) ;

	wprintw(wp, "%8d KB: %5d ", 1 << 0, mon_swap.mon_avail[0]) ;

#ifdef	V4_ULTRIX
	i = ffs(btok(mon_swap.mon_dmmax)) - 1;
	wprintw(wp, "\n%8d KB: %5d", 1 << i, mon_swap.mon_avail[i]) ;
#else
	for(i = start, printed = 1; i < end; i++) {
		wprintw(wp, "%8d KB: %5d", 1 << i, mon_swap.mon_avail[i]) ;

		printed++ ;

		if((printed % 3) == 0 )
			wprintw(wp, "\n") ;
		else
			wprintw(wp, " ") ;
	}
#endif

	wclrtoeol(wp) ;
}

