/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 monitor.c
 *	Date:	 5/23/90
 *	Version: 1.46
 *
 *	monitor.c - This file contains the main program and other functions
 *	needed to drive the "collect" and "display" functions.
 *
 *	This program is based on the various programs that collect and
 *	display system data.  Among them are:
 *
 *		MONITOR - VMS Utility.
 *		mon     - 4.2BSD UNIX* utility to visually monitor
 *			  system activity, by Phillip Dykstra.
 *		vmstat  - 4.2BSD UNIX* utility to monitor virtual memory
 *			  stats.
 *		iostat  - 4.2BSD UNIX* utility to monitor I/O stats.
 *		netstat - 4.2BSD UNIX* utility to monitor network stats.
 *
 *	*UNIX is a trademark of AT&T Labs.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)monitor.c	1.46 (monitor) 5/23/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 * 28-December-1988 -- arr
 *
 *	Dust off V3.0 lint.
 *
 * Dec. 25, 1989 -- arr
 *
 *	Return a predicatable value and clean up unneeded #includes.
 *
 * Dec. 27, 1989 -- arr
 *
 *	Turn the sleep into a select().
 *
 * May 23, 1990 -- arr
 *
 *	Added 'p'ause.
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "include.h"
#include "options.h"

/*
 *	Data objects define elsewhere.
 */
extern	int	killed ;

/*
 *	Functions that don't return (int).
 */
void	exit() ;

/*
 *	Module name for error functions.
 */
static	char	*module = "monitor" ;

OPTION	option ;	/* option structure; available to catch_sigwinch() */

main(argc, argv)
int	argc ;
char	**argv ;
{
	int	rc ;				/* return code of collect */
	int	found ;				/* number of fd's with input */
	int	read_fds ;			/* mask of input fd's */
	int	command = '\0' ;		/* interactive command */
	void	catch(), catch_sigwinch() ;	/* functions to catch signals */
	struct	timeval intv ;			/* interval timer */

	parse(argc, argv, &option);		/* parse arguments */

	if( option.open_collect )
		(*option.open_collect)(&option) ;

	/*
	 *	Die gracefully on interrupt, hangup, quit and terminate.
	 */
	(void)signal(SIGINT, catch);
	(void)signal(SIGHUP, catch);
	(void)signal(SIGQUIT, catch);
	(void)signal(SIGTERM, catch);
	/*
	 *	Ignore (by default) SIGALRM's.  This should fix
	 *	the alarm clock problem.
	 */
	(void)signal(SIGALRM, SIG_IGN) ;

	/*
	 *	Perform magick on a SIGWINCH.
	 */
	(void)signal(SIGWINCH, catch_sigwinch);

	if( option.open_display )
		(*option.open_display)(&option);

	intv.tv_sec = option.opt_sleep ;
	intv.tv_usec = option.opt_fract * 1000000 ;

	/*
	 *    Until we are "killed" or see an exit command:
	 *
	 *	  Execute the collection function execute the
	 *	  the display function if everything is NORMAL
	 *	  and wait for the next sample.
	 */
	while( command != 'q' && command != 'e' && !killed ) {
		if( option.collect )
			rc = (*option.collect)(&option) ;

		if( rc == MON_EXIT )
			break ;
		else if( rc == MON_CONTINUE )
			continue ;
		else {
			if( option.display )
				(*option.display)(&option);
		}

		if( option.opt_inter ) {
			screen_prompt(&option);
			read_fds = 1 << fileno(stdin) ;
			found = select(1, &read_fds, (int *)0, (int *)0, &intv);

			if( found && read_fds )
				command = exec_command(&option) ;
		}
		else if( option.opt_sleep ) {
			if( select(0, (int *)0, (int *)0, (int *)0, &intv) == -1 )
				if( errno != EINTR )
					fatal("Sleep failed: %s.\n", module) ;
		}
		else
			;			/* don't delay */
	}

	if( option.close_display )
		(*option.close_display)(&option);

	if( option.close_collect )
		(*option.close_collect)(&option);

	return 0 ;
}

/*
 *	Input has become available on our standard input.  Read it and
 *	execute the approproiate command.
 */
exec_command(op)
OPTION	*op ;
{
	register command ;
	FILE	 *fp, *fopen() ;

	switch( command = getchar()) {
	case 'h':				/* Help */
	case '?':
		if( op->help_display )
			(*op->help_display)(op) ;

		if( op->redraw_display )
			(*op->redraw_display)(op) ;

		break ;
	case 'p':				/* pause */
		if( op->opt_curses )
			curses_pause() ;

		if( op->redraw_display )
			(*op->redraw_display)(op) ;

		break ;
	case 'u':				/* Unmagnify */
		unmagnify(op);
		break ;
	case 'm':				/* Magnify */
		magnify(op) ;

		if( op->redraw_display )
			(*op->redraw_display)(op) ;

		break ;
	case 'r':				/* redraw */
	case 12:				/* ^L */
		if( op->redraw_display )
			(*op->redraw_display)(op);

		break ;
	case 'q':				/* quit */
	case 'e':				/* exit */
		break ;
	case 'd':				/* screen dump */
		if((fp = fopen(op->opt_output, "a")) == NULL ) {
			warning("Can't open output file: %s (%s).\n", module,
				op->opt_output) ;
			return command ;
		}

		dump_screen(fp);

		if( fclose(fp) == EOF )
			warning("Can't close dump file: %s.\n", module) ;

		break ;
	default:
		break ;
	}

	return command ;
}
