/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 collect_disk.c
 *	Date:	 8/22/90
 *	Version: 1.8
 *
 *	collect_disk.c - Look at the various available data structures and
 *	figure out what disks are available to collect information
 *	on.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)collect_disk.c	1.8 (monitor) 8/22/90" ;
#endif

/*
 * 2-December-1988 -- arr
 *
 *	New file.  A merger of parts of live.c, uba.c and mba.c.  The
 *	last two should go away as a result of this.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Correctly set mon_flag with MON$M_VALID.
 *
 * Dec. 25, 1989 -- arr
 *
 *	Attempt to resolve all the different device naming
 *	conventions between V3.0, V3.1{A,B,C} and V4.0.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 * Aug. 22, 1990 -- arr
 *
 *	I found a curious bug on a V4.0 DECstation that the command
 *	line:
 *		monitor disk=rz0,rz3 disk=rz3
 *
 *	caused data for an rz3 to be collected twice.  Now I only
 *	allow any given device name once.
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <time.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>

#ifdef	V4_ULTRIX
#	include <sys/smp_lock.h>
#endif

#include <net/if.h>
#include <netinet/in.h>

#include <machine/param.h>
#include <machine/pte.h>

#ifdef	V4_ULTRIX
#	ifdef	vax
#		include <io/mba/vax/mbavar.h>
#		include <io/mba/vax/mbareg.h>
#		include <io/uba/ubavar.h>
#	elif	mips
#		include "mbavar.h"
#		include "mbareg.h"
#		include <io/uba/ubavar.h>
#	endif
#else
#	ifdef	vax
#		include <vaxmba/mbavar.h>
#		include <vaxmba/mbareg.h>
#		include <vaxuba/ubavar.h>
#	elif	mips
#		include <machine/hwconf.h>
#		include "mbavar.h"
#		include "mbareg.h"
#		include "ubavar.h"
#	endif
#endif

#include "include.h"
#include "options.h"
#include "monitor.h"
#include "extern.h"

/*
 *	The module name for the error functions.  It's only used 
 *	in the PMAX code so I'll make it conditional to keep lint 
 *	happy.
 */
static	char	*module = "collect_disk" ;

/*
 *	Declarations for function that don't return (int).
 */
char	*strcpy() ;
void	exit() ;

/*
 *	dk_map has already been declared for the names()
 *	code, so we'll reuse it here to save some space.
 *	After wandering down the UNIBUS-like and MASSBUS
 *	data structures we'll use dk_xfer to fill in any
 *	missing disks.
 */
extern long dk_xfer[DK_NDRIVE] ;
extern char dk_map[DK_NDRIVE] ;

/*
 *	Gather the static data (names of devices, physmem, etc).
 *
 *	Eventually the functions which collect names will be
 *	responseable for doing the handling the command line
 *	options like "disk=ra0,hp1,ra2".
 */
collect_disk(op)
OPTION	*op ;
{
	register i, dk ;
	float	 dk_mspw[DK_NDRIVE] ;

	/*
	 *	Clear the dk_map[] array.
	 */
	for(i = 0; i < DK_NDRIVE; i++)
		dk_map[i] = 0 ;

	/*
	 *	Determine what disks are available from looking
	 *	at the configuration.
	 */
	uba_disks(op);		/* Unibus/Qbus disks */

	mba_disks();		/* MASSBUS disks */

	/*
	 *	This will get to go away in a V4 only version
	 *	one of these days.
	 */
	scsi_disks() ;		/* Whatever is left over. */

#ifdef	vax
	/*
	 *	It's assumed that once a disk is configured the
	 *	transfer speed doesn't change.  This is probably
	 *	bogus.
	 */
	readk((long)namelist[NM_DK_MSPW].n_value, (char *)dk_mspw,
		sizeof(dk_mspw));

	for(i = 0; i < n_disk ; i++) {
		dk = disk[i].mon_dk ;
		disk[i].mon_mspw = dk_mspw[dk] ;
	}
#elif	mips
	for(i = 0; i < n_disk ; i++)
		disk[i].mon_mspw = 0.0 ;
#endif
}

/*
 *	Fill a disk structure.
 */
static fill_disk(current, name, unit, dk)
unsigned current ;
int	unit, dk ;
char	*name ;
{
	register i ;

	/*
	 *	Prevent duplicate disk names, whether the user
	 *	wants them or not.
	 */
	for(i = 0; i < current; i++) {
		if( unit == disk[i].mon_unit && strcmp(name, disk[i].mon_name) == 0 )
			return 0 ;
	}

	disk[current].mon_length = MON$S_DISK ;
	disk[current].mon_type = MON$C_DISK ;
	disk[current].mon_flag = MON$M_VALID ;

	disk[current].mon_index = i ; 
	disk[current].mon_dk = dk ;
	disk[current].mon_unit = unit ;
	disk[current].mon_name[DEV_SIZE] = '\0' ;

	(void)strcpy(disk[current].mon_name, name) ;

	return 1 ;
}

/*
 *	A device name is supposed to have a series of non-digits
 *	followed by a series of digits (rz0, ra1, foobar666).
 *
 *	This function return the address of the first digit or
 *	NULL if there isn't one.
 */
char	*find_digit(name)
char	*name ;
{
	register char *p ;

	if( name == NULL )
		panic("NULL device name in find_digit().\n", module) ;

	for(p = name; *p != '\0'; p++)
		if( isdigit(*p))
			return p ;

	info("Device name doesn't contain a digit: %s.\n", module, name) ;

	return NULL ;
}

/*
 *	Read UNIBUS device names from kernel.
 *
 *	To simplify the view of the world Ultrix seems to think that
 *	there are two kinds of busses.  The MASSBUS and the UNIBUS.
 *	All Q-BUS devices and most VAXBI nodes are made to look like
 *	UNIBUS devices.  For example:
 *
 *		The KDA50 looks like a UDA50.
 *		The KDB50 looks like a UDA50 on its own UNIBUS.
 *
 *	Since there must be an exception to this rule the PMAX chose 
 *	not to maintain this tradition.  It is quite true since the 
 *	PMAX is a busless machine.  So are the VS2000, MicroVAX 2000 
 *	and PVAX.  At least in the case of the VS2000 the tradition 
 *	has worked quite well and >>>I<<< don't see the need to change 
 *	it since it just complicted the way monitor must handle such 
 *	things.
 */

/*
 *	Walk through the UNIBUS device list.
 */
uba_disks(op)
OPTION	*op ;
{
	struct	uba_device uba_device, *up ;
	char	device[DEV_SIZE+1] ;
	int	unit ;

	device[DEV_SIZE] = '\0' ;

	/*
	 *	Make sure the beginning of the list is a reasonable
	 *	place.
	 */
	if((up = (struct uba_device *)namelist[NM_UBDINIT].n_value) == 0 )
		return ;

	for ( ; ; ) {
		/*
		 * Read the next uba_device structure from the kernel.
		 */
		readk((long)up++, (char *)&uba_device, sizeof(uba_device));

		/*
		 * If the driver entry is 0, then we're at end of the list.
		 */
		if( uba_device.ui_driver == 0 )
			break ;

		/*
		 *	If the dk and slave number is -1 then
		 *	the device is probably a communication
		 *	device that we're not interested in.
		 */
		if( uba_device.ui_dk == -1 || op->opt_disk == 0 )
			continue ;

		/*
		 *	Read the device name and save the unit number.
		 */
		readk((long)uba_device.ui_devname, device, DEV_SIZE);

		unit = uba_device.ui_unit ;

		/*
		 *	If the device is not "alive" goto the next device.
		 */
		if( !uba_device.ui_alive )
			continue ;

		/*
		 *	See if this is an "interesting" device.
		 */
		if( find_value(&disk_list, device, unit) == 0 )
			continue ;

		if( uba_device.ui_dk >= 0 && uba_device.ui_dk < DK_NDRIVE )
			dk_map[uba_device.ui_dk] = 1 ;

		if( fill_disk(n_disk, device, unit, uba_device.ui_dk) == 1 )
			n_disk++ ;
	}
}

/*
 *	Walk down the MASSBUS device list.
 */
mba_disks()
{
	struct	mba_device mba_device, *mp ;
	struct	mba_driver driver ;
	char	device[DEV_SIZE+1] ;

	/*
	 *	Make sure the beginning of the list is a reasonable
	 *	place to start.
	 */
	if((mp = (struct mba_device *)namelist[NM_MBDINIT].n_value) == NULL )
		return ;

	for( ; ; ) {
		/*
		 *	Read the next mba_device structure.
		 */
		readk((long)mp++, (char *)&mba_device, sizeof(mba_device));

		/*
		 *	If the driver entry is 0 then we're done.
		 */
		if( mba_device.mi_driver == 0 )
			break ;

		/*
		 *	The only type of interesting MASSBUS "device" are
		 *	disks, so skip the device if it's dk number is -1.
		 */
		if( mba_device.mi_dk == -1 )
			continue ;

		/*
		 *	Read the driver structure and device name.
		 */
		readk((long)mba_device.mi_driver, (char *)&driver, sizeof(driver));

		readk((long)driver.md_dname, (char *)device, DEV_SIZE);

		/*
		 *	If the device isn't alive skip it.
		 */
		if( !mba_device.mi_alive )
			continue ;

		/*
		 *	Check to see if this is an interesting device.
		 */
		if( find_value(&disk_list, device, mba_device.mi_unit) == 0 )
			continue ;

		if( mba_device.mi_dk >= 0 && mba_device.mi_dk < DK_NDRIVE )
			dk_map[mba_device.mi_dk] = 1 ;

		if( fill_disk(n_disk, device, mba_device.mi_unit, mba_device.mi_dk) == 1 )
			n_disk++ ;
	}
}

/*
 *	Life gets interesting.  The disks on the PMAX don't like
 *	disks on previous version so a lot of hand waving must
 *	take place.  This is how monitor will behave (or misbehave).
 *
 *	If a disk is on the command line list of "interesting" disks
 *	then monitor will collect data for it, EVEN IF it doesn't
 *	appear active (the user knows best).  The only requirement
 *	is that the disk name must be "rz" or "dk" and the dk number
 *	be less than DK_NDRIVE.
 *
 *	If the list is empty then the only drives that will be watched
 *	are those that appear active.  Active in this case means that
 *	the number transfers are non-zero.  (Note: iostat(1) uses the
 *	number of words tranfered as the criteria, but that's bogus
 *	because the number of words transfered has a better chance of
 *	rolling over and there can appear to be zero even though it's
 *	not.  Very unlikely, but possible).
 */
scsi_disks()
{
	register i, dk ;
	char	 device[DEV_SIZE+1], *digit, *name, *find_digit() ;

	/*
	 *	If the list is empty.
	 */
	if( disk_list.nel == 0 ) {
		/*
		 *	Get the transfer data struture.
		 */
		readk((long)namelist[NM_DK_XFER].n_value, (char *)dk_xfer,
			sizeof(dk_xfer)) ;

		for(i = 0; i < DK_NDRIVE; i++) {
			if( dk_map[i] || dk_xfer[i] == 0 )
				continue ;

			if( fill_disk(n_disk, "rz", i, i) == 1 )
				n_disk++ ;
		}
	}
	/*
	 *	Other use the command line list.
	 */
	else {
		for(dk = 0; dk < disk_list.nel; dk++) {
			name = disk_list.base[dk] ;

			if((digit = find_digit(name)) != NULL ) {
				for(i = 0; name[i] != *digit && i < DEV_SIZE; i++)
					device[i] = name[i] ;

				device[i] = '\0' ;

				/*
				 *
				 */
				if( allow_devices(device) == 0 )
					continue ;

				if( fill_disk(n_disk, device, atoi(digit), dk) == 1 )
					n_disk++ ;
			}
		}
	}
}

/*
 *	List of devices to be allowed from the command line.
 */
static char *allowed_list[] = {
	"dk",
	"rz",
	"tz",
} ;

allow_devices(name)
char	*name ;
{
	register i ;

	for(i = 0; i < sizeof(allowed_list)/sizeof(allowed_list[0]); i++)
		if( strcmp(name, allowed_list[i]) == 0 )
			return 1 ;

	return 0 ;
}
