/*
 *	Author:  Alan Rollow, EIS/CXO, Digital Equipment Corp.
 *	File:	 buf.c
 *	Date:	 3/29/90
 *	Version: 1.14
 *
 *	buf.c - Deal with the buffer cache stats.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)buf.c	1.14 (monitor) 3/29/90" ;
#endif

/*
 * Modification history.
 *
 * Jan. 24, 1989 -- arr
 *
 *	Added to code to collect data for detail structure.  Also
 *	moved f_buf() from functions.c.
 *
 * Feb. 10, 1989 -- arr
 *
 *	Redesigned the way buffer cache data is collected and
 *	stored.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Added checks of mon_flag to f_buf() and magnify_buf().
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <fstab.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/param.h>
#include <sys/user.h>
#include <sys/mount.h>

#include <net/if.h>
#include <netinet/in.h>

#include "monitor.h"
#include "options.h"
#include "include.h"
#include "extern.h"

/*
 *	Data declared elsewhere.
 */
extern	int	lines ;
extern	WINDOW	*wp ;

/*
 *	Local data.
 */
static	int		buf_ticks = 0 ;
static	int		missed_buffers = 0 ;
static	char		*module = "buf" ;
static	unsigned       	nelp = 1 ;

static	struct 	buf	*xbp = 0 ;
static	struct	buf	*addr ;

static	struct mon_buf key = {
	MON$C_BUF, MON$M_VALID, MON$S_BUF,
	0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* flags */
	-1, 0, '\0',
};

/*
 *	Functions that don't return (int).  Mostly for form and
 *	to keep lint happy.
 */
long	lseek();
char	*calloc(), *strcpy(), *lsearch(), *lfind(), *strncpy();
void	exit(), free(), perror(), nlist() ;
struct fstab *getfsent() ;

/*
 *	Setup the data structures needed to collect buffer
 *	cache info.
 */
setup_buf()
{
	unsigned nbuf ;
	int	 i ;

	if( namelist[NM_BUF].n_value == 0 ) {
		info("Buffer cache data not in namelist.\n", module);
		exit(1);
	}

	/*
	 *	Set aside enough buf entries for every file system in 
	 *	/etc/fstab + 1.  An interesting problem with this is 
	 *	that a buffer gets allocated to each disk when a chpt 
	 *	is done.  These will will eventually be recycled and 
	 *	used somewhere interesting, but they are hard allocate
	 *	a BUF structure for and might cause an entry to be
	 *	missed.
	 */
	n_buf = 1 ;

	while( getfsent() != NULL )
		n_buf++ ;

	/*
	 *	Allocate space for them.
	 */
	if((mon_buf = (struct mon_buf *)calloc(n_buf, MON$S_BUF)) == NULL )
		fatal("Can't allocate space for BUF structures: %s.\n", module);

	/*
	 *	Initialize all the record headers.
	 */
	for(i = 0; i < n_buf; i++) {
		mon_buf[i].mon_type   = MON$C_BUF ;
		mon_buf[i].mon_device = -1 ;
		mon_buf[i].mon_flag   = 0 ;
		mon_buf[i].mon_length = MON$S_BUF ;
	}

	/*
	 *	Read the number of buffers.
	 */
	mon_buf[0].mon_nbuf = nbuf = get_word((long)namelist[NM_NBUF].n_value) ;

	/*
	 *	Allocate space for the buffer headers.
	 */
	if((xbp = (struct buf *)calloc(nbuf, sizeof (struct buf))) == NULL )
		fatal("Can't allocate space for buffer list", module);

	/*
	 *	Read some more stuff from the kernel.  Addr is the address
	 *	of the beginning of the buffer cache headers.  Bufpages
	 *	is the number of pages set aside for buffers.
	 */
	addr = (struct buf *)get_word((long)namelist[NM_BUF].n_value) ;

	mon_buf[0].mon_bufpages = get_word((long)namelist[NM_BUFPAGES].n_value) ;

	/*
	 *	Calculate the size of "bufpages".
	 */
	mon_buf[0].mon_bufpages *= CLSIZE * NBPG ;
	mon_buf[0].mon_flag = MON$M_VALID ;
	(void)strcpy(mon_buf[0].mon_fsname, "Total:") ;
	
}

/*
 *	Compare function for the qsort(3) in collect_buffer_cache().
 */
int	buf_qsort(a, b)
register struct mon_buf *a, *b ;
{
        if( a->mon_nused > b->mon_nused )
                return -1 ;
        else if( a->mon_nused == b->mon_nused )
                return 0 ;
        else
                return 1 ;
}

/*
 *	Read a copy of the buffer headers and make a pass
 *	through them collecting the data.
 */
collect_buffer_cache()
{
	register i ;
	unsigned nbuf = mon_buf[0].mon_nbuf ;

	mon_buf[0].mon_bcount = 0 ;
	mon_buf[0].mon_bufsize = 0 ;
	mon_buf[0].mon_nused = 0 ;
	mon_buf[0].mon_cache = 0 ;
	mon_buf[0].mon_delwri = 0 ;
	mon_buf[0].mon_read = 0 ;
	mon_buf[0].mon_busy = 0 ;
	mon_buf[0].mon_done = 0 ;
	mon_buf[0].mon_inval = 0 ;
	mon_buf[0].mon_error = 0 ;

	if( xbp == NULL || mon_buf == NULL )
		setup_buf() ;

	/*
	 *	Restart the list.
	 */
	nelp = 1 ;

	/*
	 *	Invalidate all the existing records, except the first.
	 */
	for(i = 1; i < n_buf; i++)
		mon_buf[i].mon_flag = 0 ;

	/*
	 *	Read the headers of the buffer cache.
	 */
	readk((long)addr, (char *)xbp, (int)(nbuf * sizeof(struct buf))) ;

	/*
	 *	Summarize the data from each buffer.
	 */
	for(i = 0; i < nbuf; i++)
		if( xbp[i].b_dev != -1 )
			add_buf(xbp + i) ;

	for(i = 1; i < n_buf; i++)
		get_fsname(mon_buf[i].mon_device, mon_buf[i].mon_fsname) ;

	if( n_buf >= 2 )
		qsort((char *)(mon_buf + 1), (int)(n_buf - 1), MON$S_BUF, 
			buf_qsort) ;
}

/*
 *	Comparison function for lsearch(3) or lfind().
 */
buf_equal(a, b)
struct mon_buf *a, *b ;
{
        if( a->mon_device == b->mon_device )
                return 0 ;
        else
                return -1 ;
}

/*
 *
 */
add_buf(bp)
register struct buf *bp ;
{
        register struct mon_buf *p ;
	char	*(*funct)() ;

	/*
	 *	Do the totals.
	 */
	mon_buf[0].mon_bcount  += bp->b_bcount ;
	mon_buf[0].mon_bufsize += bp->b_bufsize ;
	mon_buf[0].mon_nused++ ;

	check_bits(bp->b_flags, mon_buf + 0) ;

	/*
	 *	Setup the key for the search.
	 */
        key.mon_device = bp->b_dev ;

	/*
	 *	If we've run out of table, arrange to just do a lookup.
	 */
	if( nelp == n_buf )
		funct = lfind ;
	else
		funct = lsearch ;

	/*
	 *	Search the table for this buffer entry.
	 */
        p = (struct mon_buf *)(*funct)((char *)&key, (char *)mon_buf, &nelp,
                sizeof(key), buf_equal) ;

	/*
	 *	If we didn't find it, print a message and increment
	 *	a counter.  This should only happen when the search
	 *	function is lsearch(3).
	 */
	if( p == NULL ) {
		missed_buffers++ ;
		return ;
	}

	/*
	 *	Then the per device.
	 */
        p->mon_bcount  += bp->b_bcount;
        p->mon_bufsize += bp->b_bufsize;
        p->mon_nused++ ;

	check_bits(bp->b_flags, p) ;
}

check_bits(flags, p)
register long flags ;
register struct mon_buf *p ;
{
        if( flags & B_CACHE )
		p->mon_cache++ ;

	if( flags & B_DELWRI )
		p->mon_delwri++ ;

	if( flags & B_READ )
		p->mon_read++ ;

	if( flags & B_BUSY )
		p->mon_busy++ ;

	if( flags & B_DONE )
		p->mon_done++ ;

	if( flags & B_INVAL )
		p->mon_inval++ ;

	if( flags & B_ERROR )
		p->mon_error++ ;
}

/*
 *	Clean-up when everything is done.
 */
finish_buf()
{
	free((char *)xbp);

	free((char *)mon_buf) ;
}

/*
 *	Function to dump a record for easy reading.
 */
f_buf(p)
struct mon_buf *p ;
{
	if((p->mon_flag & MON$M_VALID) == 0 )
                return ;

        printf("%s.\n", records[MON$C_BUF].string) ;

	printf("\tnbuf:     %d\n", p->mon_nbuf) ;
	printf("\tbufpages: %d\n", p->mon_bufpages) ;
	printf("\tnused:    %d\n", p->mon_nused) ;
	printf("\tbcount:   %d\n", p->mon_bcount) ;
	printf("\tbufsize:  %d\n", p->mon_bufsize) ;
	printf("\tcache:    %d\n", p->mon_cache) ;
	printf("\tdelwri:   %d\n", p->mon_delwri) ;
	printf("\tread:     %d\n", p->mon_read) ;
	printf("\tbusy:     %d\n", p->mon_busy) ;
	printf("\tdone:     %d\n", p->mon_done) ;
	printf("\tinval:    %d\n", p->mon_inval) ;
	printf("\terror:    %d\n", p->mon_error) ;
	printf("\tdevice:   <%d,%d>\n", major(p->mon_device),
		minor(p->mon_device)) ;
	printf("\tfsname:   %s\n", p->mon_fsname) ;
}

/*
 *	Number of lines used by the buffer cache screen.
 */
#define	BUF_LINES	(6)

/*
 *	Constants for how things are placed.
 */
#define	BUF_X		(0)
#define	BUF_Y		(3)
#define	FS_WIDTH	(24)

#define	HALF		(22)
#define	BUF_X_OFFSET	(7)
#define	BUF_X2_OFFSET	(31)

/*
 *      Setup the static screen for displaying buffer cache 
 *      information.
 *
 *      ARGSUSED
 */
open_buf(op)
OPTION	*op ;
{
	sample_header() ;

	lines = BUF_LINES ;

	wmove(wp, BUF_Y, 0) ;

	wprintw(wp, "There are %d buffers using %d bytes of memory.\n\n",
		mon_buf[0].mon_nbuf, mon_buf[0].mon_bufpages) ;

	wprintw(wp, "%-*.*s %9s %8s %12s %10s", FS_WIDTH-1, FS_WIDTH-1,
		"File System", "Bufs used", "Cache", "Bytes used", "Bufsize") ;
}

/*
 *      Display the dynamic data of the buffer cache information.
 *
 *      ARGSUSED
 */
magnify_buf(op)
OPTION	*op ;
{
	register i ;

	sample_body((double)(sample.mon_ticks - buf_ticks)/first.mon_hz) ;
        buf_ticks = sample.mon_ticks ;

	wmove(wp, BUF_LINES, 0) ;

	lines = BUF_LINES ;

	for(i = 1; i < n_buf ; i++) {
		if( mon_buf[i].mon_nused == 0 || 
		    (mon_buf[i].mon_flag & MON$M_VALID) == 0 )
			continue ;

		wprintw(wp, "%-*.*s %8u %8u %12u %10u\n", 
			FS_WIDTH, FS_WIDTH, mon_buf[i].mon_fsname,
			mon_buf[i].mon_nused, mon_buf[i].mon_cache,
			mon_buf[i].mon_bcount, mon_buf[i].mon_bufsize) ;

		lines++ ;

		if( lines == (LINES - 3))
			break ;
	}

	lines += 3 ;

	wprintw(wp, "\n%-*.*s %8u %8u %12u %10u", 
		FS_WIDTH, FS_WIDTH, mon_buf[0].mon_fsname,
		mon_buf[0].mon_nused, mon_buf[0].mon_cache,
		mon_buf[0].mon_bcount, mon_buf[0].mon_bufsize) ;

	wclrtobot(wp) ;
}

#define	FS_BUFS	(32)

get_fsname(dev, buf)
dev_t	dev ;
char	*buf ;
{
	struct fs_data buffer[FS_BUFS] ;
	register i ;
	int	start = 0, rc ;

	while((rc = getmountent(&start, buffer, FS_BUFS)) > 0 ) {
		for(i = 0; i < rc; i++)
			if( dev == buffer[i].fd_req.dev ) {
				(void)strcpy(buf, buffer[i].fd_req.path) ;
				buf[FS_WIDTH-1] = '/' ;
				return ;
			}
	}
	
	(void)sprintf(buf, "<%d,%d>", major(dev), minor(dev)) ;

	return ;
}
