/* advtrm.c - terminal i/o routines */
/*
        Copyright (c) 1986, by David Michael Betz
        All rights reserved
*/
#pragma lint -1
#pragma noroot

#if STACKDEBUG
#pragma debug 24
#endif

#if OPTIMIZE
#pragma optimize 9
#endif

#include "allint.h" /* TUU */
/* #include <stdio.h>  already in allint.h */
#include <string.h> /* TUU */
#ifdef __ORCAC__
#include <gsos.h>  /* TUU */
#define GetVectors 0x8007

short gConsoleCOutLow;
short gConsoleCOutHigh;

typedef struct
  {
   long COutVector,KeyInVector;
  } GetVectorsList,*GetVectorsListPtr;

#define dispatcher 0xE10000
extern pascal void DebugStr(char *) inline(0x09FF,dispatcher);
#endif

#ifdef CONSOLE
Word gConsoleDevNum;
Boolean gAtLeast601=0;
#endif

/* useful definitions */
#define TRUE    1
#define FALSE   0
#define EOS     '\0'
#define LINEMAX 200
#define WORDMAX 100

/* global variables */
char line[LINEMAX+1];

/* local variables */
static int col,maxcol,row,maxrow;
static int scnt,wcnt;
static char theWord[WORDMAX+1],*wptr;
static FILE *logfp = NULL;

/* forward declarations in allint.h
char *trm_line();
***/

/* trm_init - initialize the terminal module */
void trm_init(MyInt rows,MyInt cols,char *name)
{
#ifdef CONSOLE             /* TUU */
   DevNumRecGS myDevNumRec;
   DAccessRecGS myDAccessRec;
   VersionRecGS myVersionRec;

   myVersionRec.pCount=1;
   GetVersion(&myVersionRec);

   gAtLeast601 = (myVersionRec.version >= 0x0402);
   if (gAtLeast601)
     {            
      printf("You're using at least System 6.01. Cool. Output will be fast!\n\n");
      myDevNumRec.pCount = 2;
      myDevNumRec.devName = (GSString32Ptr) "\x8\x0.Console";
      GetDevNumber(&myDevNumRec);
      gConsoleDevNum=myDevNumRec.devNum;  /* saved for possible future use */

    /* get the ConsoleCOut vector */
      myDAccessRec.pCount=5;
      myDAccessRec.devNum=gConsoleDevNum;
      myDAccessRec.code = GetVectors;
      myDAccessRec.list = (Pointer) &myDevNumRec;
      myDAccessRec.requestCount = 8;
      DStatus(&myDAccessRec);
      gConsoleCOutLow = (short) (((GetVectorsListPtr) myDAccessRec.list)->COutVector);
      gConsoleCOutHigh= (((GetVectorsListPtr) myDAccessRec.list)->COutVector)>>16;
     }
   else
     printf("You're running a System version lower than 6.01. Output'll be the normal\n"
            "UNIX-type routines (i.e. slow).  You can get System 6.01 free from\n"
            "ftp.apple.com. Output is much, much faster under System 6.01.\n\n");
#endif

    /* initialize the terminal i/o variables */
    maxcol = cols-1; col = 0;
    maxrow = rows-1; row = 0;
    wptr = theWord; wcnt = 0;
    scnt = 0;

    /* open the log file */
    if (name && (logfp = fopen(name,"w")) == NULL)
        error("can't open log file");
}

/* trm_done - finish terminal i/o */
void trm_done(void)
{
    if (wcnt) trm_word();
    if (logfp) fclose(logfp);
}

/* trm_get - get a line */
char *trm_get(char *line)
{
    if (wcnt) trm_word();
    while (scnt--) putchr(' ');
    row = col = scnt = 0;
    return (trm_line(line));
}

/* trm_str - output a string */
void trm_str(char *str)
{
    while (*str)
        trm_chr(*str++);
}

/* trm_xstr - output a string without logging or word wrap */
void trm_xstr(char *str)
{
 fputs(str,stdout); /* TUU - works since no logging or word wrap */

/**** TUU was
    while (*str)
        putch(*str++,stdout);
***/
}

/* trm_chr - output a character */
void trm_chr(int ch)
{
    switch (ch) {
    case ' ':
            if (wcnt)
                trm_word();
            scnt++;
            break;
    case '\t':
            if (wcnt)
                trm_word();
            scnt = (col + 8) & ~7;
            break;
    case '\n':
            if (wcnt)
                trm_word();
            trm_eol();
            scnt = 0;
            break;
    default:
            if (wcnt < WORDMAX) {
                *wptr++ = ch;
                wcnt++;
            }
            break;
    }
}

/* trm_word - output the current word */
void trm_word(void)
{
    if (col + scnt + wcnt > maxcol)
        trm_eol();
    else
        while (scnt--)
            { putchr(' '); col++; }
    for (wptr = theWord; wcnt--; col++)
        putchr(*wptr++);
    wptr = theWord;
    wcnt = 0;
    scnt = 0;
}

/* trm_eol - end the current line */
void trm_eol(void)
{
#ifdef CONSOLE
   if (gAtLeast601)
     putchr(0xD);
   else
     putchr('\n');
#else
   putchr('\n');
#endif
    if (++row >= maxrow)
        { trm_wait(); row = 0; }
    col = 0;
}

/* trm_wait - wait for the user to type return */
void trm_wait(void)
{
    trm_xstr("  << MORE >>");
    waitch();
    trm_xstr("            \r");
}

/* trm_line - get an input line */
char *trm_line(char *line)
{
    char *p;
    int ch;

    p = line;
    while ((ch = getchr()) != EOF && ch != '\n')
        switch (ch) {
        case '\177':
        case '\010':
                if (p != line) {
                    if (ch != '\010') putchr('\010');
                    putchr(' ');
                    putchr('\010');
                    p--;
                }
                break;
        default:
                if ((p - line) < LINEMAX)
                    *p++ = ch;
                break;
        }
    *p = 0;
    return (ch == EOF ? NULL : line);
}

/* getchr - input a single character */
int getchr(void)
{
    int ch;

    if ((ch = getch()) != EOF && logfp)
        putch(ch,logfp);
    return (ch);
}

/* putchr - output a single character */
void putchr(int ch)
{
 if (logfp) putch(ch,logfp);
#ifdef CONSOLE
 if (gAtLeast601)
   {
    if ((ch<32) && (ch!='\r'))
      ch=0; /* null -- do nothing -- simple workaround for control characters
               that could screwup current state of Console Driver */

#include "consoleassem.c"    /* TUU workaround for lame bug in Orca/C */
   }
 else
   putch(ch,stdout);
#else
  putch(ch,stdout);
#endif
}
