/***********************************************************************
 *                                                                     *
 * Program to control LORAN-C radio                                    *
 *                                                                     *
 * Tables used for scheduling, waveshaping and navigation              *
 *                                                                     *
 ***********************************************************************
 */

/*
 * Current LORAN-C chains by gri (master listed first)
 *
 * 9990 North Pacific		St. Paul Island, Attu, Port Clarence,
 *							Narrow Cape
 * 9980 Icelandic Sea
 * 9970 Northwest Pacific	Iwo Jima, Marcus Island, Hokkaido, Gesashi,
 *							Yap Island
 * 9960 Northeast US		Seneca, Caribou, Nantucket, Carolina Beach,
 *							Dana, Wildwood
 * 9940 West Coast US		Fallon, George, Middletown, Searchlight
 * 9610 South Central US	Boise City, Gillette, Searchlight, Las
 *							Cruces, Raymondville
 * 8970 Great Lakes US		Dana, Malone, Seneca, Baudette, Boise City,
 * 							Wildwood
 * 8290 North Central US	?
 * 7990 Mediterranian Sea	Sellia Marina, Lampedusa, Kargabarun,
 *							Estartit
 * 7980 Southeast US		Malone, Grangeville, Raymondville, Jupiter,
 *							Carolina Beach
 * 7970 Norwegian Sea		Ejde, Sylt, B0, Sandur, Jan Mayen
 * 7960 Gulf of Alaska		Tok, Narrow Cape, Shoal Cove, Port Clarence
 * 7930 Labrador Sea		Fox Harbour, Cape Race, Angissoq
 * 5990 West Central Canada	Williams Lake, Shoal Cove, George, Port
 *							Hardy
 * 5930 East Central Canada	Caribou, Nantucket, Cape Race. Fox Harbour
 * 4990 Central Pacific		Johnson Island, Upolu Point, Kure Island
 */

#include "loran.h"

/*
 * STC setup. Note gri = 99600, pci = 400 and stb = 10 (us).
 * (Some of these values can be overriden by the program.)
 *
 * Counter 1 generates a 200-kHz signal from the 5-MHz master VCO. This
 * signal is a slightly assymetrical square wave (duty factor 12/13). An
 * external flipflop divides this signal by two to get the 100-kHz gri
 * clock which drives counter 2. All other counters are driven from the
 * 5-MHz source. The 200-kHz and 100-kHz signals are used by the
 * synchronous demodulator in the receiver.
 *
 * Counter 2 generates the gri (pulse-code) gate, which repeats at the
 * interval assigned to the LORAN-C chain; e.g., 9960 for the Northeast
 * U.S. chain. The signal consists of a high 8-ms interval preceeded by
 * a programmable low interval normally equal to the gri interval less 8
 * ms. Counter 3 generates the pulse-code (pci) gate, which enables the
 * receiver only when a pulse group is expected. The signal consists of
 * eight 400-us high intervals beginning at the high interval of counter
 * 2. Counter 4 generates the stb (cycle) gate used during envelope
 * scan. The signal consists of a high 10-us interval preceeded by a
 * programmable interval in the range up to about 400 us.
 *
 * Counter 5 operates as a gated divider to drive the frequency scalar
 * and output divider, which produces the output signals for external
 * equipment. The gating signal is generated by counter 4, which can be
 * enabled for this purpose under program control. When so enabled,
 * counter 5 is stopped for the interval programmed in counter 4,
 * enabling precise alignment of the frequency scalar and output divider
 * to UTC. The output signal can be at 1 pps and any decimal multiple up
 * to 100 kHz plus 5 MHz or, if UTC alignment is not necessary, any
 * binary or decimal submultiple of 5 MHz. Note that all counters
 * operate in binary mode, except the frequency scalar and output
 * divider, which normally operate in bcd mode.
 */
int init[] = {					/* stc initialization vector */
	0x0062 + OSC,   12,   13,	/* counter 1 (p0) */
	0x0262, 9160,  GRI,			/* counter 2 (gri) */
	0x8062 + OSC, PCX, 5000 - PCX, /* counter 3 (pcx) */
	0xc062 + OSC,    0,   STROBE, /* counter 4 (stb) */
	0x0062 + OSC,   25,   25 	/* counter 5 (out) */
	};

/*
 * Standard envelope model cycle-amplitude and cycle-weighting vectors.
 * The model cycle amplitudes are matched with the received envelope
 * amplitudes to compute the rms error, assuming the reference cycle (3)
 * is at the reference phase in the envelope window. The cycle amplitude
 * values (peak-normalized to 100) are taken from the standard LORAN-C
 * transmission specification and are averaged over the entire complete
 * cycle. Cycle zero (the period before the first carrier cycle) is used
 * to obtain a noise estimate. This cycle and cycle 8 (the first cycle
 * after the max envelope cycle) are also used as a buffer should the
 * strobe wiggle by a cycle due to noise.
 *
 *			cycl   0   1     2     3*    4     5     6     7     8 */
double envcyc[] = {0, 3.7, 24.8, 50.6, 73.0, 88.7, 97.4,100.0, 97.9};

/*
 * Initial LMS weight vector. The LMS algorithm is intended to deal with
 * cochannel CW interference, but is not yet fully developed.
 */
double wgtlms[] = {3.2, -2.3, -.17, .46, .29, .17, -.24, -.58, -.31};

/* end program */
