(*
   XDR.m3
   Marshalling routines for Sun RPC.
   David Nichols, Xerox PARC
   July, 1991

   $Id: XDR.m3,v 1.5 1992/03/31 01:21:26 nichols Exp $
*)

(* Copyright (c) 1991, 1992 Xerox Corporation.  All rights reserved.

   Use and copying of this software and preparation of derivative works
   based upon this software are permitted.  Any distribution of this
   software or derivative works must comply with all applicable United
   States export control laws.  This software is made available AS IS, and
   Xerox Corporation makes no warranty about the software, its performance
   or its conformity to any specification. *)

UNSAFE MODULE XDR;

IMPORT FloatMode, Text, TextF, Thread, Word, XDRFloat;

(*
 * Sources
 *)

PROCEDURE GetInteger (s: Source): INTEGER RAISES {Failed, Thread.Alerted} =
  BEGIN
    RETURN s.GetLong();
  END GetInteger;

PROCEDURE GetShort (s: Source): INTEGER RAISES {Failed, Thread.Alerted} =
  BEGIN
    RETURN s.GetLong();
  END GetShort;

PROCEDURE GetChar (s: Source): CHAR RAISES {Failed, Thread.Alerted} =
  BEGIN
    RETURN VAL(Word.Extract(s.GetLong(), 0, 8), CHAR);
  END GetChar;

PROCEDURE GetBoolean (s: Source): BOOLEAN RAISES {Failed, Thread.Alerted} =
  BEGIN
    RETURN s.GetLong() # 0;
  END GetBoolean;

PROCEDURE GetReal (s: Source): REAL RAISES {Failed, Thread.Alerted} =
  VAR i := s.GetLong();
  BEGIN
    TRY
      RETURN XDRFloat.WordToReal(i);
    EXCEPT
      FloatMode.Trap (f) =>
        RAISE Failed(
                NEW(FloatConversionFailure,
                    info := "Floating point unmarshalling failed.",
                    flag := f));
    END;
  END GetReal;

PROCEDURE GetLongReal (s: Source): LONGREAL
  RAISES {Failed, Thread.Alerted} =
  VAR high, low: INTEGER;
  BEGIN
    high := s.GetLong();
    low := s.GetLong();
    TRY
      RETURN XDRFloat.WordToLongReal(high, low);
    EXCEPT
      FloatMode.Trap (f) =>
        RAISE Failed(
                NEW(FloatConversionFailure,
                    info := "Floating point unmarshalling failed.",
                    flag := f));
    END;
  END GetLongReal;

PROCEDURE GetText (s: Source): TEXT RAISES {Failed, Thread.Alerted} =
  VAR
    length    : INTEGER;        (* length of the string *)
    t         : TEXT    := NIL; (* the result *)
    alignBytes: INTEGER;        (* number of alignment bytes *)
    buf: ARRAY [0 .. 3] OF CHAR;
  BEGIN
    length := s.GetLong();
    alignBytes := 3 - (length + 3) MOD 4;
    t := TextF.New(length);
    s.GetBytes(SUBARRAY(t^, 0, length));
    (* Realign. *)
    s.GetBytes(SUBARRAY(buf, 0, alignBytes));
    RETURN t;
  END GetText;

(* Get n bytes and realign. *)
PROCEDURE GetBytes (s: Source; VAR v: ARRAY OF CHAR)
  RAISES {Failed, Thread.Alerted} =
  VAR
    alignBytes                         := 3 - (NUMBER(v) + 3) MOD 4;
    buf       : ARRAY [0 .. 3] OF CHAR;
  BEGIN
    s.GetBytes(v);
    s.GetBytes(SUBARRAY(buf, 0, alignBytes));
  END GetBytes;

(*
 * Sinks
 *)

PROCEDURE PutInteger (s: Sink; v: INTEGER)
  RAISES {Failed, Thread.Alerted} =
  BEGIN
    s.PutLong(v);
  END PutInteger;

PROCEDURE PutShort (s: Sink; v: INTEGER) RAISES {Failed, Thread.Alerted} =
  BEGIN
    s.PutLong(v);
  END PutShort;

PROCEDURE PutChar (s: Sink; v: CHAR) RAISES {Failed, Thread.Alerted} =
  BEGIN
    s.PutLong(ORD(v));
  END PutChar;

PROCEDURE PutBoolean (s: Sink; v: BOOLEAN)
  RAISES {Failed, Thread.Alerted} =
  BEGIN
    IF v THEN s.PutLong(1); ELSE s.PutLong(0); END;
  END PutBoolean;

PROCEDURE PutReal (s: Sink; v: REAL) RAISES {Failed, Thread.Alerted} =
  BEGIN
    s.PutLong(XDRFloat.RealToWord(v));
  END PutReal;

PROCEDURE PutLongReal (s: Sink; v: LONGREAL)
  RAISES {Failed, Thread.Alerted} =
  VAR high, low: INTEGER;
  BEGIN
    XDRFloat.LongRealToWord(v, high, low);
    s.PutLong(high);
    s.PutLong(low);
  END PutLongReal;

PROCEDURE PutText (s: Sink; v: TEXT) RAISES {Failed, Thread.Alerted} =
  VAR
    len                                := Text.Length(v);
    alignBytes                         := 3 - (len + 3) MOD 4;
    buf       : ARRAY [0 .. 3] OF CHAR;
  BEGIN
    s.PutLong(len);
    s.PutBytes(SUBARRAY(v^, 0, len));
    s.PutBytes(SUBARRAY(buf, 0, alignBytes));
  END PutText;

PROCEDURE PutBytes (s: Sink; READONLY v: ARRAY OF CHAR)
  RAISES {Failed, Thread.Alerted} =
  VAR
    alignBytes                         := 3 - (NUMBER(v) + 3) MOD 4;
    buf       : ARRAY [0 .. 3] OF CHAR;
  BEGIN
    s.PutBytes(v);
    s.PutBytes(SUBARRAY(buf, 0, alignBytes));
  END PutBytes;

BEGIN
END XDR.
