/*************************************************************/
/*							     */
/*							     */
/*			  TANGOFSA.C			     */
/*							     */
/*							     */
/*************************************************************/
/*    Copyright 1989 Brown University -- John T. Stasko      */

#include  "tangolocal.h"
#include  "tangofsa.h"

int	      evt_to_proc();
int	      valid_evt();
int	      valid_proc();
void	      nfa_to_dfa();
DFASTATE_PTR  are_trans_from();
void	      make_eps_closure();
DFASTATE_PTR  already_a_state();
int	      trans_exists();
void	      add_actions();
void	      make_trans_table();

NFASTATE	 NFAstate[MAXSIZE];
int		 NFAstates_num = 0;
INTNODE_PTR	 NFAstart_states = NULL;

DFASTATE_PTR	 DFAstates = NULL;
int		 DFAstates_num = 0;

TRANSITION_PTR	 Transitions = NULL;

TRANSITION_PTR	 Trans_table[MAXSIZE][MAXSIZE];

int		 State;




/***************************************************************/
/*							       */
/*  read_mappings - read the mappings from the .anim file,     */
/*	create an NFA, create the corresponding DFA, then      */
/*	make a transition table.			       */
/*							       */
/***************************************************************/

int
read_mappings(fp)
   FILE *fp;
{
   unsigned long starthi,startlo;  /* sets of start and final NFA states */
   unsigned long finalhi,finallo;

   if (!evt_to_proc(fp,&starthi,&startlo,&finalhi,&finallo)) return(0);
   nfa_to_dfa(starthi,startlo,finalhi,finallo);
   make_trans_table();

   return(1);
}



/***************************************************************/
/*							       */
/*  evt_to_proc - read the algo ops to animation scene	       */
/*	definitions from the .anim file.  Create the non-det   */
/*	FSA states from them.				       */
/*							       */
/***************************************************************/

int
evt_to_proc(fp,starthi,startlo,finalhi,finallo)
   FILE *fp;
   unsigned long *starthi,*startlo,*finalhi,*finallo;
{
   int num;
   int legal_end,no_evts_yet,no_fcts_yet;
   int evtnum,fctnum;
   char s[STRINGLENGTH];
   char *str;
   char token[STRINGLENGTH];
   INTNODE_PTR ptail,pptr;

   num = 0;
   legal_end = 0;
   *startlo = 1; /* state 0 is a start state */
   *starthi = *finalhi = *finallo = 0;
   fgets(s,STRINGLENGTH,fp);  /* spit out the \n after the %% */
   if (Tango_debug)
      printf("With transition(s):\n");
   while (1)  /* get a new mapping */
      { num++;
	no_evts_yet = 1;
	if (!fgets(s,STRINGLENGTH,fp))	/* EOF */
	   { if (!legal_end)
		{ fprintf(stderr,"tango: premature EOF in mappings in .anim file\n");
		  fclose(fp);
		  return(0);
		}
	     else
		{ NFAstates_num = num;
		  return(1);
		}
	   }

	if (Tango_debug)
	   printf("    %s",s);
	str = s;
	while (1) /* get the algo evts */
	   { if (!(str = get_next_token(str,token)))  /* end of the line */
		{ if (!legal_end)
		     { fprintf(stderr,"tango: premature EOF in mappings in .anim file\n");
		       fclose(fp);
		       return(0);
		     }
		  else
		     { NFAstates_num = num;
		       return(1);
		     }
		}
	     if (strcmp(token,"->") == 0)
		{ if (no_evts_yet)
		     { fprintf(stderr,"tango: -> encountered before algo ops\n");
		       fclose(fp);
		       return(0);
		     }
		  else
		     { NFAstate[num].final = 1;
		       NFAstate[num].nextstate = -1;
		       NFAstate[num].trans = -1;
		       NFAstate[num].procs = NULL;
		       if (num < 32)
			  *finallo = *finallo | (1<<num);
		       else if (num <64)
			  *finalhi = *finalhi | (1<<num-32);
		     }
		  break;
		}
	     else
		{ if ((evtnum = valid_event(token)) >= 0)
		     { if (no_evts_yet)  /* a start state */
			  { if (num < 32)
			       *startlo = *startlo | (1<<num);
			    else if (num <64)
			       *starthi = *starthi | (1<<num-32);
			  }
		       NFAstate[num].final = 0;
		       NFAstate[num].trans = evtnum;
		       NFAstate[num].nextstate = num+1;
		       NFAstate[num].procs = NULL;
		       num++;
		       legal_end = 0;
		       no_evts_yet = 0;
		     }
		  else
		     { fprintf(stderr,"tango: unknown algorithm op %s encountered\n",token);
		       fclose(fp);
		       return(0);
		     }
		}
	   }  /* while */
	ptail = NULL;
	no_fcts_yet = 1;
	while (1)  /* get appropriate fcts to call */
	   { if (!(str = get_next_token(str,token)))  /* end of the line */
		{ if (no_fcts_yet)  /* "evt -> (nothing)" case */
		     NFAstate[num].procs = NULL;
		  legal_end = 1;
		  break;
		}
	     if ((fctnum = valid_proc(token)) >= 0)
		{ pptr = (INTNODE_PTR) malloc( sizeof( struct _INTNODE));
		  pptr->num = fctnum;
		  pptr->next = NULL;
		  if (!(NFAstate[num].procs))
		     NFAstate[num].procs = pptr;
		  else
		     ptail->next = pptr;
		  ptail = pptr;
		  no_fcts_yet = 0;
		}
	     else
		{ fprintf(stderr,"tango: unknown anim scene %s encountered\n",token);
		  fclose(fp);
		  return(0);
		}
	   }  /* while */
	} /* while */
}





/***************************************************************/
/*							       */
/*  valid_event - check the event name and see if it is a      */
/*	valid event entered earlier in the .anim file.	       */
/*	If so, return its number.  If not, return -1.	       */
/*							       */
/***************************************************************/

int
valid_event(name)
   char *name;
{
   int i;

   for (i=0; i<Num_algoevts; ++i)
      if (strcmp(name,Algoevts[i].name) == 0)
	 return(i);
   return(-1);
}





/***************************************************************/
/*							       */
/*  valid_proc - check the proc name and see if it is a        */
/*	valid animation procedure entered earlier in the       */
/*	.anim file.  If so, return its number.	If not, return */
/*	 -1.						       */
/*							       */
/***************************************************************/

int
valid_proc(name)
   char *name;
{
   int i;

   for (i=0; i<Num_animprocs; ++i)
      if (strcmp(name,Animprocs[i].name) == 0)
	 return(i);
   return(-1);
}





/***************************************************************/
/*							       */
/*  nfa_to_dfa - convert the NFA to a DFA.  Use the algo from  */
/*	the Dragon book.  The DFA emerges as a list of	       */
/*	transitions.					       */
/*							       */
/***************************************************************/

void
nfa_to_dfa(starthi,startlo,finalhi,finallo)
   unsigned long starthi,startlo,finalhi,finallo;
{
   DFASTATE_PTR     new_state,tostate,d;
   DFASTATE_PTR     marked,markedtail,unmarked,unmarkedtail;
   TRANSITION_PTR   new_trans,ttail = NULL;
   int		    i;
   INTNODE_PTR	    ip;

   if (Tango_debug)
      { printf("Start states %x %x\n",starthi,startlo);
	printf("Final states %x %x\n",finalhi,finallo);
      }

   marked = markedtail = unmarked = unmarkedtail = NULL;

   new_state = (DFASTATE_PTR) malloc( sizeof( struct _DFASTATE));
   new_state->lo = startlo;
   new_state->hi = starthi;
   new_state->next = NULL;
   unmarked = unmarkedtail = new_state;

   while (unmarked)
      { if (markedtail)
	   markedtail->next = unmarked;
	else
	   marked = unmarked;
	markedtail = unmarked;
	if (!unmarked->next)
	   unmarkedtail = NULL;
	unmarked = unmarked->next;
	markedtail->next = NULL;

	for (i=0; i<Num_algoevts; ++i)
	   { if (new_state = are_trans_from(markedtail,i))
		{ make_eps_closure(new_state,starthi,startlo,finalhi,finallo);
		  if (!(tostate = already_a_state(new_state,marked,unmarked)))
		     { if (unmarkedtail)
			  unmarkedtail->next = new_state;
		       else
			  unmarked = new_state;
		       unmarkedtail = new_state;
		       tostate = new_state;
		     }
		  else
		     free(new_state);

		  if (!(trans_exists(markedtail,tostate,i)))
		     { new_trans = (TRANSITION_PTR) malloc( sizeof( struct _TRANSITION));
		       new_trans->from = markedtail;
		       new_trans->to = tostate;
		       new_trans->transevt = i;
		       new_trans->next = NULL;
		       if (ttail)
			  ttail->next = new_trans;
		       else
			  Transitions = new_trans;
		       ttail = new_trans;
		       add_actions(new_trans,finalhi,finallo);
		     }
		} /* if */
	   } /* for */
      } /* while */

   if (Tango_debug)
      { printf("\nTransitions:\n\n");
	for (new_trans=Transitions; new_trans; new_trans=new_trans->next)
	   { printf("from state %x %x\n",new_trans->from->hi,new_trans->from->lo);
	     printf("to state %x %x\n",new_trans->to->hi,new_trans->to->lo);
	     printf("on algo op # %d\n",new_trans->transevt);
	     printf("with actions ");
	     for (ip=new_trans->actions; ip; ip=ip->next)
		printf("%d ",ip->num);
	     printf("\n\n\n");
	   }
      }

   for (d=marked,i=0; d; d=d->next,++i)
      d->num = i;

   DFAstates = marked;
   DFAstates_num = i;
}




/***************************************************************/
/*							       */
/*  are_trans_from - determine if there are any transitions    */
/*	of the given type from the given DFA state.  This      */
/*	DFA state is a set of NFA states.  If there are        */
/*	transitions, allocate a new DFA state with the set     */
/*	of NFA states to which the given state exhibits        */
/*	transitions.  Return this new state, or return NULL    */
/*	if there are no transitions.			       */
/*							       */
/***************************************************************/

DFASTATE_PTR
are_trans_from(fromstate,evt)
   DFASTATE_PTR fromstate;
   int evt;
{
   int i;
   unsigned long loset,hiset;
   DFASTATE_PTR new_state;

   loset = 0;
   hiset = 0;
   for (i=0; i<NFAstates_num; ++i)
      { if (i<32)
	   { if (((1 << i) & fromstate->lo) && (NFAstate[i].trans == evt))
		{ if (NFAstate[i].nextstate < 32)
		     loset = loset | (1 << NFAstate[i].nextstate);
		  else
		     hiset = hiset | (1 << NFAstate[i].nextstate-32);
		}
	   }
	else if (i<64)
	   { if (((1 << i-32) & fromstate->hi) && (NFAstate[i].trans == evt))
		{ if (NFAstate[i].nextstate < 32)
		     loset = loset | (1 << NFAstate[i].nextstate);
		  else
		     hiset = hiset | (1 << NFAstate[i].nextstate-32);
		}
	   }
      } /* for */

   if (loset || hiset)
      { new_state = (DFASTATE_PTR) malloc( sizeof( struct _DFASTATE));
	new_state->lo = loset;
	new_state->hi = hiset;
	new_state->next = NULL;
	return(new_state);
      }
   else
      return(NULL);
}




/***************************************************************/
/*							       */
/*  make_eps_closure - given a DFA state (set of NFA states)   */
/*	determine its epsilon-closure in the NFA.  For our     */
/*	simplified NFAs, we know that the only states having   */
/*	epsilon transitions that need to be added are final.   */
/*	Their epsilon-closure includes simply the set of       */
/*	start states.					       */
/*							       */
/***************************************************************/

void
make_eps_closure(state,starthi,startlo,finalhi,finallo)
   DFASTATE_PTR state;
   unsigned long starthi,startlo,finalhi,finallo;
{
   if ((state->lo & finallo) || (state->hi & finalhi))
      { state->lo = state->lo | startlo;
	state->hi = state->hi | starthi;
      }
}



/***************************************************************/
/*							       */
/*  already_a_state - check if the given DFA state is already  */
/*	on either the list of marked or unmarked DFA states.   */
/*							       */
/***************************************************************/

DFASTATE_PTR
already_a_state(state,marked,unmarked)
   DFASTATE_PTR state,marked,unmarked;
{
   DFASTATE_PTR s;

   for (s=marked; s; s=s->next)
      if ((state->lo == s->lo) && (state->hi == s->hi))
	 return(s);
   for (s=unmarked; s; s=s->next)
      if ((state->lo == s->lo) && (state->hi == s->hi))
	 return(s);
   return(NULL);
}






/***************************************************************/
/*							       */
/*  trans_exists - check if the given transition already       */
/*	exists in the list of valid DFA transitions.	       */
/*							       */
/***************************************************************/

int
trans_exists(from,to,evtnum)
   DFASTATE_PTR from,to;
   int evtnum;
{
   TRANSITION_PTR t;

   for (t=Transitions; t; t=t->next)
      { if ((t->from == from) && (t->to == to) && (t->transevt == evtnum))
	   return(1);
      }
   return(0);
}




/***************************************************************/
/*							       */
/*  add_actions - given a transition, add any actions that     */
/*	it should take (anim proc calls) if it occurs.	       */
/*	Basically, we check if its destination DFA state       */
/*	includes any NFA final states.	If so, add their       */
/*	corresponding animation procedure # calls.	       */
/*							       */
/***************************************************************/

void
add_actions(trans,finalhi,finallo)
   TRANSITION_PTR trans;
   unsigned long finalhi,finallo;
{
   int i;
   unsigned long l,h;
   INTNODE_PTR tail,ptr;

   tail = NULL;

   l = trans->to->lo & finallo;  /* does it include final states? */
   h = trans->to->hi & finalhi;

   if (l || h)
      { for (i=0; i<NFAstates_num; ++i)
	   if ( ((i<32) && ((1<<i)&l)) || ((i>=32) && ((1<<i-32)&h)) )
	      { ptr = (INTNODE_PTR) malloc( sizeof (struct _INTNODE));
		ptr->num = i;
		ptr->next = NULL;
		if (tail)
		   tail->next = ptr;
		else
		   trans->actions = ptr;
		tail = ptr;
	      }
      }
   else
      trans->actions = NULL;
}





/***************************************************************/
/*							       */
/*  make_trans_table - create the transition table that will   */
/*	be used to process incoming algorithm operations.      */
/*							       */
/***************************************************************/

void
make_trans_table()
{
   int i,j;
   TRANSITION_PTR t;

   for (i=0; i<DFAstates_num; ++i)
      for (j=0; j<Num_algoevts; ++j)
	 Trans_table[i][j] = NULL;

   for (t=Transitions; t; t=t->next)
      Trans_table[t->from->num][t->transevt] = t;

   State = 0;
}

