/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		xrscsyn.y						*/
/*									*/
/*	Syntax definitions for c xref scanning				*/
/*									*/
/************************************************************************/


#include "xrsc_local.h"

#define YYMAXDEPTH	500



/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


#define MAX_ARGS	128


static	Character	cur_filename[256];
static	Integer 	fct_line;
static	Integer 	fct_argct;
static	String		fct_args[MAX_ARGS];
static	String		fct_name;
static	String		last_name;



typedef struct _FILE_DATA *	FILE_DATA;
typedef struct _FILE_DATA {
   Integer line;
   Character name[256];
   FILE * file;
   FILE_DATA previous;
} FILE_DATA_INFO;

static	FILE_DATA	file_stack;



typedef struct _FCT_DATA *	FCT_DATA;
typedef struct _FCT_DATA {
   Integer line;
   Integer argct;
   String args[MAX_ARGS];
   String name;
   FCT_DATA previous;
} FCT_DATA_INFO;

static	FCT_DATA	fct_stack;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/

static	String		makeid();
static	void		push_function();
static	void		pop_function();



%}



%union {
   String	string;
   Integer	intval;
   XRSC_SCLASS	sclass;
}


%token			LX_APAREN LX_BPAREN LX_CPAREN
%token	<sclass>	LX_SCID
%token	<string>	LX_ID
%token			LX_STRING LX_NUM
%token			LX_STRUCT LX_UNION LX_ENUM LX_STMTWORD
%token			LX_EQOP
%token			LX_OTHER LX_ELLIPSES LX_NOT
%token			LX_SPECID LX_CLASS LX_COLCOL LX_PUBLIC

%type	<string>	sue_type structid unionid enumid classid




%%



/************************************************************************/
/*									*/
/*	Top level scanning						*/
/*									*/
/************************************************************************/


start	:
			{ XRSC_decl_begin(SCOPE_EXTERN,cur_filename,yylineno); }
		    program
			{ XRSC_decl_end(cur_filename,yylineno); }
	;


program :	/* empty */
			{ fct_name = NULL;
			  XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name);
			  fct_stack = NULL;
			}
	|	program prog_scan
			{ fct_name = NULL;
			  XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name);
			  fct_stack = NULL;
			}
	;



prog_scan :	data_decl_opt ';'
	|	fct_decl
	|	error
	;



/************************************************************************/
/*									*/
/*	Statement scanning (internal scanning)				*/
/*									*/
/*		stmt_scan -- scan start of statement/decl		*/
/*		non_data_scan -- internal starting with non-decl comp	*/
/*		other_scan -- scan internals				*/
/*		other_scan_nc -- other scan without commas		*/
/*									*/
/************************************************************************/


stmt_scan :	stmt_scan ';' stmt_scan_1
	|	stmt_scan_1
	;


stmt_scan_1 :	stmt_decl_scan
	|	LX_PUBLIC ':' stmt_scan_1
	|	error
	;



non_data_scan : /* empty */
	|	non_data_comp other_scan
	;


other_scan :	/* empty */
	|	other_scan other_comp
	|	other_scan comma
	;


other_scan_nc : /* empty */
	|	other_scan_nc other_comp
	;





/************************************************************************/
/*									*/
/*	Function declarations						*/
/*									*/
/************************************************************************/


fct_decl :	fct_decl_scan comma
			{ XRSC_decl_end(cur_filename,yylineno);
			  XRSC_expr_op();
			  pop_function();
			}
		    data_decl_2 ';'
	|	fct_decl_scan ';'
			{ XRSC_decl_end(cur_filename,yylineno);
			  XRSC_expr_op();
			  pop_function();
			}
	|	fct_decl_scan
			{ XRSC_decl_begin(SCOPE_INTERN,cur_filename,fct_line); }
		    arg_def_scan fct_body
			{ XRSC_decl_end(cur_filename,yylineno);
			  pop_function();
			}
	;


arg_def_scan :	/* empty */
			{ XRSC_decl_end(cur_filename,yylineno);
			  XRSC_decl_fct(cur_filename,fct_line,fct_name,fct_argct,fct_args);
			}
	|	data_decl ';' arg_def_scan
	;


fct_body :	'{'
			{ XRSC_decl_begin(SCOPE_INTERN,cur_filename,yylineno); }
		    stmt_scan '}'
			{ XRSC_decl_scope_use();
			  XRSC_decl_end(cur_filename,yylineno);
			}
	;





/************************************************************************/
/*									*/
/*	Data declarations						*/
/*									*/
/*		data_decl_opt -- optional normal data declaration	*/
/*		data_decl -- normal data declaration			*/
/*		arg_list -- inside argument list			*/
/*									*/
/************************************************************************/


stmt_decl_scan : stmt_decl_1
	;


stmt_decl_1 :	data_decl_scan stmt_decl_2
	|
			{ XRSC_decl_scan_invalid(); }
		non_data_scan
	;


stmt_decl_2 :	comma stmt_decl_1
	|	rhs_expr stmt_decl_3
	|
			{ XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name); }
		    non_data_scan
	;


stmt_decl_3 :	/* empty */
			{ XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name); }
	|	comma stmt_decl_1
	;


data_decl_opt : /* empty */
	|	data_decl
	;


data_decl :	data_decl_0
			{ XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name); }
	;


data_decl_0 :	data_decl_1
	|	data_decl_1 comma data_decl_2
	;



data_decl_1 :	data_decl_scan rhs_expr_opt
	;


data_decl_2 :	data_decl_3
	|	data_decl_3 comma data_decl_2
	;


data_decl_3 :	decl_scan rhs_expr_opt
	;


arg_list :	/* empty */
	|	arg_list_1
			{ XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name); }
	;


arg_list_1 :	arg_list_2
	|	arg_list_2 comma arg_list_1
	;


arg_list_2 :	data_decl_1
			{ fct_args[fct_argct++] = last_name;
			  XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name);
			}
	|	LX_ELLIPSES
			{ fct_args[fct_argct++] = "..."; }
	;


old_arg_list :	LX_ID
			{ fct_args[fct_argct++] = $1; }
	|	old_arg_list comma LX_ID
			{ fct_args[fct_argct++] = $3; }
	;


rhs_expr_opt :	/* empty */
	|	rhs_expr
	;


rhs_expr :	'='
			{ XRSC_decl_begin(SCOPE_NONE,cur_filename,yylineno);
			  XRSC_expr_assignop();
			}
		    other_scan_nc
			{ XRSC_decl_end(cur_filename,yylineno); }
	;



/************************************************************************/
/*									*/
/*	Declaration scanning						*/
/*									*/
/*		fct_decl_scan -- scan for function			*/
/*		data_decl_scan -- scan for data 			*/
/*		decl_scan -- scan for anything				*/
/*									*/
/************************************************************************/


fct_decl_scan : dd_scan_0 decl_comp_3
	;


data_decl_scan : dd_scan_0
			{ last_name = XRSC_decl_inq_name();
			  XRSC_decl_finish(FINISH_COMMA,cur_filename,yylineno,fct_name);
			}
	|	dd_scan_3
	;


dd_scan_0 :	dd_scan_1 decl_comp_2 dd_scan_2
	;


dd_scan_1 :	/* empty */
	|	dd_scan_3
	;


dd_scan_3 :	decl_comp_1
	|	dd_scan_3 decl_comp_1
	;


dd_scan_2 :	/* empty */
	|	dd_scan_2 decl_comp_1
	|	dd_scan_2 decl_comp_2
	|	dd_scan_2 decl_comp_5
	;


decl_scan :	decl_scan_0
			{ XRSC_decl_finish(FINISH_COMMA,cur_filename,yylineno,fct_name); }
	;


decl_scan_0 :	/* empty */
	|	decl_comp_1 decl_scan
	|	decl_comp_2 decl_scan_1
	;


decl_scan_1 :	/* empty */
	|	decl_comp_4
	|	decl_comp_1 decl_scan_1
	|	decl_comp_2 decl_scan_1
	;




/************************************************************************/
/*									*/
/*	Declaration components						*/
/*									*/
/*		decl_comp_1 -- basic components 			*/
/*		decl_comp_2 -- identifier				*/
/*		decl_comp_3 -- arg list 				*/
/*		decl_comp_4 -- arg type list				*/
/*									*/
/************************************************************************/



decl_comp_4 :	left_paren
			{ XRSC_expr_call(fct_name);
			  XRSC_expr_push();
			}
		    other_scan ')'
			{ XRSC_expr_pop();
			  XRSC_decl_scan_type("()");
			}
	;


decl_comp_3 :	'('
			{ push_function();
			  fct_line = yylineno;
			  fct_argct = 0;
			  fct_name = XRSC_decl_inq_name();
			  XRSC_decl_scan_type("()");
			  XRSC_decl_set_fclass();
			  XRSC_decl_finish(FINISH_COMMA,cur_filename,yylineno,fct_stack->name);
			  XRSC_decl_begin(SCOPE_ARGS,cur_filename,yylineno);
			}
		    arg_list ')'
	|	LX_BPAREN
			{ push_function();
			  fct_line = yylineno;
			  fct_argct = 0;
			  fct_name = XRSC_decl_inq_name();
			  XRSC_decl_scan_type("()");
			  XRSC_decl_set_fclass();
			  XRSC_decl_finish(FINISH_COMMA,cur_filename,yylineno,fct_name,fct_stack->name);
			  XRSC_decl_begin(SCOPE_ARGS,cur_filename,yylineno);
			}
		    old_arg_list ')'
	;



decl_comp_5 :	LX_APAREN				/* APAREN = ( w/*id) after */
			{ XRSC_decl_scan_type("(");
			  XRSC_expr_push();
			  XRSC_expr_op();
			}
		    decl_scan ')'
			{ XRSC_decl_scan_type(")");
			  XRSC_expr_pop();
			}
	|	LX_CPAREN				/* CPAREN = ( w/*id)( after */
			{ XRSC_decl_scan_type("(");
			  XRSC_expr_push();
			  XRSC_expr_op();
			}
		    decl_scan ')' left_paren other_scan ')'
			{ XRSC_decl_scan_type(")");
			  XRSC_expr_pop();
			  XRSC_decl_scan_type("()");
			}
	;



decl_comp_2 :	LX_ID
			{ XRSC_decl_scan_id($1,yylineno);
			  XRSC_expr_id(cur_filename,yylineno,$1);
			  SFREE($1);
			}
	|	LX_ID LX_COLCOL LX_ID
			{ XRSC_decl_scan_id($3,yylineno);
			  XRSC_expr_id(cur_filename,yylineno,$3);
			  SFREE($1);
			  SFREE($3);
			}
	|	LX_ID LX_COLCOL LX_NOT LX_ID
			{ XRSC_decl_scan_id(makeid("~",$4),yylineno);
			  XRSC_expr_op();
			  XRSC_expr_id(cur_filename,yylineno,$4);
			  SFREE($1);
			  SFREE($4);
			}
	|	LX_NOT LX_ID
			{ XRSC_decl_scan_id(makeid("~",$2),yylineno);
			  XRSC_expr_op();
			  XRSC_expr_id(cur_filename,yylineno,$2);
			  SFREE($2);
			}
	|	sue_type
			{ XRSC_decl_scan_id($1,yylineno);
			  SFREE($1);
			  XRSC_expr_op();
			}
	;


decl_comp_1 :	LX_SCID
			{ XRSC_decl_set_sclass($1);
			  XRSC_expr_op();
			}
	|	'*'
			{ XRSC_decl_scan_type("*");
			  XRSC_expr_op();
			}
	|	'['
			{ XRSC_decl_begin(SCOPE_NONE,cur_filename,yylineno);
			  XRSC_expr_push();
			}
		    other_scan ']'
			{ XRSC_decl_end(cur_filename,yylineno);
			  XRSC_decl_scan_type("[]");
			  XRSC_expr_pop();
			}
	|	LX_SPECID
	|	LX_ELLIPSES
	;




/************************************************************************/
/*									*/
/*	Other components						*/
/*									*/
/************************************************************************/


other_comp :	LX_ID
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  SFREE($1);
			}
	|	'*'
			{ XRSC_expr_op(); }
	|	'['
			{ XRSC_expr_push(); }
		    other_scan ']'
			{ XRSC_expr_pop(); }
	|	sue_type
			{ SFREE($1);
			  XRSC_expr_op();
			}
	|	'='
			{ XRSC_expr_assignop(); }
	|	LX_COLCOL
			{ XRSC_decl_scan_invalid(); }
	|	LX_SCID
			{ XRSC_expr_op(); }
	|	LX_SPECID
			{ XRSC_expr_op(); }
	|	LX_NOT
			{ XRSC_expr_op(); }
	|	LX_ELLIPSES
	|	non_data_comp
	;


non_data_comp : left_paren
			{ XRSC_expr_call(fct_name);
			  XRSC_expr_push();
			}
		    stmt_scan ')'
		       { XRSC_expr_pop(); }
	|	LX_APAREN
			{ XRSC_expr_call(fct_name);
			  XRSC_expr_push();
			}
		    other_scan ')'
			{ XRSC_expr_pop(); }
	|	LX_CPAREN
			{ XRSC_expr_call(fct_name);
			  XRSC_expr_push();
			}
		    other_scan ')'
			{ XRSC_expr_pop(); }
	|	non_decl_comp
	;



non_decl_comp : LX_STMTWORD
			{ XRSC_expr_op(); }
	|	LX_STRING
			{ XRSC_expr_op(); }
	|	LX_NUM
			{ XRSC_expr_op(); }
	|	operator_1
			{ XRSC_expr_op(); }
	|	scope_comp
	;



operator_1 :	LX_OTHER
	|	':'
	|	'~'
	|	LX_EQOP
			{ XRSC_expr_assignop(); }
	;



scope_comp :	'{'
			{ XRSC_decl_begin(SCOPE_INTERN,cur_filename,yylineno);
			  XRSC_expr_op();
			}
		    stmt_scan '}'
			{ XRSC_decl_end(cur_filename,yylineno);
			  XRSC_expr_op();
			}
	;





/************************************************************************/
/*									*/
/*	Struct/union/enum/class definitions				*/
/*									*/
/************************************************************************/


sue_type :	LX_STRUCT '{' sudefs
			{ $$ = makeid("struct ?",NULL); }
	|	structid
			{ $$ = $1; }
	|	structid '{' { XRSC_decl_sue_id(cur_filename,yylineno,$1); } sudefs
			{ $$ = $1; }
	|	LX_UNION '{' sudefs
			{ $$ = makeid("union ?",NULL); }
	|	unionid
			{ $$ = $1; }
	|	unionid '{' { XRSC_decl_sue_id(cur_filename,yylineno,$1); } sudefs
			{ $$ = $1; }
	|	LX_ENUM '{' endefs
			{ $$ = makeid("enum ?",NULL); }
	|	enumid
			{ $$ = $1; }
	|	enumid '{' { XRSC_decl_sue_id(cur_filename,yylineno,$1); } endefs
			{ $$ = $1; }
	|	LX_CLASS '{' sudefs
			{ $$ = makeid("class ?",NULL); }
	|	classid
			{ $$ = $1; }
	|	classid '{' { XRSC_decl_sue_id(cur_filename,yylineno,$1); } sudefs
			{ $$ = $1; }
	;


structid :	LX_STRUCT LX_ID
			{ $$ = makeid("struct ",$2);
			  SFREE($2);
			}
	|	LX_STRUCT LX_ID ':' public_opt LX_ID
			{ $$ = makeid("struct ",$2);
			  SFREE($2);
			  SFREE($5);
			}
	;


unionid :	LX_UNION LX_ID
			{ $$ = makeid("union ",$2);
			  SFREE($2);
			}
	|	LX_UNION LX_ID ':' public_opt LX_ID
			{ $$ = makeid("union ",$2);
			  SFREE($2);
			  SFREE($5);
			}
	;


enumid	:	LX_ENUM LX_ID
			{ $$ = makeid("enum ",$2);
			  SFREE($2);
			}
	;


classid :	LX_CLASS LX_ID
			{ $$ = makeid("class ",$2);
			  SFREE($2);
			}
	|	LX_CLASS LX_ID ':' public_opt LX_ID
			{ $$ = makeid("class ",$2);
			  SFREE($2);
			  SFREE($5);
			}
	;


sudefs	:
			{ XRSC_decl_begin(SCOPE_SUE,cur_filename,yylineno); }
		    stmt_scan '}'
			{ XRSC_decl_end(cur_filename,yylineno); }
	;


endefs	:
			{ XRSC_decl_begin(SCOPE_SUE,cur_filename,yylineno); }
		    en_list optcom '}'
			{ XRSC_decl_end(cur_filename,yylineno); }
	;


en_list :	en_term
	|	en_list comma en_term
	;


en_term :	LX_ID
			{ XRSC_decl_enum(cur_filename,yylineno,$1);
			  SFREE($1);
			}
		   rhs_expr_opt
	;


optcom	:	/* empty */
	|	comma
	;


public_opt :	/* empty */
	|	LX_PUBLIC
	;


left_paren :	'('
	|	LX_BPAREN
	;


comma	:	','
			{ XRSC_expr_op(); }
	;


%%


#include "xrsclex.c"





/************************************************************************/
/*									*/
/*	XRSC_parse -- parse a file					*/
/*									*/
/************************************************************************/


void
XRSC_parse(file,name,inf)
   String file;
   String name;
   FILE * inf;
{
   file_stack = NULL;
   if (file == NULL) {
      XRSC__use_cpp = FALSE;
      yyin = inf;
      file = name;
    }
   else yyin = fopen(file,"r");

   fct_name = NULL;
   fct_line = 0;
   fct_argct = 0;
   last_name = NULL;

   if (yyin != NULL) {
      XRSC_find_file_line(file);
      XRSC_set_file(file,1);
      yyparse();
    }
   else {
      if (XRSC__interact) fprintf(stderr,"Can't open file \"%s\"\n",file);
    };
};




/************************************************************************/
/*									*/
/*	yyerror -- handle errors					*/
/*									*/
/************************************************************************/


yyerror()
{
   if (XRSC__interact)
      fprintf(stderr,"syntax error at %d in %s\n",yylineno,cur_filename);
};



/************************************************************************/
/*									*/
/*	XRSC_set_file -- set file and line				*/
/*									*/
/************************************************************************/


static void
XRSC_set_file(file,line)
   String file;
   Integer line;
{
   strcpy(cur_filename,file);
   yylineno = line;
   fct_line = line;		/* don't allow fct line and file to differ */
};





/************************************************************************/
/*									*/
/*	XRSC_push_file -- push a new file onto the stack		*/
/*									*/
/************************************************************************/


static void
XRSC_push_file(file,glbl)
   String file;
   Boolean glbl;
{
   FILE * newf;
   Character nfil[256];
   FILE_DATA fd;

   if (XRSC_find_file(file,glbl,cur_filename,nfil)) return;

   XRSC_out_header(nfil);

   newf = fopen(nfil,"r");
   if (newf == NULL) {
      if (XRSC__interact) fprintf(stderr,"Can't open include file %s\n",nfil);
      return;
    };

   fd = PALLOC(FILE_DATA_INFO);
   strcpy(fd->name,cur_filename);
   fd->line = yylineno;
   fd->file = yyin;
   fd->previous = file_stack;
   file_stack = fd;

   strcpy(cur_filename,nfil);

   yylineno = 1;
   yyin = newf;
};





yywrap()
{
   FILE_DATA fd;

   if (file_stack == NULL) return 1;

   fclose(yyin);

   fd = file_stack;
   file_stack = fd->previous;
   yyin = fd->file;
   yylineno = fd->line;
   strcpy(cur_filename,fd->name);

   free(fd);

   return 0;
};





/************************************************************************/
/*									*/
/*	makeid -- make a new id from two old ones			*/
/*									*/
/************************************************************************/


static String
makeid(s1,s2)
   String s1;
   String s2;
{
   Character buf[256];
   String s;

   if (s2 == NULL) s2 = "";

   sprintf(buf,"%s%s",s1,s2);

   for (s = buf; *s != 0; ++s) {
      if (*s == ' ') *s = SPACE_CHARACTER;
    };

   return SALLOC(buf);
};





/************************************************************************/
/*									*/
/*	push_function -- handle nested function definitions		*/
/*	pop_function -- restore previous				*/
/*									*/
/************************************************************************/


static void
push_function()
{
   FCT_DATA fd;
   Integer i;

   fd = PALLOC(FCT_DATA_INFO);

   fd->line = fct_line;
   fd->argct = fct_argct;
   for (i = 0; i < fd->argct; ++i) fd->args[i] = fct_args[i];
   fd->name = fct_name;
   fd->previous = fct_stack;
   fct_stack = fd;
};





static void
pop_function()
{
   FCT_DATA fd;
   Integer i;

   fd = fct_stack;
   if (fd == NULL) {
      fct_line = 0;
      fct_argct = 0;
      fct_name = NULL;
    }
   else {
      fct_stack = fd->previous;
      fct_line = fd->line;
      fct_argct = fd->argct;
      for (i = 0; i < fd->argct; ++i) fct_args[i] = fd->args[i];
      fct_name = fd->name;
      free(fd);
    };
};





/* end of xrscsyn.y */


