/*	Copyright 1989 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		xrscpsyn.y						*/
/*									*/
/*	Syntax definitions for C++ xref scanning			*/
/*									*/
/************************************************************************/


#include "xrsc_local.h"


/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define CC_COMMAND	"CC"
#define FILT_COMMAND	"ddtfilter"





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


#define MAX_ARGS	128


static	Character	cur_filename[256];
static	String		cls_name;
static	String		cls_ctype;
static	Integer 	cls_line;
static	Integer 	cls_flags;
static	String		last_name;

static	String		ccname = NULL;
static	String		filtname = NULL;





typedef struct _FILE_DATA *	FILE_DATA;
typedef struct _FILE_DATA {
   Integer line;
   Character name[256];
   FILE * file;
   FILE_DATA previous;
   Boolean ignore;
} FILE_DATA_INFO;

static	FILE_DATA	file_stack;



typedef struct _CLS_DATA *	CLS_DATA;
typedef struct _CLS_DATA {
   String name;
   String ctype;
   Integer line;
   Integer flags;
   CLS_DATA previous;
} CLS_DATA_INFO;

static	CLS_DATA	cls_stack;



/************************************************************************/
/*									*/
/*	Flag values							*/
/*									*/
/************************************************************************/


#define IS_PUBLIC	0x1
#define IS_VIRTUAL	0x2
#define IS_STATIC	0x4
#define IS_FRIEND	0x8
#define IS_FUNCTION	0x10
#define IS_INLINE	0x20
#define IS_PRIVATE	0x40
#define IS_PROTECTED	0x80
#define IS_PURE 	0x100
#define IS_CONST	0x200
#define IS_IGNORE	0x400

#define TEST_PUBLIC(f)	(((f)&IS_PUBLIC) != 0)
#define TEST_VIRTUAL(f) (((f)&IS_VIRTUAL) != 0)





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		push_class();
static	void		pop_class();
static	void		begin_members();
static	void		finish_member();
static	void		scan_result();
static	String		command_name();



%}



%union {
   String	string;
   Integer	intval;
}


%token	<string>	LX_ID
%token			LX_VIRTUAL LX_PRIVATE LX_PUBLIC LX_PROTECTED
%token			LX_STATIC LX_EXTERN LX_CONST LX_VOLATILE LX_INLINE
%token			LX_VIRTUAL LX_FRIEND LX_CLASS LX_STRUCT LX_UNION LX_ENUM
%token			LX_COLCOL LX_APAREN LX_FCLASS LX_PURE LX_TYPEDEF


%%



program :	/* empty */
	|	program class_def
	;


class_def :	class_spec
	|	class_key id_opt ';'
	|	error
	;


class_spec  :	class_head '{'
			{ if (STRNEQ(cls_name,"*PRIVATE*"))
			     XRSC_out_decl(cur_filename,yylineno,cls_name,"",
					      0,SCLASS_CLASSID,0);
			}
		    member_list '}'
	;


class_head :	class_key id_opt base_spec_opt
	;


class_key :	LX_CLASS
			{ cls_ctype = "class"; }
	|	LX_STRUCT
			{ cls_ctype = "struct"; }
	|	LX_UNION
			{ cls_ctype = "union"; }
	;


id_opt	:	/* empty */
			{ cls_name = "*PRIVATE*"; }
	|	LX_ID
			{ cls_name = $1; }
	;


base_spec_opt : /* empty */
	|	':' base_list
	;


base_list :	base_spec
	|	base_list ',' base_spec
	;


base_spec :	base_acc_spec_list LX_ID
			{ XRSC_out_hier(cur_filename,yylineno,cls_name,$2,
						TEST_PUBLIC(cls_flags),
						TEST_VIRTUAL(cls_flags),
						FALSE);
			}
	;


base_acc_spec_list : /* empty */
			{ cls_flags = (STREQL(cls_ctype,"class") ? 0 : IS_PUBLIC); }
	|	base_acc_spec_list base_acc_spec
	;


base_acc_spec : LX_VIRTUAL
			{ cls_flags |= IS_VIRTUAL; }
	|	LX_PRIVATE
			{ cls_flags &= ~IS_PUBLIC; }
	|	LX_PUBLIC
			{ cls_flags |= IS_PUBLIC; }
	;


member_list :	member_scan
	;

member_scan :	/* empty */
			{ begin_members(); }
	|	member_scan member_elt
	;


member_elt :	name
	|	LX_STATIC
			{ cls_flags |= IS_STATIC; }
	|	LX_TYPEDEF
			{ cls_flags |= IS_IGNORE; }
	|	LX_EXTERN
	|	LX_CONST
			{ cls_flags |= IS_CONST; }
	|	LX_VOLATILE
	|	LX_INLINE
	|	LX_VIRTUAL
			{ cls_flags |= IS_VIRTUAL; }
	|	LX_FCLASS LX_ID ';'
			{ finish_member(TRUE);
			  XRSC_out_hier(cur_filename,yylineno,cls_name,$2,
					   FALSE,FALSE,TRUE);
			}
	|	LX_FRIEND
			{ cls_flags |= IS_FRIEND; }
	|	class_def
	|	enum_def
	|	LX_APAREN member_scan ')'
	|	'=' ';'
			{ finish_member(TRUE); }
	|	LX_PURE
			{ cls_flags |= IS_PURE;
			  finish_member(TRUE);
			}
	|	'(' nested_class ')'
			{ cls_flags |= IS_FUNCTION;
			  cls_flags &= ~IS_CONST;
			}
	|	'[' nested_class ']'
	|	'{' nested_class '}'
			{ cls_flags |= IS_INLINE;
			  finish_member(TRUE);
			}
	|	','
			{ finish_member(FALSE); }
	|	';'
			{ finish_member(TRUE); }
	|	LX_PRIVATE
			{ cls_flags &= ~(IS_PUBLIC|IS_PROTECTED|IS_PRIVATE);
			  cls_flags |= IS_PRIVATE;
			};
	|	LX_PUBLIC
			{ cls_flags &= ~(IS_PUBLIC|IS_PROTECTED|IS_PRIVATE);
			  cls_flags |= IS_PUBLIC;
			};
	|	LX_PROTECTED
			{ cls_flags &= ~(IS_PUBLIC|IS_PROTECTED|IS_PRIVATE);
			  cls_flags |= IS_PROTECTED;
			};
	|	'*'
	|	'&'
	|	':'
	;


name	:	LX_ID
			{ last_name = $1;
			  cls_line = yylineno;
			}
	|	LX_ID LX_COLCOL name
	;


enum_def :	LX_ENUM
	|	LX_ENUM LX_ID
	|	LX_ENUM '{' '}'
	|	LX_ENUM LX_ID '{' '}'
	;


nested_class :	/* empty */
	|	{ push_class(); } nested_class_list { pop_class(); }
	;


nested_class_list :	class_def
	|	nested_class_list class_def
	;


%%


#include "xrscplex.c"




/************************************************************************/
/*									*/
/*	XRSC_parse -- parse a file					*/
/*									*/
/************************************************************************/


void
XRSC_parse(file,name,inf)
   String file;
   String name;
   FILE * inf;
{
   scan_result(file,name);
   XRSC__level = 1;

   file_stack = NULL;
   if (file == NULL) {
      yyin = inf;
      file = name;
    }
   else yyin = fopen(file,"r");

   if (yyin != NULL) {
      XRSC_find_file_line(file);
      XRSC_set_file(file,1);
      clevel = 0;
      bracect = 0;
      yyparse();
    }
   else {
      if (XRSC__interact) fprintf(stderr,"Can't open file \"%s\"\n",file);
    };
};




/************************************************************************/
/*									*/
/*	yyerror -- handle errors					*/
/*									*/
/************************************************************************/


yyerror()
{
   if (XRSC__interact)
      fprintf(stderr,"syntax error at %d in %s\n",yylineno,cur_filename);
};



/************************************************************************/
/*									*/
/*	XRSC_set_file -- set file and line				*/
/*									*/
/************************************************************************/


static void
XRSC_set_file(file,line)
   String file;
   Integer line;
{
   strcpy(cur_filename,file);
   yylineno = line;
};





/************************************************************************/
/*									*/
/*	XRSC_push_file -- push a new file onto the stack		*/
/*									*/
/************************************************************************/


static void
XRSC_push_file(file,glbl)
   String file;
   Boolean glbl;
{
   FILE * newf;
   Character nfil[256];
   FILE_DATA fd;

   newf = fopen(nfil,"r");
   if (newf == NULL) {
      if (XRSC__interact) fprintf(stderr,"Can't open include file %s\n",nfil);
      return;
    };

   fd = PALLOC(FILE_DATA_INFO);
   strcpy(fd->name,cur_filename);
   fd->line = yylineno;
   fd->file = yyin;
   fd->previous = file_stack;
   fd->ignore = XRSC__ignore;
   file_stack = fd;

   strcpy(cur_filename,nfil);
   yylineno = 1;
   yyin = newf;

   if (XRSC_find_file(file,glbl,cur_filename,nfil)) {
      XRSC__ignore = TRUE;
    };
};





yywrap()
{
   FILE_DATA fd;

   if (file_stack == NULL) return 1;

   fclose(yyin);

   fd = file_stack;
   file_stack = fd->previous;
   yyin = fd->file;
   yylineno = fd->line;
   strcpy(cur_filename,fd->name);
   XRSC__ignore = fd->ignore;

   free(fd);

   return 0;
};





/************************************************************************/
/*									*/
/*	push_class -- handle nested class definitions			*/
/*	pop_class -- restore previous					*/
/*									*/
/************************************************************************/


static void
push_class()
{
   CLS_DATA cd;
   Integer i;

   cd = PALLOC(CLS_DATA_INFO);

   cd->name = cls_name;
   cd->line = cls_line;
   cd->flags = cls_flags;
   cd->ctype = cls_ctype;
   cd->previous = cls_stack;
   cls_stack = cd;
};





static void
pop_class()
{
   CLS_DATA cd;
   Integer i;

   cd = cls_stack;
   if (cd == NULL) {
      cls_name = NULL;
      cls_line = 0;
      cls_flags = 0;
      cls_ctype = NULL;
    }
   else {
      cls_stack = cd->previous;
      cls_name = cd->name;
      cls_line = cd->line;
      cls_flags = cd->flags;
      cls_ctype = cd->ctype;
      free(cd);
    };
};





/************************************************************************/
/*									*/
/*	begin_members -- start scanning members of class		*/
/*	finish_member -- finish scanning current member 		*/
/*									*/
/************************************************************************/


static void
begin_members()
{
   if (STREQL(cls_ctype,"class")) {
      cls_flags = IS_PRIVATE;
    }
   else cls_flags = IS_PUBLIC;

   last_name = NULL;
};





static void
finish_member(done)
   Boolean done;
{
   String prot;
   Boolean isdata,isinline,isfrnd,isvirt,isstat,ispure,iscon,ignore;

   if (cls_flags & IS_PUBLIC) prot = "PUBLIC";
   else if (cls_flags & IS_PRIVATE) prot = "PRIVATE";
   else if (cls_flags & IS_PROTECTED) prot = "PROTECTED";
   else prot = "NONE";

   isdata = ((cls_flags & IS_FUNCTION) == 0);
   if (isdata) isinline = FALSE;
   else isinline = ((cls_flags & IS_INLINE) != 0);
   if (isdata) isfrnd = FALSE;
   else isfrnd = ((cls_flags & IS_FRIEND) != 0);
   if (isdata) isvirt = FALSE;
   else isvirt = ((cls_flags & IS_VIRTUAL) != 0);
   if (isdata) isstat = ((cls_flags & IS_STATIC) != 0);
   else isstat = FALSE;
   if (isdata) ispure = FALSE;
   else ispure = ((cls_flags & IS_PURE) != 0);
   iscon = ((cls_flags & IS_CONST) != 0);
   ignore = ((cls_flags & IS_IGNORE) != 0);

   if (isfrnd) prot = "PUBLIC";

   if (last_name != NULL && !ignore) {
      XRSC_out_memb(cur_filename,cls_line,cls_name,last_name,prot,
		       isdata,isinline,isfrnd,isvirt,isstat,ispure,iscon);
    };

   last_name = NULL;

   if (done) {
      cls_flags &= (IS_PUBLIC|IS_PRIVATE|IS_PROTECTED);
    };
};





/************************************************************************/
/*									*/
/*	scan_result -- run CC and scan the result			*/
/*									*/
/************************************************************************/


static void
scan_result(file,name)
   String file;
   String name;
{
   Character cmd[10240],buf[1024];
   Integer i;
   String s;

   if (ccname == NULL) {
      ccname = CC_COMMAND;
      s = getenv("CC_COMMAND");
      if (s != NULL) ccname = s;
      filtname = FILT_COMMAND;
      s = getenv("FILT_COMMAND");
      if (s != NULL) filtname = s;
    };

   s = getenv("XREF_CFRONT");
   if (s != NULL && *s != 0) sprintf(cmd,"cfrontC=%s ",s);
   else cmd[0] = 0;

   strcat(cmd,ccname);
   strcat(cmd," +e1 +d -c -F");
   if (s != NULL) strcat(cmd," +X");

#ifndef NOANSI
   strcat(cmd," +a1");
#endif

   if (XRSC__comp_flags != NULL) {
      strcat(cmd," ");
      strcat(cmd,XRSC__comp_flags);
    };

   for (i = 0; i < XRSC__num_include; ++i) {
      strcat(cmd," -I");
      strcat(cmd,XRSC__include[i]);
    };

   strcat(cmd," ");
   strcat(cmd,name);

   command_name("xrefscan",buf);
   strcat(cmd," | ");
   strcat(cmd,buf);
   if (XRSC__level > 0) strcat(cmd," -l");
   strcat(cmd," -p -C -a -N");
   strcat(cmd,name);

   if (!XRSC__interact) strcat(cmd," -b");
   if (XRSC__scanfile != NULL) {
      strcat(cmd," -S");
      strcat(cmd,XRSC__scanfile);
    };

   strcat(cmd," - | ");

   command_name(filtname,buf);
   strcat(cmd,buf);

   sprintf(buf," -c%c",SPACE_CHARACTER);
   strcat(cmd,buf);

   system(cmd);
};





/************************************************************************/
/*									*/
/*	command_name -- get command name to execute			*/
/*									*/
/************************************************************************/


static String
command_name(name,buf)
   String name;
   String buf;
{
   Boolean experiment;

   experiment = FALSE;
   if (getenv("FIELD_DEBUG") != NULL) experiment = TRUE;

   if (name[0] == '/') {
      strcpy(buf,name);
    }
   else {
      sprintf(buf,"%s/bin/field/%s",BWElocal_project(),name);
      if (experiment || access(buf,1) < 0) {
	 sprintf(buf,"%s/bin/field/%s/%s",BWElocal_project(),BWEarch(),name);
	 if (experiment || access(buf,1) < 0) {
	    sprintf(buf,"%s/field/cmd/%s/%s",BWElocal_project(),BWEarch(),name);
	  };
       };
   };

   return buf;
};






/* end of xrscpsyn.y */



