/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		xrsclex.l						*/
/*									*/
/*	Lexical definitions for cross reference scanner 		*/
/*									*/
/************************************************************************/




#undef YYLMAX
#define YYLMAX		20480


static	Integer 	scan_id();
static	Integer 	handle_num();
static	Integer 	handle_string();
static	void		set_file_line();
static	void		include_file();
static	void		scan_vptr();
static	void		scan_vtbl();


static	String		cur_macro = NULL;

static	String	stmtwords[] = {
   "break",
   "case",
   "continue",
   "default",
   "do",
   "else",
   "for",
   "goto",
   "if",
   "return",
   "switch",
   "while",
   0
};



%}

%a 10240
%o 10240
%p 10240
%e 10240
%n 4096

%START PREP PRED PREX PREI PREL


WHITE	([\ \t\f])+
WHITE0	([\ \t\f])*

INT	([0-9]+)
HEXINT	(0x[0-9a-f]+)

STRING	(\"([^\n"\\]|(\\.)|(\\\n))*\")

ID	([a-zA-Z_$][a-zA-Z_$0-9]*)
FILE	([-0-9/.a-zA-Z_$+]+)

CHAR0	(\'.\')
CHAR1	(\'\\.\')
CHAR2	(\'\\([0-7]{1,3})\')
CHAR	({CHAR0}|{CHAR1}|{CHAR2})

FLT0	([0-9]+\.[0-9]*)
EXP	([Ee][-+]?[0-9]+)
FLT1	({INT}{EXP})
FLT2	({FLT0}{EXP})
FLT	({FLT0}|{FLT1}|{FLT2})

CMMT	("/*"((("*"+[^*/])|[^*])*)("*"+"/"))

DLOP	("++"|"--")
EQOP	("+="|"-="|"*="|"/="|"%="|">>="|"<<="|"&="|"^="|"|=")
SINGLE	([\[\](){};,=*:~])

ARGS	({ID}{WHITE0}(","{WHITE0}{ID}{WHITE0})*)

OPPAIR	("<>"|"<="|">="|"++"|"--"|"<<"|">>"|"&&"|"||"|"=="|"!="|"->")

APTY	("("/{WHITE0}"*"{WHITE0}{ID}{WHITE0}")")
APTY1	("("/{WHITE0}"*"{WHITE0}{ID}{WHITE0}"("{WHITE0}")"{WHITE0}")")
CPTY	("("/{WHITE0}"*"{WHITE0}{ID}{WHITE0}")"{WHITE0}"(")
NOTTY	("~"/{ID})

VPTR	("((*(((".*")(".*" -> __vptr__"{INT}{ID}" ["{INT}"]).f))))")
VTBL	("struct __mptr __vtbl__"{INT}{ID}"[] = {"[^}]*"};")


%%

^"#"{WHITE}/[1-9]               { BEGIN PREL; }
^"#"{WHITE0}                    { BEGIN PREX; }
<PREX>"define"{WHITE}           { BEGIN PRED; }
<PREX>"undef"{WHITE}            { BEGIN PRED; }
<PREX>"include"{WHITE}          { BEGIN PREI; }
<PREX>"line"{WHITE}             { BEGIN PREL; }
<PREX>{ID}			{ BEGIN PREP; }
<PREX>. 			;
<PRED>{ID}			{ XRSC_out_decl(cur_filename,yylineno,
						   yytext,"",0,SCLASS_MACRO,NULL);
				  cur_macro = SALLOC(yytext);
				  BEGIN PREP;
				}
<PREP>{ID}			{ XRSC_out_reference(yytext,cur_filename,
							yylineno,cur_macro);
				}
<PREP>{STRING}			{ handle_string(TRUE); }
<PREP>{INT}			{ handle_num(TRUE); }
<PREP>{HEXINT}			{ handle_num(TRUE); }
<PREP>{FLT}			{ handle_num(TRUE); }
<PREP>{WHITE}			;
<PREP>{EQOP}			;
<PREP>{DLOP}			;
<PREP>"..."                     ;
<PREP>{CHAR}			{ handle_string(TRUE); }
<PREP>{APTY}			;
<PREP>{APTY1}			;
<PREP>"(*)"                     ;
<PREP>{CPTY}			;
<PREP>{VPTR}			;
<PREP>{VTBL}			;
<PREP>{OPPAIR}			;
<PREP>"("/{ARGS}")"             ;
<PREP,PREX,PRED,PREL,PREI>"\\\n" ;
<PREP,PREX,PRED,PREL,PREI>"\n"  { if (cur_macro != NULL) {
				      SFREE(cur_macro);
				      cur_macro = NULL;
				    };
				   BEGIN 0;
				}
<PREP>. 			;
<PREL>{INT}{WHITE}\"{FILE}\"    { BEGIN PREP;
				  set_file_line(yytext);
				}
<PREL>. 			{ BEGIN PREP; }
<PREI>\<{FILE}\>.*\n		{ include_file(yytext); BEGIN 0; }
<PREI>\"{FILE}\".*\n            { include_file(yytext); BEGIN 0; }
<PREI>{OPPAIR}			{ BEGIN PREP; }
<PREI>. 			{ BEGIN PREP; }

{EQOP}				{ return LX_EQOP; }
{DLOP}				{ return LX_OTHER; }
{NOTTY} 			{ return LX_NOT; }
{SINGLE}			{ return yytext[0]; }
"..."                           { return LX_ELLIPSES; }

{ID}				{ return scan_id(FALSE); }
{STRING}			{ return handle_string(FALSE); }
{CHAR}				{ return handle_string(FALSE); }
{INT}				{ return handle_num(FALSE); }
{HEXINT}			{ return handle_num(FALSE); }
{FLT}				{ return handle_num(FALSE); }

{APTY}				{ return LX_APAREN; }
{APTY1} 			{ return LX_APAREN; }
{CPTY}				{ return LX_CPAREN; }
"(*)"                           { return '*'; }
"("/{ARGS}")"                   { return LX_BPAREN; }
{VPTR}				{ scan_vptr(); return '('; }
{VTBL}				{ scan_vtbl(); return LX_STRUCT; }
[\n]				;
{WHITE} 			;
{CMMT}				;
{OPPAIR}			{ return LX_OTHER; }
"::"                            { return LX_COLCOL; }
.				{ return LX_OTHER; }





%%

static Integer
scan_id(mac)
{
   Integer i;

   if (STREQL(yytext,"static")) {
      yylval.sclass = SCLASS_STATIC;
      return LX_SCID;
    }
   else if (STREQL(yytext,"extern")) {
      yylval.sclass = SCLASS_EXTERN;
      return LX_SCID;
    }
   else if (STREQL(yytext,"auto")) {
      yylval.sclass = SCLASS_AUTO;
      return LX_SCID;
    }
   else if (STREQL(yytext,"register")) {
      yylval.sclass = SCLASS_REGISTER;
      return LX_SCID;
    }
   else if (STREQL(yytext,"typedef")) {
      yylval.sclass = SCLASS_TYPEDEF;
      return LX_SCID;
    }
   else if (XRSC__cplusplus && STREQL(yytext,"friend")) {
      yylval.sclass = SCLASS_FRIEND;
      return LX_SCID;
    }
   else if (XRSC__cplusplus && STREQL(yytext,"const")) {
      yylval.sclass = SCLASS_CONST;
      return LX_SCID;
    }
   else if (XRSC__cplusplus &&
	     (STREQL(yytext,"overload") || STREQL(yytext,"inline") ||
		    STREQL(yytext,"virtual") || STREQL(yytext,"volatile")))
	 return LX_SPECID;
   else if (XRSC__cplusplus && (STREQL(yytext,"new") || STREQL(yytext,"delete")))
      return LX_OTHER;
   else if (XRSC__cplusplus && STREQL(yytext,"public")) return LX_PUBLIC;
   else if (STREQL(yytext,"struct")) return LX_STRUCT;
   else if (STREQL(yytext,"union")) return LX_UNION;
   else if (STREQL(yytext,"enum")) return LX_ENUM;
   else if (XRSC__cplusplus && STREQL(yytext,"class")) return LX_CLASS;
   else if (STREQL(yytext,"sizeof")) return LX_OTHER;
   else {
      for (i = 0; stmtwords[i] != 0; ++i) {
	 if (STREQL(yytext,stmtwords[i])) break;
       };
      if (stmtwords[i] != 0) return LX_STMTWORD;
    };

   yylval.string = SALLOC(yytext);

   XRSC_out_reference(yytext,cur_filename,yylineno,(mac ? cur_macro: fct_name));

   return LX_ID;
};




static int
handle_num(mac)
   Boolean mac;
{
   if (USE_NUMBERS && STRNEQ(yytext,"0") && STRNEQ(yytext,"1")) {
      XRSC_out_reference(yytext,cur_filename,yylineno,(mac ? cur_macro : fct_name));
    };

   return LX_NUM;
};





static int
handle_string(mac)
   Boolean mac;
{
   if (USE_STRINGS) {
      XRSC_out_reference(yytext,cur_filename,yylineno,(mac ? cur_macro : fct_name));
    };

   return LX_STRING;
};




static void
set_file_line(buf)
   String buf;
{
   Integer line;
   String file,s;

   line = atol(buf);
   file = index(buf,'"');
   if (file != NULL && line != 0) {
      ++file;
      s = index(file,'"');
      if (s != NULL) {
	 *s = 0;
	 XRSC_set_file(file,line-1);
       };
    };
};





static void
include_file(name)
   String name;
{
   String s;
   Boolean gbl;

   gbl = (name[0] == '<');
   ++name;
   if (gbl) s = index(name,'>');
   else s = index(name,'"');

   *s = 0;

   XRSC_push_file(name,gbl);
};





/************************************************************************/
/*									*/
/*	scan_vptr -- handle virtual function calls			*/
/*	scan_vtbl -- handle virtual function table			*/
/*									*/
/************************************************************************/




#define MATCH1		" -> __vptr__"
#define LEN1		12
#define MATCH2		"]).f))))"
#define LEN2		8


static void
scan_vptr()
{
#ifdef DO_VIRTUALS
   String s,t;
   Integer lvl,cnt;
   Integer i,j;
   Boolean fg;
   Character buf[256];

   s = yytext;
   lvl = 0;
   cnt = 0;
   fg = FALSE;

   while (*s != 0) {
      if (*s == '(') {
	 ++lvl;
	 if (lvl == 5) ++cnt;
	 if (cnt > 2) break;
       }
      else if (*s == ')') --lvl;
      if (lvl == 5 && cnt == 2 && strncmp(s,MATCH1,LEN1) == 0 && isdigit(s[LEN1])) {
	 s += LEN1;
	 i = atol(s);
	 while (isdigit(*s)) ++s;
	 t = s;
	 s += i;
	 if (s[0] == ' ' && s[1] == '[' && isdigit(s[2])) {
	    s += 2;
	    j = atol(s);
	    while (isdigit(*s)) ++s;
	    if (strncmp(s,MATCH2,LEN2) == 0) {
	       s += LEN2;
	       fg = TRUE;
	       break;
	     };
	  };
       };
      ++s;
    };

   if (fg) {
      sprintf(buf,"____VIRTUAL__%d__%d",j,i);
      strncat(buf,t,i);
      i = strlen(buf);
      for (t = &yytext[yyleng]; t >= s; --t) t[i] = t[0];
      for (j = 0; j < i; ++j) s[j] = buf[j];
      yyleng += i;
    };
#endif

   yyless(1);
};





static void
scan_vtbl()
{
#ifdef DO_VIRTUALS
   String s,t;
   Integer i;
   Integer cnt;
   Character tbuf[256],nbuf[256],rbuf[256];

   s = index(yytext,' ');
   ++s;
   s = index(s,' ');
   s += 9;		/* skip " __vtbl__" */
   t = tbuf;
   while (*s != '[') *t++ = *s++;
   *t = 0;

   s = index(s,'{');
   ++s;

   cnt = 0;
   while (*s != '}') {
      if ((cnt%3) != 2) {
	 while (!isspace(*s) && *s != ',') ++s;
       }
      else if (*s == '0') ++s;
      else {
	 if (*s == '(') {
	    while (*s != ')') ++s;
	    ++s;
	  };
	 t = nbuf;
	 while (*s != ' ' && *s != ',' && *s != '\n') *t++ = *s++;
	 *t = 0;
	 sprintf(rbuf,"____VIRTUAL__%d__%s",cnt/3,tbuf);
	 XRSC_out_map(rbuf,nbuf);
       };
      while (isspace(*s) || *s == ',') ++s;
      ++cnt;
    };
#endif

   yyless(6);			/* Keep the initial 'struct'    */
};





/* end of xrsclex.l */
