/************************************************************************/
/*									*/
/*		xrscdecl.c						*/
/*									*/
/*	Declaration handler for C cross referencing			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "xrsc_local.h"




/************************************************************************/
/*									*/
/*	Local Types							*/
/*									*/
/************************************************************************/


typedef struct _SCOPE * 	SCOPE;
typedef struct _DECL *		DECL;



typedef enum _DECL_MODE {
   MODE_EXTERN,
   MODE_ARGS,
   MODE_INTERN,
   MODE_STMTS,
   MODE_SUE,
   MODE_NONE,
} DECL_MODE;



typedef struct _SCOPE {
   DECL_MODE mode;
   XRSC_SCOPE_TYPE type;
   Integer ndecl;
   Integer id;
   Integer start_line;
   DECL decl;
   SCOPE previous;
   Boolean scanrhs;
} SCOPE_INFO;



typedef struct _DECL {
   XRSC_SCLASS sclass;
   Character base_type[256];
   Character local_type[256];
   Character name[256];
   Character fct[256];
   Integer line;
   Integer scope;
   Boolean havebase;
   Boolean havelocal;
   Boolean valid;
} DECL_INFO;




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	SCOPE		cur_scope;
static	Integer 	scope_id;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		start_decl();





/************************************************************************/
/*									*/
/*	XRSC_decl_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
XRSC_decl_init()
{
   cur_scope = NULL;
   scope_id = 0;
};





/************************************************************************/
/*									*/
/*	XRSC_decl_begin -- start a scope				*/
/*	XRSC_decl_end -- finish a scope 				*/
/*	XRSC_decl_statements -- start statements in a scope		*/
/*									*/
/************************************************************************/


void
XRSC_decl_begin(typ,file,line)
   XRSC_SCOPE_TYPE typ;
   String file;
   Integer line;
{
   SCOPE s;

   if (typ == SCOPE_NONE && cur_scope == NULL) return;

   s = PALLOC(SCOPE_INFO);
   s->type = typ;
   s->id = ++scope_id;

   switch (typ) {
      case SCOPE_EXTERN :
	 s->mode = MODE_EXTERN;
	 break;
      case SCOPE_ARGS :
	 s->mode = MODE_ARGS;
	 break;
      case SCOPE_INTERN :
	 s->mode = MODE_INTERN;
	 break;
      case SCOPE_SUE :
	 s->mode = MODE_SUE;
	 break;
      case SCOPE_NONE :
	 s->mode = MODE_NONE;
	 s->id = cur_scope->id;
	 break;
    };

   s->ndecl = 0;
   s->start_line = line;
   s->previous = cur_scope;
   s->decl = PALLOC(DECL_INFO);
   s->scanrhs = FALSE;

   cur_scope = s;

   start_decl(s->decl);
};





void
XRSC_decl_end(file,line)
   String file;
   Integer line;
{
   SCOPE s;
   Integer inid;

   if (cur_scope == NULL) return;

   s = cur_scope;
   cur_scope = s->previous;

   if (s->ndecl > 0) {
      if (cur_scope == NULL) inid = 0;
      else {
	 inid = cur_scope->id;
	 ++cur_scope->ndecl;
       };
      XRSC_out_scope(s->id,s->type,file,s->start_line,line,inid);
    };

   free(s->decl);
   free(s);
};





void
XRSC_decl_statements()
{
   if (cur_scope == NULL) return;

   cur_scope->mode = MODE_STMTS;
   XRSC_decl_scan_invalid();
};





void
XRSC_decl_scope_use()
{
   if (cur_scope == NULL) return;

   ++cur_scope->ndecl;
};





/************************************************************************/
/*									*/
/*	XRSC_decl_finish -- finish current declaration			*/
/*	XRSC_decl_inq_name -- return name from current declaration	*/
/*									*/
/************************************************************************/


void
XRSC_decl_finish(new,file,line,fct)
   XRSC_FINISH new;
   String file;
   Integer line;
   String fct;
{
   DECL d;
   Character tbuf[256];

   if (cur_scope == NULL) return;
   d = cur_scope->decl;
   if (d->line < 0) d->line = line;
   if (fct != NULL) strcpy(d->fct,fct);

   if (new == FINISH_BAD) d->valid = FALSE;

   if (d->valid && !d->havelocal && cur_scope->mode == MODE_EXTERN &&
	  new == FINISH_SEMI) {
      strcpy(d->name,d->base_type);
      strcpy(d->base_type,"int");
      d->havelocal = TRUE;
    };

   if (d->valid && d->havelocal) {
      if (d->local_type[0] != 0)
	 sprintf(tbuf,"%s %s",d->base_type,d->local_type);
      else
	 strcpy(tbuf,d->base_type);
      XRSC_out_decl(file,d->line,d->name,tbuf,d->scope,d->sclass,d->fct);
      ++cur_scope->ndecl;
    };

   if (new == FINISH_SEMI || new == FINISH_BAD) start_decl(d);
   else {
      if (!d->havelocal) d->valid = FALSE;
      d->havelocal = FALSE;
      d->local_type[0] = 0;
      d->name[0] = 0;
      d->fct[0] = 0;
      if (new == FINISH_EQ) cur_scope->scanrhs = TRUE;
      else cur_scope->scanrhs = FALSE;
    };
};





String
XRSC_decl_inq_name()
{
   DECL d;
   String s;

   if (cur_scope == NULL) return NULL;
   d = cur_scope->decl;

   if (d->valid && !d->havelocal && cur_scope->mode == MODE_EXTERN) {
      strcpy(d->name,d->base_type);
      strcpy(d->base_type,"int");
      d->havelocal = TRUE;
    };

   if (d->valid && d->havelocal) {
      s = SALLOC(d->name);
    }
   else s = NULL;

   return s;
};





/************************************************************************/
/*									*/
/*	XRSC_decl_scan_id -- scan an identifier 			*/
/*	XRSC_decl_set_sclass -- set storage class			*/
/*	XRSC_decl_scan_type -- add text to local type modifier		*/
/*	XRSC_decl_scan_invalid -- mark declaration as invalid		*/
/*	XRSC_decl_function_body -- mark declaration as function body	*/
/*									*/
/************************************************************************/


void
XRSC_decl_scan_id(txt,line)
   String txt;
   Integer line;
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;
   if (!d->valid) return;

   if (d->havelocal) {
      d->valid = FALSE;
      return;
    };

   if (d->havebase && !d->havelocal && STREQL(txt,"int") &&
	  (STREQL(d->base_type,"long") || STREQL(d->base_type,"short")))
      return;

   if (!d->havebase) {
      strcpy(d->base_type,txt);
      d->havebase = TRUE;
    }
   else {
      strcpy(d->name,txt);
      d->havelocal = TRUE;
      d->line = line;
    };
};





void
XRSC_decl_set_sclass(cls)
   XRSC_SCLASS cls;
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;
   if (!d->valid) return;

   if (cls == SCLASS_CONST && d->havebase) {
      XRSC_decl_scan_type("const ");
      return;
    };

   d->sclass = cls;
};





void
XRSC_decl_set_fclass()
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;
   if (!d->valid) return;

   switch (d->sclass) {
      case SCLASS_STATIC :
	 d->sclass = SCLASS_SFUNCTION;
	 break;
      case SCLASS_EXTERN :
      case SCLASS_EXTDEF :
	 d->sclass = SCLASS_EFUNCTION;
	 break;
      default :
	 d->sclass = SCLASS_EFUNCTION;
	 break;
    };
};





void
XRSC_decl_scan_type(txt)
   String txt;
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;
   if (!d->valid) return;

   strcat(d->local_type,txt);
};





void
XRSC_decl_scan_invalid()
{
   DECL d;

   if (cur_scope == NULL) return;
   d = cur_scope->decl;
   d->valid = FALSE;
};





void
XRSC_decl_function_body()
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;
   if (!d->valid) return;

   switch (d->sclass) {
      case SCLASS_STATIC :
	 d->sclass = SCLASS_SFUNCTION;
	 break;
      default :
	 d->sclass = SCLASS_EFUNCTION;
	 break;
    };
};





/************************************************************************/
/*									*/
/*	XRSC_decl_set_type -- set type for pascal declaration		*/
/*	XRSC_decl_set_id -- set id for pascal declaration		*/
/*	XRSC_decl_add_type -- add to type for pascal declaration	*/
/*									*/
/************************************************************************/


void
XRSC_decl_set_type(name)
   String name;
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;

   start_decl(d);

   d->havebase = TRUE;
   d->line = -1;

   strcpy(d->base_type,name);
};





void
XRSC_decl_add_type(txt)
   String txt;
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;

   strcat(d->base_type,txt);
};





void
XRSC_decl_set_id(id)
   String id;
{
   DECL d;

   if (cur_scope == NULL || cur_scope->scanrhs) return;
   d = cur_scope->decl;

   strcpy(d->name,id);
   d->havelocal = TRUE;
};





/************************************************************************/
/*									*/
/*	XRSC_decl_sue_id -- declare struct/union/enum id		*/
/*	XRSC_decl_enum -- declare enumerant				*/
/*									*/
/************************************************************************/


void
XRSC_decl_sue_id(file,line,id)
   String file;
   Integer line;
   String id;
{
   XRSC_SCLASS typ;

   if (strncmp(id,"struct",6) == 0) {
      id += 7;
      typ = SCLASS_STRUCTID;
    }
   else if (strncmp(id,"union",5) == 0) {
      id += 6;
      typ = SCLASS_UNIONID;
    }
   else if (strncmp(id,"enum",4) == 0) {
      id += 5;
      typ = SCLASS_ENUMID;
    }
   else if (strncmp(id,"class",5) == 0) {
      id += 6;
      typ = SCLASS_CLASSID;
    }
   else return;

   XRSC_out_decl(file,line,id,"",cur_scope->id,typ,NULL);
};





void
XRSC_decl_enum(file,line,id)
   String file;
   Integer line;
   String id;
{
   XRSC_out_decl(file,line,id,"",cur_scope->id,SCLASS_CONST,NULL);
};





/************************************************************************/
/*									*/
/*	XRSC_decl_fct -- handle function declaration			*/
/*									*/
/************************************************************************/


void
XRSC_decl_fct(file,line,name,argct,args)
   String file;
   Integer line;
   String name;
   Integer argct;
   String args[];
{
   Character buf[2048];
   Integer i,sid;

   buf[0] = 0;
   for (i = 0; i < argct; ++i) {
      if (i != 0) strcat(buf,",");
      if (args[i] == NULL) strcat(buf,"?");
      else {
	 strcat(buf,args[i]);
	 SFREE(args[i]);
       };
    };

   sid = cur_scope->id;
   cur_scope->ndecl++;

   if (name != NULL) XRSC_out_proc(file,line,name,sid,argct,buf);
};





/************************************************************************/
/*									*/
/*	start_decl -- setup a declaration				*/
/*									*/
/************************************************************************/


static void
start_decl(d)
   DECL d;
{
   d->scope = cur_scope->id;
   d->valid = TRUE;
   d->havebase = FALSE;
   d->havelocal = FALSE;
   d->base_type[0] = 0;
   d->local_type[0] = 0;
   d->fct[0] = 0;
   d->name[0] = 0;
   cur_scope->scanrhs = FALSE;

   switch (cur_scope->mode) {
      case MODE_EXTERN :
	 d->sclass = SCLASS_EXTDEF;
	 break;
      case MODE_ARGS :
	 d->sclass = SCLASS_PARAM;
	 break;
      case MODE_INTERN :
	 d->sclass = SCLASS_AUTO;
	 break;
      case MODE_STMTS :
      case MODE_NONE :
	 d->valid = FALSE;
	 break;
      case MODE_SUE :
	 d->sclass = SCLASS_FIELD;
	 break;
    };
};





/* end of xrscdecl.c */

