/************************************************************************/
/*									*/
/*		xrefrun.c						*/
/*									*/
/*	Routines to run the xrefdb system from xref			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "xref_local.h"
#include <sys/types.h>
#include <sys/time.h>





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_xrefs;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		system_checker();
static	void		run_start();
static	void		setup_system();
static	void		run_include();
static	void		xref_read_db();




/************************************************************************/
/*									*/
/*	XREF_run_init -- module initialization				*/
/*									*/
/************************************************************************/


void
XREF_run_init()
{
   all_xrefs = NULL;
};




/************************************************************************/
/*									*/
/*	XREF_set_system -- set new system				*/
/*									*/
/************************************************************************/


XREF_DATA
XREF_set_system(sys,new)
   String sys;
   Boolean new;
{
   XREF_DATA xd;
   Sequence l;
   String sysname;

   if (sys == NULL) return NULL;

   sysname = rindex(sys,'/');
   if (sysname == NULL) sysname = sys;
   else ++sysname;

   forin (xd,XREF_DATA,l,all_xrefs) {
      if (STREQL(xd->system,sysname)) break;
      if (STREQL(xd->system,sys)) break;
    };

   if (xd == NULL) {
      forin (xd,XREF_DATA,l,all_xrefs) {
	 if (XREF_check_for_system(xd,sysname)) break;
       };
    };

   if (new && xd == NULL) {
      xd = PALLOC(XREF_DATA_INFO);
      xd->pid = 0;
      xd->system = SALLOC(sys);
      xd->cmd_file = NULL;
      xd->data_file = NULL;
      xd->outfct = NULL;
      xd->outdata = NULL;
      xd->outedit = NULL;
      run_start(xd);
      setup_system(xd,sys,0);
      XREF_load_system(xd);
      PROTECT;
      all_xrefs = CONS(xd,all_xrefs);
      UNPROTECT;
    };

   return xd;
};





/************************************************************************/
/*									*/
/*	XREF_run_query -- process a query				*/
/*									*/
/************************************************************************/


/* VARARGS4 */

void
XREF_run_query(xw,ret,sel,fil,fild)
   XREF_WIN xw;
   String ret;
   String sel;
   Function_Ptr fil;
   Universal fild;
{
   PROTECT;
   while (xw->data->outfct != NULL) {
      UNPROTECT;
      RIPuser_pick(NULL);
    };
   xw->data->outfct = fil;
   xw->data->outdata = fild;
   xw->data->outedit = xw;
   UNPROTECT;

   printf("XREF: (%s) %s ;\n",ret,sel);
   fprintf(xw->data->cmd_file,"(%s) %s ;\n",ret,sel);
   fflush(xw->data->cmd_file);
};





/************************************************************************/
/*									*/
/*	XREF_inq_systems -- return list of systems active		*/
/*									*/
/************************************************************************/


Integer
XREF_inq_systems(mx,sys)
   Integer mx;
   String sys[];
{
   XREF_DATA xd;
   Sequence l;
   Integer ct;

   ct = 0;
   forin (xd,XREF_DATA,l,all_xrefs) {
      if (ct < mx && xd->system != NULL) sys[ct++] = xd->system;
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	XREF_check_for_system -- see if system is included in database	*/
/*									*/
/************************************************************************/


typedef struct _CHECK_SYS {
   String name;
   Boolean found;
   Boolean done;
} CHECK_SYS;



Boolean
XREF_check_for_system(xd,sys)
   XREF_DATA xd;
   String sys;
{
   CHECK_SYS chk;

   if (xd == NULL || xd->cmd_file == NULL) return FALSE;

   PROTECT;
   while (xd->outfct != NULL) {
      UNPROTECT;
      RIPuser_pick(NULL);
    };
   xd->outfct = (Function_Ptr) system_checker;
   xd->outdata = (Universal) &chk;
   xd->outedit = NULL;
   UNPROTECT;

   chk.name = sys;
   chk.found = FALSE;
   chk.done = FALSE;

   fprintf(xd->cmd_file,"# ? ;\n");
   fflush(xd->cmd_file);

   while (!chk.done) {
      CMPXselect(0);
    };

   return chk.found;
};





/*ARGSUSED*/

static void
system_checker(xw,txt,chk)
   XREF_WIN xw;
   String txt;
   CHECK_SYS * chk;
{
   if (txt == NULL) chk->done = TRUE;
   else if (STREQL(txt,chk->name)) chk->found = TRUE;
};






/************************************************************************/
/*									*/
/*	XREF_load_system -- load/reload system				*/
/*									*/
/************************************************************************/


void
XREF_load_system(xd)
   XREF_DATA xd;
{
   if (xd->system == NULL || xd->cmd_file == NULL) return;

   fprintf(xd->cmd_file,"# @ %s ;\n",xd->system);

   fflush(xd->cmd_file);
};





/************************************************************************/
/*									*/
/*	run_start -- start xref database for a system			*/
/*									*/
/************************************************************************/


static void
run_start(xd)
   XREF_DATA xd;
{
   Integer ip[2],op[2];
   Character buf[128];
   Integer pid;
   Boolean errfg;
   fd_set rmask;
   String s;

   s = getenv("XREFDB");
   if (s != NULL) strcpy(buf,s);
   else MSGcommand_name(XRDB_CMD_NAME,buf);

   pipe(ip);
   pipe(op);

   errfg = FALSE;

   pid = vfork();

   if (pid < 0) {
      fprintf(stderr,"Can't start xref database\n");
      exit(1);
    };

   if (pid == 0) {
      dup2(ip[0],0);
      dup2(op[1],1);
      close(ip[1]);
      close(op[0]);
      execl(buf,XRDB_SYSNAME,"-b",0);
      errfg = TRUE;
      _exit(1);
    };

   if (!errfg) {
      xd->pid = pid;
      xd->cmd_file = fdopen(ip[1],"w");
      xd->data_file = fdopen(op[0],"r");
      FD_ZERO(&rmask);
      FD_SET(op[0],&rmask);
      CMPXregister_add(op[0]+1,&rmask,NULL,NULL,xref_read_db);
      close(ip[0]);
      close(op[1]);
    };
};




/************************************************************************/
/*									*/
/*	setup_system -- setup a new system				*/
/*									*/
/************************************************************************/


static void
setup_system(xd,sys,lvl)
   XREF_DATA xd;
   String sys;
   Integer lvl;
{
   Character buf[1024],nam[1024];
   Character nsys[1024];
   FILE * fp;

   sprintf(buf,"%s/.xrefrc",sys);
   if (access(buf,4) >= 0) {
      nsys[0] = 0;
      fp = fopen(buf,"r");
      if (fp != NULL) {
	 if (lvl == 0) run_include(xd,NULL);
	 while (fgets(buf,1024,fp) != NULL) {
	    if (sscanf(buf,"SOURCE %s",nam) == 1) {
	       strcpy(nsys,nam);
	     }
	    else if (sscanf(buf,"INCLUDE %s",nam) == 1) {
	       run_include(xd,nam);
	     }
	  };
       };
      fclose(fp);
      if (nsys[0] != 0 && STRNEQ(sys,nsys) && lvl < 10) {
	 setup_system(xd,nsys,lvl+1);
	 return;
       };
    };
};





/************************************************************************/
/*									*/
/*	run_include -- add an include file				*/
/*									*/
/************************************************************************/


static void
run_include(xd,file)
   XREF_DATA xd;
   String file;
{
   if (xd == NULL || xd->system == NULL || xd->cmd_file == NULL) return;

   if (file != NULL) fprintf(xd->cmd_file,"# + %s ;\n",file);
   else fprintf(xd->cmd_file,"# + ;\n");

   fflush(xd->cmd_file);
};




/************************************************************************/
/*									*/
/*	xref_read_db -- handle read ready from database 		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
xref_read_db(rmsk,wmsk,emsk)
   fd_set *rmsk;
   fd_set *wmsk;
   fd_set *emsk;
{
   Sequence l;
   XREF_DATA xd;
   Character buf[1024];
   Integer ln;

   forin (xd,XREF_DATA,l,all_xrefs) {
      if (xd->data_file == NULL || !FD_ISSET(fileno(xd->data_file),rmsk))
	 continue;

      if (xd->outedit != NULL) XREF_output_begin(xd->outedit);

      buf[0] = 0;

      for ( ; ; ) {
	 if (fgets(buf,1024,xd->data_file) == NULL) break;
	 ln = strlen(buf);
	 if (buf[ln-1] == '\n') buf[ln-1] = 0;
	 if (STREQL(buf,END_QUERY_MARKER)) break;
	 if (xd->outfct != NULL) (*xd->outfct)(xd->outedit,buf,xd->outdata);
       };

      if (xd->outfct != NULL) {
	 (*xd->outfct)(xd->outedit,NULL,xd->outdata);
       };

      if (xd->outedit != NULL) XREF_output_done(xd->outedit);

      xd->outfct = NULL;
      xd->outdata = NULL;
      xd->outedit = NULL;
   };
};





/* end of xrefrun.c */

