/************************************************************************/
/*									*/
/*		xrdbscan.c						*/
/*									*/
/*	File to read xrefscan output and manage internal forms		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "xrdb_local.h"
#include <varargs.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define LINE_LENGTH	1024




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	XRDB_SCOPE_ID * scope_map;
static	Integer 	scope_map_max;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		scan_file();
static	void		scan_scope();
static	void		scan_ref();
static	void		scan_decl();
static	void		scan_call();
static	void		scan_proc();
static	void		scan_assign();
static	void		scan_hier();
static	void		scan_member();
static	void		scan_map();
static	void		clear_scope_map();
static	void		set_scope_map();
static	XRDB_SCOPE_ID	find_scope();
static	XRDB_SCOPE_CLASS find_scope_class();
static	XRDB_DECL_CLASS find_decl_class();
static	XRDB_PROT_CLASS find_prot_class();
static	void		fix_string();





/************************************************************************/
/*									*/
/*	XRDB_scan_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
XRDB_scan_init()
{
   scope_map_max = 0;
   scope_map = NULL;
};




/************************************************************************/
/*									*/
/*	XRDB_scan_file -- scan input from passed file			*/
/*									*/
/************************************************************************/


void
XRDB_scan_file(inf)
   FILE * inf;
{
   Character buf[LINE_LENGTH];

   clear_scope_map();
   XRDB_db_file_scan();

   while (fgets(buf,LINE_LENGTH,inf) != NULL) {
      switch (buf[0]) {
	 case 'F' :                     /* FIL file_name directory      */
	    scan_file(buf);
	    break;

	 case 'S' :                     /* SCP file from to id type     */
	    scan_scope(buf);
	    break;

	 case 'R' :                     /* REF file line name           */
	    scan_ref(buf);
	    break;

	 case 'D' :                     /* DCL file line name scope class type */
	    scan_decl(buf);
	    break;

	 case 'C' :                     /* CAL file line proc from      */
	    scan_call(buf);
	    break;

	 case 'P' :                     /* PRC file line name scope #arg args...        */
	    scan_proc(buf);
	    break;

	 case 'A' :                     /* ASG file line name           */
	    scan_assign(buf);
	    break;

	 case 'H' :                     /* HIE file line class parent friend    */
	    scan_hier(buf);
	    break;

	 case 'M' :                     /* MEM file line cls mem prot isdata inline... */
	    scan_member(buf);
	    break;

	 case 'm' :                     /* MAP from to                  */
	    scan_map(buf);
	    break;
       };
    };
};





/************************************************************************/
/*									*/
/*	XRDB_scan_string -- scan string for patterned args		*/
/*									*/
/************************************************************************/


Integer
XRDB_scan_string(va_alist)
   va_dcl
{
   va_list ap;
   register String str,pat,rs;
   register Integer v,sg;
   Integer * dp;
   Integer ct;

   va_start(ap);
   str = va_arg(ap,String);
   pat = va_arg(ap,String);
   ct = 0;

   while (*pat != 0) {
      if (*pat != '%') {
	 if (*str != *pat) return ct;
	 ++pat;
	 ++str;
       }
      else {
	 ++pat;
	 if (*pat == 'd') {
	    while (isspace(*str)) ++str;
	    sg = 1;
	    if (*str == '-') {
	       sg = -1;
	       ++str;
	     };
	    v = 0;
	    while (isdigit(*str)) v = v*10 + *str++ - '0';
	    if (sg < 0) v = -v;
	    dp = va_arg(ap,Integer *);
	    *dp = v;
	    ++ct;
	    ++pat;
	  }
	 else if (*pat == 's') {
	    while (isspace(*str)) ++str;
	    rs = va_arg(ap,String);
	    ++pat;
	    while (*str != 0 && *str != '\n') {
	       if (*str == ' ' && *pat == ' ') break;
	       *rs++ = *str++;
	     };
	    *rs = 0;
	    ++ct;
	  }
	 else return -1;
       };
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	scan_file -- scan the FIL line					*/
/*									*/
/************************************************************************/


static void
scan_file(buf)
   String buf;
{
   Character fil[1024],dir[1024];
   Integer lvl;

   XRDB_scan_string(buf,FILE_FORMAT,fil,dir,&lvl);

   XRDB_db_new_file(fil,dir,TRUE,lvl);
};




/************************************************************************/
/*									*/
/*	scan_scope -- scan SCP line					*/
/*									*/
/************************************************************************/


static void
scan_scope(buf)
   String buf;
{
   Character fil[1024],typ[256];
   Integer from,to,id,inid;
   XRDB_SCOPE_ID nid;

   XRDB_scan_string(buf,SCOPE_FORMAT,fil,&from,&to,&id,&inid,typ);

   nid = find_scope(id);

   XRDB_db_set_scope(nid,XRDB_db_find_file(fil),from,to,find_scope_class(typ),
			find_scope(inid));
};






/************************************************************************/
/*									*/
/*	scan_ref -- scan REF line					*/
/*									*/
/************************************************************************/


static void
scan_ref(buf)
   String buf;
{
   Character fil[1024],nam[256],fct[256];
   Integer line;

   XRDB_scan_string(buf,REF_FORMAT,fil,&line,fct,nam);
   fix_string(fct);
   fix_string(nam);

   XRDB_db_ref(XRDB_db_find_file(fil),line,XRDB_db_find_name(nam),FALSE,
		  XRDB_db_find_name(fct));
};





/************************************************************************/
/*									*/
/*	scan_decl -- scan a declaration entry				*/
/*									*/
/************************************************************************/


static void
scan_decl(buf)
   String buf;
{
   Character fil[1024],nam[256],typ[256],scls[256],fct[256];
   Integer line,scope;
   Integer i;

   i = XRDB_scan_string(buf,DECL_FORMAT,fil,&line,nam,&scope,scls,fct,typ);
   fix_string(nam);
   fix_string(fct);
   fix_string(typ);

   if (i == 6) strcpy(typ,"*UNKNOWN*");

   XRDB_db_decl(XRDB_db_find_file(fil),line,XRDB_db_find_name(nam),
		   find_scope(scope),find_decl_class(scls),
		   XRDB_db_find_type(typ),XRDB_db_find_name(fct));
};





/************************************************************************/
/*									*/
/*	scan_call -- scan a call entry					*/
/*									*/
/************************************************************************/


static void
scan_call(buf)
   String buf;
{
   Character fil[1024],rtn[256],from[256];
   Integer line;

   XRDB_scan_string(buf,CALL_FORMAT,fil,&line,rtn,from);
   fix_string(rtn);
   fix_string(from);

   XRDB_db_call(XRDB_db_find_file(fil),line,XRDB_db_find_name(from),
		   XRDB_db_find_name(rtn));
};





/************************************************************************/
/*									*/
/*	scan_proc -- scan a procedure definition entry			*/
/*									*/
/************************************************************************/


static void
scan_proc(buf)
   String buf;
{
   Character fil[1024],nam[256],args[1024];
   Integer line,scope,narg;

   XRDB_scan_string(buf,PROC_FORMAT,fil,&line,nam,&scope,&narg,args);
   fix_string(nam);
   fix_string(args);

   if (narg == 0) args[0] = 0;

   XRDB_db_fct(XRDB_db_find_file(fil),line,XRDB_db_find_name(nam),
		  find_scope(scope),narg,args);
};





/************************************************************************/
/*									*/
/*	scan_assign -- scan an assignment entry 			*/
/*									*/
/************************************************************************/


static void
scan_assign(buf)
   String buf;
{
   Character fil[1024],nam[256];
   Integer line;

   XRDB_scan_string(buf,ASSIGN_FORMAT,fil,&line,nam);
   fix_string(nam);

   XRDB_db_ref_assign(XRDB_db_find_file(fil),line,XRDB_db_find_name(nam),0);
};





/************************************************************************/
/*									*/
/*	scan_hier -- scan a class hierarchy entry			*/
/*									*/
/************************************************************************/


static void
scan_hier(buf)
   String buf;
{
   Character fil[1024],cls[256],par[256];
   Integer line,pub,virt,frnd;

   XRDB_scan_string(buf,HIER_FORMAT,fil,&line,cls,par,&pub,&virt,&frnd);
   fix_string(cls);
   fix_string(par);

   XRDB_db_hier(XRDB_db_find_file(fil),line,XRDB_db_find_name(cls),
		   XRDB_db_find_name(par),pub,virt,frnd);
};





/************************************************************************/
/*									*/
/*	scan_member -- scan a class-member entry			*/
/*									*/
/************************************************************************/


static void
scan_member(buf)
   String buf;
{
   Character fil[1024],cls[256],mem[256],prot[256];
   Integer line,isdata,isin,frnd,virt,stat,pure,con;

   XRDB_scan_string(buf,MEMB_FORMAT,fil,&line,cls,mem,prot,&isdata,&isin,&frnd,
	     &virt,&stat,&pure,&con);
   fix_string(cls);
   fix_string(mem);

   XRDB_db_memb(XRDB_db_find_file(fil),line,XRDB_db_find_name(cls),
		   XRDB_db_find_name(mem),find_prot_class(prot),
		   isdata,isin,frnd,virt,stat,pure,con);
};





/************************************************************************/
/*									*/
/*	scan_map -- scan and process a map entry			*/
/*									*/
/************************************************************************/


static void
scan_map(buf)
   String buf;
{
   Character frm[1024],to[1024];

   XRDB_scan_string(buf,MAP_FORMAT,frm,to);

   XRDB_db_map_name(XRDB_db_find_name(frm),XRDB_db_find_name(to));
};





/************************************************************************/
/*									*/
/*	clear_scope_map -- clear map of local->global scopes		*/
/*	set_scope_map -- add map entry					*/
/*	find_scope -- look up map entry 				*/
/*									*/
/************************************************************************/


static void
clear_scope_map()
{
   if (scope_map == NULL) {
      scope_map_max = 1020;
      scope_map = (XRDB_SCOPE_ID *) calloc(sizeof(XRDB_SCOPE_ID),scope_map_max);
    };
};





static void
set_scope_map(lcl,gbl)
   Integer lcl;
   XRDB_SCOPE_ID gbl;
{
   Integer i;

   if (lcl >= scope_map_max) {
      i = scope_map_max;
      while (lcl >= scope_map_max) scope_map_max *= 2;
      scope_map = (XRDB_SCOPE_ID *) realloc(scope_map,
					       scope_map_max*sizeof(XRDB_SCOPE_ID));
      for ( ; i < scope_map_max; ++i) scope_map[i] = 0;
    };

   scope_map[lcl] = XRDB_db_new_scope();
};





static XRDB_SCOPE_ID
find_scope(lcl)
   Integer lcl;
{
   if (lcl == 0) return 0;

   if (scope_map[lcl] == 0) set_scope_map(lcl);

   return scope_map[lcl];
};





/************************************************************************/
/*									*/
/*	find_scope_class -- find scope class from name			*/
/*	find_decl_class -- find declaration class from name		*/
/*	find_prot_class -- find protection class from name		*/
/*									*/
/************************************************************************/


static XRDB_SCOPE_CLASS
find_scope_class(nam)
   String nam;
{
   XRDB_SCOPE_CLASS r;

   if (STREQL(nam,"EXTERN")) r = XRDB_SCOPE_EXTERN;
   else if (STREQL(nam,"INTERN")) r = XRDB_SCOPE_INTERN;
   else if (STREQL(nam,"ARGS")) r = XRDB_SCOPE_ARGS;
   else if (STREQL(nam,"SUE")) r = XRDB_SCOPE_SUE;
   else r = XRDB_SCOPE_NONE;

   return r;
};





static XRDB_DECL_CLASS
find_decl_class(nam)
   String nam;
{
   XRDB_DECL_CLASS r;

   if (STREQL(nam,"STATIC")) r = XRDB_DECL_STATIC;
   else if (STREQL(nam,"EXTERN")) r = XRDB_DECL_EXTERN;
   else if (STREQL(nam,"AUTO")) r = XRDB_DECL_AUTO;
   else if (STREQL(nam,"REGISTER")) r = XRDB_DECL_REGISTER;
   else if (STREQL(nam,"TYPEDEF")) r = XRDB_DECL_TYPEDEF;
   else if (STREQL(nam,"EXTDEF")) r = XRDB_DECL_EXTDEF;
   else if (STREQL(nam,"PARAM")) r = XRDB_DECL_PARAM;
   else if (STREQL(nam,"FIELD")) r = XRDB_DECL_FIELD;
   else if (STREQL(nam,"EFUNCTION")) r = XRDB_DECL_EFUNCTION;
   else if (STREQL(nam,"SFUNCTION")) r = XRDB_DECL_SFUNCTION;
   else if (STREQL(nam,"STRUCTID")) r = XRDB_DECL_STRUCTID;
   else if (STREQL(nam,"CLASSID")) r = XRDB_DECL_CLASSID;
   else if (STREQL(nam,"UNIONID")) r = XRDB_DECL_UNIONID;
   else if (STREQL(nam,"ENUMID")) r = XRDB_DECL_ENUMID;
   else if (STREQL(nam,"CONST")) r = XRDB_DECL_CONST;
   else if (STREQL(nam,"MACRO")) r = XRDB_DECL_MACRO;
   else r = XRDB_DECL_NONE;

   return r;
};





static XRDB_PROT_CLASS
find_prot_class(nam)
   String nam;
{
   XRDB_PROT_CLASS p;

   if (STREQL(nam,"PUBLIC")) p = XRDB_PROT_PUBLIC;
   else if (STREQL(nam,"PRIVATE")) p = XRDB_PROT_PRIVATE;
   else if (STREQL(nam,"PROTECTED")) p = XRDB_PROT_PROTECTED;
   else p = XRDB_PROT_NONE;

   return p;
};






/************************************************************************/
/*									*/
/*	fix_string -- fix up spaces in name string			*/
/*									*/
/************************************************************************/


static void
fix_string(s)
   String s;
{
   for ( ; *s != 0; ++s) {
      if (*s == SPACE_CHARACTER) *s = ' ';
    };
};





/* end of xrdbscan.c */
