/************************************************************************/
/*									*/
/*		xrdbfile.c						*/
/*									*/
/*	File management for xreference database 			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "xrdb_local.h"

#include <fsym.h>




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Integer 	num_include;
static	String		include[32];
static	Sequence	all_files;
static	String		comp_flags;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	scan_directory();
static	Boolean 	scan_src_file();
static	Boolean 	scan_binary();
static	void		scan_binary_dir();
static	void		scan_binary_so();
static	String		get_command_name();
static	void		scan_rc_file();




/************************************************************************/
/*									*/
/*	XRDB_file_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
XRDB_file_init()
{
   num_include = 0;
   all_files = NULL;
   XRDB__cplus2 = FALSE;
   comp_flags = NULL;

   if (getenv("CPLUS12") != NULL) XRDB__cplus2 = FALSE;
   if (getenv("CPLUS20") != NULL) XRDB__cplus2 = TRUE;
};





/************************************************************************/
/*									*/
/*	XRDB_file_include -- set file to be use for includes		*/
/*	XRDB_comp_flags -- set compilation flags			*/
/*									*/
/************************************************************************/


void
XRDB_file_include(name)
   String name;
{
   Integer i;

   if (name == NULL) {
      for (i = 0; i < num_include; ++i) SFREE(include[i]);
      num_include == 0;
    }
   else include[num_include++] = SALLOC(name);
};





void
XRDB_comp_flags(str)
   String str;
{
   if (comp_flags != NULL) SFREE(comp_flags);
   comp_flags = NULL;

   if (str != NULL) comp_flags = SALLOC(str);
};






/************************************************************************/
/*									*/
/*	XRDB_file_setup -- setup and scan a file/directory		*/
/*									*/
/************************************************************************/


Boolean
XRDB_file_setup(name,noload)
   String name;
   Boolean noload;
{
   Character buf[1024],buf1[1024];
   String fil,dir;
   struct stat sinfo;
   time_t time;
   Boolean ckfg;
   Sequence l;

   if (stat(name,&sinfo) < 0) return FALSE;

   if (all_files != NULL) {
      forin (fil,String,l,all_files) {
	 SFREE(fil);
       };
      LFREE(all_files);
      all_files = NULL;
    };

   ckfg = FALSE;

   if ((sinfo.st_mode & S_IFREG) != 0) {
      strcpy(buf,name);
      fil = rindex(buf,'/');
      if (fil == NULL) {
	 dir = ".";
	 fil = buf;
       }
      else {
	 dir = buf;
	 *fil++ = 0;
       };
      sprintf(buf1,"%s/.%s.xrefrc",dir,fil);
      scan_rc_file(buf1);
      sprintf(buf1,"%s/.%s.xref",dir,fil);
      XRDB_db_system(fil,dir);
      if (!noload) {
	 time = XRDB_db_load(buf1);
	 ckfg = TRUE;
       }
      else {
	 time = 0;
       };
      if (scan_binary(name,time)) {
	 if (ckfg) XRDB_db_check_files(all_files);
	 XRDB_db_store(buf1);
       }
      else scan_src_file(name);
    }
   else if ((sinfo.st_mode & S_IFDIR) != 0) {
      strcpy(buf,name);
      fil = rindex(buf,'/');
      if (fil == NULL) fil = name;
      else ++fil;
      XRDB_db_system(fil,name);
      sprintf(buf,"%s/.xrefrc",name);
      scan_rc_file(buf);
      sprintf(buf,"%s/.xrefdbdata",name);
      if (!noload) {
	 time = XRDB_db_load(buf);
       }
      else {
	 time = 0;
       };
      if (scan_directory(name,time)) XRDB_db_store(buf);
    }
   else return FALSE;

   XRDB_db_fixup_member_defs();

   return TRUE;
};




/************************************************************************/
/*									*/
/*	XRDB_file_check -- check if file already scanned		*/
/*									*/
/************************************************************************/


Boolean
XRDB_file_check(name)
   String name;
{
   Sequence l;
   String s;

   forin (s,String,l,all_files) {
      if (STREQL(s,name)) return TRUE;
    };

   all_files = CONS(SALLOC(name),all_files);

   return FALSE;
};





/************************************************************************/
/*									*/
/*	scan_directory -- handle a directory				*/
/*									*/
/************************************************************************/


static Boolean
scan_directory(name,time)
   String name;
   time_t time;
{
   DIR * dp;
   struct direct * ent;
   Character buf[1024];
   struct stat sinfo;
   Boolean fg;

   dp = opendir(name);
   fg = FALSE;

   for (ent = readdir(dp); ent != NULL; ent = readdir(dp)) {
      if (ent->d_name[0] == '.') continue;
      strcpy(buf,name);
      strcat(buf,"/");
      strcat(buf,ent->d_name);
      if (stat(buf,&sinfo) >= 0) {
	 if ((sinfo.st_mode & S_IFDIR) != 0) {
	    fg |= scan_directory(buf,time);
	  }
	 else if ((sinfo.st_mode & S_IFREG) != 0 &&
		     (time == 0 || sinfo.st_mtime > time)) {
	    fg |= scan_src_file(buf);
	  };
       };
    };

   closedir(dp);

   return fg;
};




/************************************************************************/
/*									*/
/*	scan_src_file -- handle a file					*/
/*									*/
/************************************************************************/


static Boolean
scan_src_file(name)
   String name;
{
   String s,t,sf;
   Character buf[256],cmd[1024],cbf[256],fbf[256];
   Integer i;
   FILE * fp;
   Boolean rslt;
   Sequence l;

   s = rindex(name,'.');
   if (s == NULL) {
      XRDB_db_add_system(name);
      return FALSE;
    };

   if (all_files != NULL) {
      strcpy(fbf,"/tmp/XRDBXXXXXX");
      mktemp(fbf);
      fp = fopen(fbf,"w");
      forin (t,String,l,all_files) {
	 fprintf(fp,"%s\n",t);
       };
      fclose(fp);
      sf = fbf;
    }
   else sf = NULL;

   if (XRDB_file_check(name)) return TRUE;

   fprintf(stderr,"XREFDB: Scanning %s\n",name);

   if (s != name && s[-1] == '.') --s;

   if (STREQL(s,".C") || STREQL(s,"..c") || (XRDB__cplus2 && STREQL(s,".c"))) {
      get_command_name(CPXREF,buf);
      strcpy(cmd,buf);
      strcat(cmd," -a -C");
      for (i = 0; i < num_include; ++i) {
	 strcat(cmd," -I");
	 strcat(cmd,include[i]);
       };
      if (comp_flags != NULL) {
	 strcat(cmd," '-F");
	 strcat(cmd,comp_flags);
	 strcat(cmd,"'");
       };
      if (sf != NULL) {
	 strcat(cmd," -S");
	 strcat(cmd,sf);
       };
      strcat(cmd," ");
      strcat(cmd,name);
      fp = popen(cmd,"r");
      if (fp != NULL) {
	 XRDB_scan_file(fp);
	 pclose(fp);
       };
      rslt = TRUE;
    }
   else if (STREQL(s,".c")) {
      get_command_name(CXREF,buf);
      strcpy(cmd,buf);
      if (STREQL(s,".C") || STREQL(s,"..c")) strcat(cmd," -C");
      for (i = 0; i < num_include; ++i) {
	 strcat(cmd," -I");
	 strcat(cmd,include[i]);
       };
      if (comp_flags != NULL) {
	 strcat(cmd," '-F");
	 strcat(cmd,comp_flags);
	 strcat(cmd,"'");
       };
      if (sf != NULL) {
	 strcat(cmd," -S");
	 strcat(cmd,sf);
       };
      strcat(cmd," ");
      strcat(cmd,name);
      fp = popen(cmd,"r");
      if (fp != NULL) {
	 XRDB_scan_file(fp);
	 pclose(fp);
       };
      rslt = TRUE;
    }
   else if (STREQL(s,".p")) {
      get_command_name(PXREF,buf);
      strcpy(cmd,buf);
      for (i = 0; i < num_include; ++i) {
	 strcat(cmd," -I");
	 strcat(cmd,include[i]);
       };
      if (comp_flags != NULL) {
	 strcat(cmd," '-F");
	 strcat(cmd,comp_flags);
	 strcat(cmd,"'");
       };
      if (sf != NULL) {
	 strcat(cmd," -S");
	 strcat(cmd,sf);
       };
      strcat(cmd," ");
      strcat(cmd,name);
      fp = popen(cmd,"r");
      if (fp != NULL) {
	 XRDB_scan_file(fp);
	 pclose(fp);
       };
      rslt = TRUE;
    }
   else rslt = FALSE;

   if (sf != NULL) unlink(sf);

   return rslt;
};





/************************************************************************/
/*									*/
/*	scan_binary -- scan a binary system				*/
/*	scan_binary_dir -- handle directory entry			*/
/*	scan_binary_so -- handle source symbol				*/
/*									*/
/************************************************************************/


typedef struct _BIN_DATA {
   String dir;
   time_t time;
} BIN_DATA;



static Boolean
scan_binary(sys,time)
   String sys;
   time_t time;
{
   FSYM_ID fsym;
   FSYM_CALL fsymcall[3];
   BIN_DATA bindata;

   fsym = FSYMopen(sys,FALSE);
   if (fsym == NULL) return FALSE;

   fsymcall[0].type = FSYM_TYPE_DIR;
   fsymcall[0].addr = scan_binary_dir;
   fsymcall[1].type = FSYM_TYPE_SO;
   fsymcall[1].addr = scan_binary_so;
   fsymcall[2].type = FSYM_TYPE_END;

   bindata.dir = NULL;
   bindata.time = time;

   FSYMsymbols(fsym,fsymcall,&bindata);

   FSYMclose(fsym);

   XRDB_db_add_system(sys);

   return TRUE;
};





/*ARGSUSED*/

static void
scan_binary_dir(name,desc,val,ft,bd)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   BIN_DATA * bd;
{
   String s;

   for (s = name; s != NULL; s = index(&s[1],'/')) {
      if (strncmp(s,"/tmp/CC.",8) == 0) {
	 XRDB__cplus2 = TRUE;
	 return;
       };
    };

   bd->dir = SALLOC(name);
};





/*ARGSUSED*/

static void
scan_binary_so(name,desc,val,ft,bd)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   BIN_DATA * bd;
{
   String s;
   Character buf[1024],cbf[1024];
   struct stat sinfo;
   Integer i;

   if (bd->dir == NULL || name[0] == '/') {
      strcpy(buf,name);
    }
   else {
      sprintf(buf,"%s%s",bd->dir,name);
    };

   s = buf;
   i = strlen(buf);
   if (i >= 3 && s[i-2] == '.' && s[i-1] == 'c' && access(s,4) < 0) {
      strcpy(cbf,s);
      cbf[i-1] = 'C';
      if (access(cbf,4) >= 0) s = cbf;
      else if (cbf[i-3] == '.') {
	 cbf[i-2] = 'C';
	 cbf[i-1] = 0;
	 if (access(cbf,4) >= 0) s = cbf;
       };
    };

   if (XRDB_db_check_scan(s)) XRDB_file_check(s);
   else scan_src_file(s);
};





/************************************************************************/
/*									*/
/*	get_command_name -- get command path				*/
/*									*/
/************************************************************************/


static String
get_command_name(name,buf)
   String name;
   String buf;
{
   Boolean experiment;

   experiment = FALSE;
   if (getenv("FIELD_DEBUG") != NULL) experiment = TRUE;

   if (name[0] == '/') {
      strcpy(buf,name);
    }
   else {
      sprintf(buf,"%s/bin/field/%s",BWElocal_project(),name);
      if (experiment || access(buf,1) < 0) {
	 sprintf(buf,"%s/bin/field/%s/%s",BWElocal_project(),BWEarch(),name);
	 if (experiment || access(buf,1) < 0) {
	    sprintf(buf,"%s/field/cmd/%s/%s",BWElocal_project(),BWEarch(),name);
	  };
       };
    };

   return buf;
};






/************************************************************************/
/*									*/
/*	scan_rc_file -- scan .xrefrc file for system			*/
/*									*/
/************************************************************************/


static void
scan_rc_file(name)
   String name;
{
   FILE * inf;
   Character buf[1024],nam[1024];
   String s,t;

   XRDB_file_include(NULL);

   if (name == NULL) inf = NULL;
   else inf = fopen(name,"r");

   if (inf != NULL) {
      while (fgets(buf,1024,inf) != NULL) {
	 if (sscanf(buf,"INCLUDE %s",nam) == 1) {
	    XRDB_file_include(nam);
	  }
	 else if (strncmp(buf,"FLAGS ",6) == 0) {
	    s = &buf[6];
	    while (isspace(*s)) ++s;
	    XRDB_comp_flags(s);
	  }
	 else if (strncmp(buf,"CPLUSPLUS",5) == 0) {
	    XRDB__cplus2 = TRUE;
	  }
	 else if (strncmp(buf,"COMPRESS",5) == 0) {
	    XRDB__compress = TRUE;
	  };
       };
      fclose(inf);
    }
   else {
      s = getenv("INCLUDE_PATH");
      if (s != NULL) {
	 strncpy(buf,s,1024);
	 for (s = buf; s != NULL; s = t) {
	    t = index(s,':');
	    if (t != NULL) *t++ = 0;
	    XRDB_file_include(s);
	  };
       };
      XRDB_comp_flags(getenv("XRDB_FLAGS"));
    };
};





/* end of xrdbfile.c */
