/************************************************************************/
/*									*/
/*		viewwin.c						*/
/*									*/
/*	Window manipulation routines for viewing windows		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "view_local.h"
#include <leaf.h>
#include <stem.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	setup_window();
static	int		view_control();
static	void		check_windows();
static	int		view_edit_control();
static	void		view_remove();
static	VIEW_WTYPE	request_window_type();
static	Integer 	system_btn();
static	Integer 	quit_btn();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define EDIT_WIN	2



static	LEAF_DATA	view_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { EDIT_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





static STEM_PDM_DATA	view_menus[] = {
   { STEM_PSTATE_MENU,	"View",         NULL },
   { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX,   "Set System",   system_btn },
   { STEM_PSTATE_BTN,	"Options",      VIEW_ask_btn },
   { STEM_PSTATE_BTN,	"Remove",       VIEW_remove_btn },
   { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX,   "Quit",         quit_btn },

   { STEM_PSTATE_END }
};

#define OPTION_INDEX	2
#define REMOVE_INDEX	3





/************************************************************************/
/*									*/
/*	VIEW_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
VIEW_win_init()
{
   all_windows = NULL;
};





/************************************************************************/
/*									*/
/*	VIEW_create -- create a new viewation window			*/
/*									*/
/************************************************************************/


VIEW_WIN
VIEW_create(w,type)
   ASH_WINDOW w;
   String type;
{
   register VIEW_WIN vw;
   register VIEW_WTYPE vwt;

   if (type != NULL) {
      vwt = VIEW_find_window_type(type);
      if (vwt == NULL) type = NULL;
    };

   if (type == NULL) vwt = request_window_type(w);

   if (vwt == NULL) return NULL;

   ASHset_window_name(w,vwt->name);
   ASHset_window_id(w,vwt->name);
   ASHset_window_defaults(w);

   vw = PALLOC(VIEW_WIN_INFO);
   vw->window = w;
   vw->type = vwt;
   vw->edit_info = NULL;
   vw->willow_fg = FALSE;
   vw->num_disp = 0;
   vw->displays = NULL;
   vw->max_disp = 0;

   PROTECT;
   all_windows = CONS(vw,all_windows);
   UNPROTECT;

   if (!setup_window(vw)) return NULL;

   if (vw->menu_win != NULL) {
      if (vwt->options_name != NULL) {
	 view_menus[OPTION_INDEX].name = vwt->options_name;
       };
      if (vwt->options_rtn == NULL)
	 view_menus[OPTION_INDEX].state = STEM_PSTATE_IGNORE;

      if (vwt->remove_name != NULL) {
	 view_menus[REMOVE_INDEX].name = vwt->remove_name;
       };
      if (vwt->remove_rtn == NULL)
	 view_menus[REMOVE_INDEX].state = STEM_PSTATE_IGNORE;

      STEMpdm_define(vw->menu_win,vw,view_menus);

      view_menus[OPTION_INDEX].name = "Options";
      view_menus[OPTION_INDEX].state = STEM_PSTATE_BTN;
      view_menus[REMOVE_INDEX].name = "Remove";
      view_menus[REMOVE_INDEX].state = STEM_PSTATE_BTN;

      if (vwt->simple && vwt->options_rtn == NULL && vwt->remove_rtn == NULL) {
	 STEMpdm_menu_remove(vw->menu_win,"View");
       };
    };

   VIEW_edit_setup(vw);
   VIEW_text_setup(vw);
   VIEW_mon_setup(vw);

   if (vw->type->askfirst) {
      VIEW_ask_btn(vw,"<DUMMY>","Ask");
    };

   return vw;
};





/************************************************************************/
/*									*/
/*	VIEW_find_window -- find view window given ash window		*/
/*									*/
/************************************************************************/


VIEW_WIN
VIEW_find_window(w)
   ASH_WINDOW w;
{
   register Sequence l;
   register VIEW_WIN vw;

   forin (vw,VIEW_WIN,l,all_windows) {
      if (vw->window == w) break;
      else if (vw->edit_win == w) break;
      else if (vw->menu_win == w) break;
    };

   return vw;
};





/************************************************************************/
/*									*/
/*	setup_window -- define the portions of an view window		*/
/*									*/
/************************************************************************/


static Boolean
setup_window(vw)
   VIEW_WIN vw;
{
   LEAFsetup_window(vw->window,view_leaf,vw);

   LEAFset_control(vw->window,view_control);
   LEAFredraw(vw->window);

   vw->edit_win = LEAFinq_window(vw->window,EDIT_WIN,0);
   if (vw->edit_win == NULL) {
      fprintf(stderr,"VIEW: Window too small to run viewer\n");
      ASHremove(vw->window);
      return FALSE;
   };

   ASHset_control(vw->edit_win,view_edit_control);
   LEAFset_refresh(vw->window,check_windows);

   vw->menu_win = LEAFinq_window(vw->window,MENU_WIN,0);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	view_control -- handle ASH and other control messages		*/
/*									*/
/************************************************************************/


static int
view_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register VIEW_WIN vw;

   vw = VIEW_find_window(w);

   if (vw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      view_remove(vw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	check_windows -- check for user doing a stupid resize		*/
/*									*/
/************************************************************************/


static void
check_windows(w)
   ASH_WINDOW w;
{
   VIEW_WIN vw;
   ASH_WINDOW ew;

   vw = VIEW_find_window(w);

   if (vw == NULL) return;

   ew = LEAFinq_window(w,EDIT_WIN,0);

   if (ew != vw->edit_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };
};





/************************************************************************/
/*									*/
/*	view_edit_control -- handle ASH messages for editor		*/
/*									*/
/************************************************************************/


static int
view_edit_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register VIEW_WIN vw;

   vw = VIEW_find_window(w);

   if (vw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"EDIT$REMOVE")) {
      ASHremove(vw->window);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	view_remove -- remove given view				*/
/*									*/
/************************************************************************/


static void
view_remove(vw)
   VIEW_WIN vw;
{
   PROTECT;

   all_windows = REMOB(vw,all_windows);

   UNPROTECT;

   vw->window = NULL;

   VIEW_mon_remove(vw);
};





/************************************************************************/
/*									*/
/*	request_window_type -- get window type from user		*/
/*									*/
/************************************************************************/


static VIEW_WTYPE
request_window_type(w)
   ASH_WINDOW w;
{
   VIEW_WTYPE vwt[32];
   register Integer i,ct,wct;
   Character btn[32][64];
   String menu[40];
   Integer val;

   wct = VIEW_inq_window_types(32,vwt);
   if (wct == 0) return NULL;
   if (wct == 1) return vwt[0];

   ct = 0;
   menu[ct++] = "%CViewer Window Type\n";

   for (i = 0; i < wct; ++i) {
      sprintf(btn[i],"%%0.%do %s",i,vwt[i]->name);
      menu[ct++] = btn[i];
    };

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   val = 0;
   if (!STEMdialog(w,menu,&val)) return NULL;

   return vwt[val];
};





/************************************************************************/
/*									*/
/*	system_btn -- handle set system command 			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
system_btn(vw,menu,btn)
   VIEW_WIN vw;
   String menu;
   String btn;
{
   Character buf[128];

   if (vw == NULL || vw->window == NULL) return FALSE;

   if (vw->system == NULL) buf[0] = 0;
   else strcpy(buf,vw->system);

   if (!FIELDsystem_request(vw->window,vw->type->name,buf)) return FALSE;

   if (vw->system != NULL) SFREE(vw->system);
   if (buf[0] == 0 || STREQL(buf,"*")) vw->system = NULL;
   else vw->system = SALLOC(buf);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	quit_btn -- handle quit command 				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
quit_btn(vw,menu,btn)
   VIEW_WIN vw;
   String menu;
   String btn;
{
   if (vw == NULL || vw->window == NULL) return FALSE;

   ASHremove(vw->window);

   return TRUE;
};





/* end of viewwin.c */
