/************************************************************************/
/*                                                                      */
/*              auto.c                                                  */
/*                                                                      */
/*      Rotation/reflection automorphism finder for graphs              */
/*                                                                      */
/************************************************************************/


#include "auto.h"




/************************************************************************/
/*                                                                      */
/*      Local Storage                                                   */
/*                                                                      */
/************************************************************************/


#define MAX_NODE        512


static  Integer         nnode;
static  NODE_B *        node;

static  Integer         nclass;
static  CLASS_B *       class;

static  Integer *       sigma;
static  Integer         order;
static  Boolean         self;

static  Integer *       tau;
static  Integer *       bad_sigma;
static  Integer *       bad_tau;
static  Boolean         real_tau;

static  Integer         sig_cnt;
static  Integer         tau_cnt;

#define NOT_IN          0
#define IN_C            1
#define IN_F            2
#define IN_E            3
#define IN_W            4
#define IN_X            5

static  Sequence        freelist = NULL;




/************************************************************************/
/*                                                                      */
/*      Forward definitions                                             */
/*                                                                      */
/************************************************************************/


static  void            read_graph();
static  void            add_edge();
static  void            print_graph();
static  void            sort_sons();
static  void            equiv_class();
static  void            first_class();
static  void            compute_dsig();
static  Boolean         same_dsig();
static  Boolean         node_match();
static  void            classify();
static  Integer         find_sigma();
static  Boolean         n_okay();
static  Boolean         test_sigmas();
static  Boolean         insert_node();
static  Boolean         test_insert();
static  void            print_sigma();
static  Boolean         find_tau();
static  Boolean         insert_tau_node();
static  Boolean         test_tau_insert();
static  void            print_tau();
static  void            analyze_nodes();
static  void            print_categories();
static  void            print_stats();
static  Sequence        newlistcell();
static  Sequence        freelistcell();




/************************************************************************/
/*                                                                      */
/*      main -- main program                                            */
/*                                                                      */
/************************************************************************/


main(argc,argv)
   Integer argc;
   String argv[];
{
   Integer n;

   node = NULL;
   class = NULL;
   sigma = NULL;
   tau = NULL;
   bad_sigma = NULL;
   bad_tau = NULL;

   for (; ;) {
      read_graph();
      print_graph();
      equiv_class();
      classify();
      sig_cnt = 0;
      tau_cnt = 0;
      order = find_sigma();
      print_sigma();
      real_tau = find_tau();
      print_tau();
      analyze_nodes();
      print_categories();
      print_stats();
    };
};




/************************************************************************/
/*                                                                      */
/*      read_graph -- input the graph                                   */
/*                                                                      */
/************************************************************************/


static void
read_graph()
{
   Integer i;
   Integer x,y;

   if (scanf("%d",&nnode) != 1) exit(0);
   if (nnode <= 0) exit(0);

   if (node != NULL) free(node);
   node = (NODE_B *) calloc(nnode,sizeof(NODE_B));

   for (i = 0; i < nnode; ++i) node[i].id = i+1;

   while (scanf("%d %d",&x,&y) == 2) {
      if (x <= 0) break;
      if (x > nnode || y < 0 || y > nnode) {
	 fprintf(stderr,"Bad edge %d %d\n",x,y);
	 exit(1);
       };
      add_edge(x-1,y-1);
      if (x != y) add_edge(y-1,x-1);
    };
};





/************************************************************************/
/*                                                                      */
/*      add_edge -- add edge to graph                                   */
/*                                                                      */
/************************************************************************/


static void
add_edge(x,y)
   Integer x,y;
{
   NODE n;

   n = &node[x];
   if (MEMQ(y,n->sons)) return;

   n->sons = CONS(y,n->sons);
   n->nson++;
};





/************************************************************************/
/*                                                                      */
/*      print_graph -- print the input graph                            */
/*                                                                      */
/************************************************************************/


static void
print_graph()
{
   static Integer ctr = 0;
   Integer i,j;
   SeqOfInteger l;

   for (i = 0; i < nnode; ++i) node[i].class = i;

   printf("GRAPH %d:\n",++ctr);

   printf("\tNodes = %d\n",nnode);

   for (i = 0; i < nnode; ++i) {
      sort_sons(i);
      printf("\t\t%d\t",i+1);
      forin (j,Integer,l,node[i].sons) {
	 printf("%d ",j+1);
       };
      printf("\n");
    };

   printf("\n\n");
};





/************************************************************************/
/*                                                                      */
/*      sort_sons -- sort the sons of a node by their class number      */
/*                                                                      */
/************************************************************************/


static void
sort_sons(i)
   Integer i;
{
   SeqOfInteger ol,l,nl;
   Integer x;

   nl = NULL;

   forin (x,Integer,ol,node[i].sons) {
      if (nl == NULL || node[x].class <= node[CAR(Integer,nl)].class) {
	 nl = CONS(x,nl);
       }
      else {
	 l = nl;
	 while (CDR(l) != NULL &&
		   node[x].class > node[CADR(Integer,l)].class) {
	    l = CDR(l);
	  };
	 RPLACD(l,CONS(x,CDR(l)));
       };
    };

   LFREE(node[i].sons);
   node[i].sons = nl;
};




/************************************************************************/
/*                                                                      */
/*      equiv_class -- get equivalence classes                          */
/*                                                                      */
/************************************************************************/


static void
equiv_class()
{
   Integer clsct,altcls;
   Integer i,j,e;
   SeqOfInteger l;
   Boolean fg;

   first_class();
   clsct = nnode;

   fg = TRUE;
   while (fg) {
      fg = FALSE;
      for (i = 0; i < nnode; ++i) {
	 sort_sons(i);
	 node[i].done = FALSE;
       };
      for (i = 0; i < nnode; ++i) {
	 if (node[i].done) continue;
	 altcls = -1;
	 for (j = i+1; j < nnode; ++j) {
	    if (node[j].class == node[i].class) {
	       if (node_match(i,j)) {
		  node[j].done = TRUE;
		}
	       else {
		  fg = TRUE;
		  if (altcls < 0) altcls = clsct++;
		  node[j].class = altcls;
		  forin (e,Integer,l,node[j].sons)
		     if (!node[e].done && e != j) sort_sons(e);
		};
	     };
	  };
       };
    };
};





/************************************************************************/
/*                                                                      */
/*      first_class -- find initial equivalence classes                 */
/*                                                                      */
/************************************************************************/


static void
first_class()
{
   Integer i,j,cls;

   for (i = 0; i < nnode; ++i) {
      node[i].dsig = (Integer *) calloc(nnode,sizeof(Integer));
      compute_dsig(i);
    };

   cls = 0;
   for (i = 0; i < nnode; ++i) node[i].done = FALSE;

   for (i = 0; i < nnode; ++i) {
      if (node[i].done) continue;
      node[i].class = cls++;
      for (j = i+1; j < nnode; ++j) {
	 if (node[j].done) continue;
	 if (same_dsig(i,j)) {
	    node[j].class = node[i].class;
	    node[j].done = TRUE;
	  };
       };
    };

   for (i = 0; i < nnode; ++i) {
      free(node[i].dsig);
      node[i].dsig = NULL;
    };
};





/************************************************************************/
/*                                                                      */
/*      compute_dsig -- find distance signature                         */
/*                                                                      */
/************************************************************************/


static void
compute_dsig(n)
   Integer n;
{
   Integer i,j,k,cnt;
   SeqOfInteger l;

   for (i = 0; i < nnode; ++i) node[i].done = 0;

   node[n].done = 1;
   cnt = 1;

   for (k = 1; cnt < nnode; ++k) {
      for (i = 0; i < nnode; ++i) {
	 if (node[i].done == k) {
	    forin (j,Integer,l,node[i].sons) {
	       if (node[j].done == 0) {
		  ++node[n].dsig[k];
		  node[j].done = k+1;
		  ++cnt;
		};
	     };
	  };
       };
      if (node[n].dsig[k] == 0) {
	 ++k;
	 break;
       };
    };

   node[n].dsig[0] = k;
};





/************************************************************************/
/*                                                                      */
/*      same_dsig -- compare signatures                                 */
/*                                                                      */
/************************************************************************/


static Boolean
same_dsig(x,y)
   Integer x,y;
{
   Integer i;

   for (i = 0; i < node[x].dsig[0]; ++i) {
      if (node[x].dsig[i] != node[y].dsig[i]) return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*                                                                      */
/*      node_match -- compare two nodes for equivalence                 */
/*                                                                      */
/************************************************************************/


static Boolean
node_match(x,y)
   Integer x,y;
{
   SeqOfInteger l,la;
   Integer i,j;

   if (node[x].class != node[y].class) return FALSE;

   la = node[y].sons;
   forin (i,Integer,l,node[x].sons) {
      j = CAR(Integer,la);
      la = CDR(la);
      if (node[i].class != node[j].class) return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*                                                                      */
/*      classify -- set up class structures                             */
/*                                                                      */
/************************************************************************/


static void
classify()
{
   Integer i,j;

   if (class != NULL) free(class);

   class = (CLASS_B *) calloc(nnode,sizeof(CLASS_B));

   nclass = 0;

   for (i = 0; i < nnode; ++i) {
      for (j = 0; j < nclass; ++j) {
	 if (class[j].id == node[i].class) break;
       };
      if (j >= nclass) {
	 j = nclass++;
	 class[j].id = node[i].class;
       };
      ++class[j].nelt;
      class[j].elts = CONS(i,class[j].elts);
      node[i].class = j;
    };
};





/************************************************************************/
/*                                                                      */
/*      find_sigma -- find our rotation                                 */
/*                                                                      */
/************************************************************************/


static Integer
find_sigma()
{
   Integer i,n;

   if (sigma != NULL) free(sigma);

   sigma = (Integer *) calloc(nnode,sizeof(Integer));

   for (n = nnode; n >= 2; --n) {
      if (!n_okay(n)) continue;

      if (test_sigmas(n)) break;
    };

   if (n == 1) {
      for (i = 0; i < nnode; ++i) sigma[i] = i;
    };

   return n;
};





/************************************************************************/
/*                                                                      */
/*      n_okay -- check if an n-rotation is acceptable on the surface   */
/*                                                                      */
/************************************************************************/


static Boolean
n_okay(n)
   Integer n;
{
   Integer i,fg;
   SeqOfInteger l;
   Integer mn,mx,e,cnt;

   self = FALSE;

   for (i = 0; i < nclass; ++i) {
      class[i].self = FALSE;
      if ((class[i].nelt % n) == 0) ;
      else if (self) return FALSE;
      else if ((class[i].nelt % n) == 1) {
	 self = TRUE;
	 class[i].self = TRUE;
       }
      else return FALSE;
    };

   if (self) {
      mn = nnode;
      mx = 0;
      for (i = 0; i < nnode; ++i) {
	 if (!class[node[i].class].self) continue;
	 cnt = 0;
	 forin (e,Integer,l,node[i].sons) if (e != i) ++cnt;
	 if (cnt < mn) mn = cnt;
	 if (cnt > mx) mx = cnt;
       };
      if (mn != 0 && mx != 0 && n > mx) return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*                                                                      */
/*      test_sigmas -- try to find an appropriate rotation              */
/*                                                                      */
/************************************************************************/


static Boolean
test_sigmas(n)
   Integer n;
{
   Integer i;

   for (i = 0; i < nnode; ++i) {
      node[i].done = FALSE;
      node[i].cycle = 0;
      node[i].base = -1;
      sigma[i] = -1;
    };

   self = FALSE;

   for (i = 0; i < nnode; ++i) {
      if (!insert_node(i,n)) return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*                                                                      */
/*      insert_node -- insert node into sigma                           */
/*                                                                      */
/************************************************************************/


static Boolean
insert_node(k,n)
   Integer k;
   Integer n;
{
   Integer i,j,e,f,g,h;
   SeqOfInteger l,la;
   Integer ncand,mlen;
   Integer cand[MAX_NODE];
   Boolean slf;

   if (sigma[k] >= 0) return TRUE;

   ++sig_cnt;

   ncand = -1;

   if (node[k].base < 0) mlen = n-2;
   else if (node[node[k].base].cycle == n-1) {
      ncand = 1;
      cand[0] = node[k].base;
      mlen = nnode;
    }
   else mlen = n-2-node[node[k].base].cycle;

   if (!self && node[k].base < 0 && class[node[k].class].self) slf = TRUE;
   else slf = FALSE;

   forin (e,Integer,l,node[k].sons) {
      f = sigma[e];
      if (f >= 0) {
	 if (ncand < 0) {               /* first set of candidates      */
	    ncand = 0;
	    forin (g,Integer,la,node[f].sons) {
	       if (node[g].cycle <= mlen && !node[g].done &&
		      node[g].class == node[k].class &&
		      node[k].base != g &&
		      (g != k || slf))
		  cand[ncand++] = g;
	     };
	  }
	 else {
	    j = 0;
	    for (i = 0; i < ncand; ++i) {
	       if (MEMQ(cand[i],node[f].sons)) {
		  if (j != 0) cand[i-j] = cand[i];
		}
	       else ++j;
	     };
	    ncand -= j;
	  };
	 if (ncand == 0) break;
       };
    };

   if (ncand < 0) {
      ncand = 0;
      forin (g,Integer,l,class[node[k].class].elts) {
	 if (node[g].cycle <= mlen && !node[g].done && (g != k || slf) &&
		node[k].base != g)
	    cand[ncand++] = g;
       };
    };

   for (e = 0; e < nnode; ++e) {
      f = sigma[e];
      if (f >= 0 && !MEMQ(e,node[k].sons)) {
	 j = 0;
	 for (i = 0; i < ncand; ++i) {
	    if (!MEMQ(cand[i],node[f].sons)) {
	       if (j != 0) cand[i-j] = cand[i];
	     }
	    else ++j;
	  };
	 ncand -= j;
       };
      if (ncand == 0) break;
    };

   for (i = 0; i < ncand; ++i) {
      if (k == 0)
	 g = 0;
      if (test_insert(k,cand[i],n)) return TRUE;
    };

   return FALSE;
};




/************************************************************************/
/*                                                                      */
/*      test_insert -- try inserting a node into sigma                  */
/*                                                                      */
/************************************************************************/


static Boolean
test_insert(k,v,n)
   Integer k;
   Integer v;
   Integer n;
{
   Integer e,i,j,g;
   SeqOfInteger l;
   Integer obase,ocycle,tot;
   Integer next[MAX_NODE];
   Integer known[MAX_NODE];

   obase = node[k].base;
   ocycle = node[k].cycle;

   if (k == v) self = TRUE;
   node[v].done = TRUE;

   node[k].cycle = node[v].cycle+1;
   if (obase < 0) node[k].base = k;

   for (e = node[k].base; e != k; e = sigma[e])
      node[e].cycle += node[k].cycle;
   for (e = v; e >= 0; e = sigma[e]) node[e].base = node[k].base;

   sigma[k] = v;

   tot = 0;
   for (i = 0; i < nnode; ++i) if (sigma[i] < 0) next[tot++] = i;

   for (i = 0; i < tot; ++i) {
      known[i] = 0;
      if (node[next[i]].base >= 0)
	 known[i] = node[node[next[i]].base].cycle;
      forin (e,Integer,l,node[next[i]].sons) {
	 if (sigma[e] >= 0) known[i] += nnode-node[e].cycle;
       };
    };

   for (i = 0; i < tot; ++i) {
      for (j = i+1; j < tot; ++j) {
	 if (known[j] > known[i]) {
	    g = next[i]; next[i] = next[j]; next[j] = g;
	    g = known[i]; known[i] = known[j]; known[j] = g;
	  };
       };
      e = next[i];

      if (sigma[e] < 0 && !insert_node(e,n)) break;
    };

   if (i >= tot) return TRUE;

   sigma[k] = -1;

   for (e = obase; e >= 0 && e != k; e = sigma[e]) {
      node[e].base = obase;
      node[e].cycle -= node[k].cycle;
    };

   if (sigma[v] >= 0) {
      for (e = v; e >= 0; e = sigma[e]) node[e].base = v;
    }
   else node[v].base = -1;

   node[k].base = obase;
   node[k].cycle = ocycle;

   node[v].done = FALSE;
   if (k == v) self = FALSE;

   return FALSE;
};







/************************************************************************/
/*                                                                      */
/*      print_sigma -- output rotation                                  */
/*                                                                      */
/************************************************************************/


static void
print_sigma()
{
   Integer i;

   printf("\tSIGMA of order %d:\n\t",order);
   for (i = 0; i < nnode; ++i) printf("%3d ",i+1);

   printf("\n\t");
   for (i = 0; i < nnode; ++i) printf("%3d ",sigma[i]+1);

   printf("\n\n");
};





/************************************************************************/
/*                                                                      */
/*      find_tau -- find reflection                                     */
/*                                                                      */
/************************************************************************/


static Boolean
find_tau()
{
   Integer i,j,e,f;
   Integer cnt;
   SeqOfInteger l;
   Integer sig1[MAX_NODE];

   if (tau != NULL) free(tau);
   tau = (Integer *) calloc(nnode,sizeof(Integer));
   if (bad_tau != NULL) free(bad_tau);
   bad_tau = (Integer *) calloc(nnode,sizeof(Integer));

   if (bad_sigma != NULL) free(bad_sigma);
   bad_sigma = NULL;
   if (order%2 == 0) {
      bad_sigma = (Integer *) calloc(nnode,sizeof(Integer));
      for (i = 0; i < nnode; ++i) bad_sigma[i] = sigma[i];
      for (j = 1; j < order/2; ++j) {
	 for (i = 0; i < nnode; ++i) sig1[i] = bad_sigma[i];
	 for (i = 0; i < nnode; ++i) bad_sigma[i] = sig1[sigma[i]];
       };
    };

   for (i = 0; i < nnode; ++i) {
      node[i].done = FALSE;
      node[i].cycle = -1;
      node[i].base = -1;
      tau[i] = -1;
      bad_tau[i] = -1;
    };

   self = FALSE;

   cnt = 0;
   for (i = 0; i < nclass; ++i) {
      forin (e,Integer,l,class[i].elts) {
	 if (node[e].cycle < 0) {
	    for (f = sigma[e]; f != e; f = sigma[f]) node[f].cycle = cnt;
	    node[e].cycle = cnt++;
	  };
       };
    };

   for (i = 0; i < nnode; ++i) {
      if (!insert_tau_node(i)) break;
    };

   if (i < nnode && bad_tau[0] >= 0) {
      for (i = 0; i < nnode; ++i) tau[i] = bad_tau[i];
      i = nnode;
    };

   if (i < nnode) {
      for (i = 0; i < nnode; ++i) tau[i] = i;
    };

   for (i = 0; i < nnode; ++i) {
      if (tau[i] != i) break;
    };

   return (i < nnode);
};





/************************************************************************/
/*                                                                      */
/*      insert_tau_node -- insert node into tau                         */
/*                                                                      */
/************************************************************************/


static Boolean
insert_tau_node(k)
   Integer k;
{
   Integer i,j,e,f,g,h;
   SeqOfInteger l,la;
   Integer ncand,mlen;
   Integer cand[MAX_NODE];
   Boolean fg;

   if (tau[k] >= 0) return TRUE;

   ++tau_cnt;

   i = node[k].cycle;
   i = node[i].base;
   ncand = 0;
   forin (e,Integer,l,class[node[k].class].elts) {
      if (node[k].done && tau[e] != k) continue;
      if (!node[k].done && tau[e] >= 0) continue;
      if (!node[e].done)
	 cand[ncand++] = e;
    };

   if (ncand == 0) return FALSE;

   forin (e,Integer,l,node[k].sons) {
      f = tau[e];
      if (f >= 0) {
	 j = 0;
	 for (i = 0; i < ncand; ++i) {
	    if (MEMQ(cand[i],node[f].sons)) {
	       if (j != 0) cand[i-j] = cand[i];
	     }
	    else ++j;
	  };
	 ncand -= j;
	 if (ncand == 0) break;
       };
    };

   for (e = 0; e < nnode; ++e) {
      f = tau[e];
      if (f >= 0 && !MEMQ(e,node[k].sons)) {
	 j = 0;
	 for (i = 0; i < ncand; ++i) {
	    if (!MEMQ(cand[i],node[f].sons)) {
	       if (j != 0) cand[i-j] = cand[i];
	     }
	    else ++j;
	  };
	 ncand -= j;
       };
      if (ncand == 0) break;
    };

   for (i = 0; i < ncand; ++i) {
      if (cand[i] == k) {
	 e = (self ? 0 : ncand-1);
	 cand[i] = cand[e];
	 cand[e] = k;
	 break;
       };
    };

   for (i = 0; i < ncand; ++i) {
      if (test_tau_insert(k,cand[i])) return TRUE;
    };

   return FALSE;
};




/************************************************************************/
/*                                                                      */
/*      test_tau_insert -- try inserting a node into tau                */
/*                                                                      */
/************************************************************************/


static Boolean
test_tau_insert(k,v)
   Integer k;
   Integer v;
{
   Integer e,i,j,g;
   Integer tot;
   SeqOfInteger l;
   Integer base1,base2,c1,c2;
   Integer next[MAX_NODE];
   Integer known[MAX_NODE];
   Boolean oself;

   c1 = node[k].cycle;
   c2 = node[v].cycle;
   base1 = node[c1].base;
   base2 = node[c2].base;

   node[v].done = TRUE;

   node[c1].base = c2;
   node[c2].base = c1;

   oself = self;
   if (k != v) self = TRUE;

   tau[k] = v;

   tot = 0;
   for (i = 0; i < nnode; ++i) if (tau[i] < 0) next[tot++] = i;

   for (i = 0; i < tot; ++i) {
      known[i] = 0;
      forin (e,Integer,l,node[next[i]].sons)
	 if (tau[e] >= 0) ++known[i];
    };

   for (i = 0; i < tot; ++i) {
      for (j = i+1; j < tot; ++j) {
	 if (known[j] > known[i]) {
	    g = next[i]; next[i] = next[j]; next[j] = g;
	    g = known[i]; known[i] = known[j]; known[j] = g;
	  };
       };
      e = next[i];

      if (tau[e] < 0 && !insert_tau_node(e)) break;
    };

   if (i >= tot) {
      if (bad_sigma != NULL) {
	 for (j = 0; j < nnode; ++j) {
	    if (tau[j] != bad_sigma[j]) break;
	  };
	 if (j >= nnode) i = -1;
       };
      if (i >= tot) {
	 for (j = 0; j < nnode; ++j) {
	    if (tau[j] < 0 || tau[j] == j) break;
	  };
	 if (j >= nnode) {
	    i = -1;
	    if (bad_tau[0] < 0)
	       for (j = 0; j < nnode; ++j) bad_tau[j] = tau[j];
	  };
       };
    };

   if (i >= tot) return TRUE;

   tau[k] = -1;

   self = oself;

   node[c1].base = base1;
   node[c2].base = base2;

   node[v].done = FALSE;

   return FALSE;
};






/************************************************************************/
/*                                                                      */
/*      print_tau -- output reflection                                  */
/*                                                                      */
/************************************************************************/


static void
print_tau()
{
   Integer i;

   printf("\tTAU:\n\t");
   for (i = 0; i < nnode; ++i) printf("%3d ",i+1);

   printf("\n\t");
   for (i = 0; i < nnode; ++i) printf("%3d ",tau[i]+1);

   printf("\n\n");
};





/************************************************************************/
/*                                                                      */
/*      analyze_nodes -- categorize nodes by sigma and tau              */
/*                                                                      */
/************************************************************************/


static void
analyze_nodes()
{
   Integer i,k,e,f;

   for (i = 0; i < nnode; ++i) {
      node[i].cycle = 0;
      node[i].base = NOT_IN;
    };

   if (order != 1) {
      for (i = 0; i < nnode; ++i) {
	 if (sigma[i] == i) node[i].base = IN_C;
       };
    };

   if (real_tau) {
      for (i = 0; i < nnode; ++i) {
	 if (tau[i] == i && node[i].base != IN_C) {
	    node[i].base = IN_F;
	    k = 1;
	    for (e = sigma[i]; e != i; e = sigma[e]) {
	       node[e].base = IN_E;
	       node[e].cycle = k++;
	     };
	  };
       };
    };

   for (i = 0; i < nnode; ++i) {
      if (node[i].base == NOT_IN) {
	 node[i].base = IN_W;
	 k = 1;
	 for (e = sigma[i]; e != i; e = sigma[e]) {
	    node[e].base = IN_W;
	    node[e].cycle = k;
	    f = tau[e];
	    if (f != e && node[f].base == NOT_IN) {
	       node[f].base = IN_X;
	       node[f].cycle = k;
	     };
	    ++k;
	  };
       };
    };
};





/************************************************************************/
/*                                                                      */
/*      print_categories -- print nodes by category                     */
/*                                                                      */
/************************************************************************/


static void
print_categories()
{
   Integer i;

   printf("\tCATEGORIES:\n");

   for (i = 0; i < nnode; ++i) {
      printf("\t\t%d\t",i+1);
      switch (node[i].base) {
	 case IN_C :
	    printf("Center");
	    break;
	 case IN_F :
	    printf("Reflection Invariant");
	    break;
	 case IN_E :
	    printf("Reflection Mapped %d",node[i].cycle);
	    break;
	 case IN_W :
	    printf("Rotation %d",node[i].cycle);
	    break;
	 case IN_X :
	    printf("Rotation Reflection %d",node[i].cycle);
	    break;
       };
      printf("\n");
    };

   printf("\n\n");
};





/************************************************************************/
/*                                                                      */
/*      print_stats -- print statistics                                 */
/*                                                                      */
/************************************************************************/


static void
print_stats()
{
   printf("\tFor %d nodes there were %d classes\n",nnode,nclass);
   printf("\tSigma insert %d times\n",sig_cnt);
   printf("\tTau insert %d times\n",tau_cnt);
   printf("\n\n");
};





/************************************************************************/
/*                                                                      */
/*      List processing routines                                        */
/*                                                                      */
/************************************************************************/


Sequence
CONS(item,list)
   Universal item;
   Sequence list;
{
   Sequence l;

   l = newlistcell();
   l->Lcar = item;
   l->Lcdr = list;

   return l;
};


Sequence
RPLACD(l,v)
   Sequence l;
   Sequence v;
{
   l->Lcdr = v;

   return l;
};


Boolean
MEMQ(item,list)
   Universal item;
   Sequence list;
{
   Sequence l;

   for (l = list; !EMPTY(l); l = CDR(l))
      if (CAR(Universal,l) == item) return TRUE;

   return FALSE;
};



LFREE(list)
   Sequence list;
{
   Sequence l;

   while (list != NULL)
    { l = list;
      list = CDR(list);
      freelistcell(l);
    };
};






/************************************************************************/
/*                                                                      */
/*      listalloc -- allocate list block                                */
/*                                                                      */
/************************************************************************/


#define LBLKSIZE (512-2)
#define rplaca(list,v)  ((list)->Lcar) = ((Universal)(v))
#define rplacd(list,v)  ((list)->Lcdr) = ((Sequence)(v))



static Sequence
listalloc()
{
   SequenceB *lb;
   Integer i;

   lb = (SequenceB *) malloc(LBLKSIZE*sizeof(SequenceB));

   for (i = 1; i < LBLKSIZE; ++i)
      rplacd(&lb[i-1],&lb[i]);

   rplacd(&lb[LBLKSIZE-1],freelist);
   freelist = &lb[1];
   rplaca(&lb[0],0);
   rplacd(&lb[0],0);

   return &lb[0];
};





/************************************************************************/
/*                                                                      */
/*      newlistcell -- allocate list cell                               */
/*      freelistcell-- remove list cell                                 */
/*                                                                      */
/************************************************************************/


static Sequence
newlistcell()
{
   register Sequence l;

   if (freelist == NULL) l = listalloc();
   else {
      l = freelist;
      freelist = CDR(freelist);
    };

   return l;
};





static Sequence
freelistcell(l)
   Sequence l;
{
   rplacd(l,freelist);
   freelist = l;
};





/* end of auto.c */
