/************************************************************************/
/*									*/
/*		pmatmatch.c						*/
/*									*/
/*	Pattern matching routines					*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "pmat_local.h"




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	scan_pattern();
static	Integer 	scan_field();
static	Boolean 	match_arg();





/************************************************************************/
/*									*/
/*	PMATmake_pattern -- define a pattern from a pattern string	*/
/*									*/
/************************************************************************/


PMAT_PAT
PMATmake_pattern(str,ct,dflts)
   String str;
   Integer ct;
   Universal dflts[];
{
   register PMAT_PAT pp;

   pp = PALLOC(PMAT_PAT_INFO);

   if (!scan_pattern(str,ct,dflts,pp)) return NULL;

   return pp;
};





/************************************************************************/
/*									*/
/*	PMATfree_pattern -- free a pattern				*/
/*									*/
/************************************************************************/


void
PMATfree_pattern(pp)
   PMAT_PAT pp;
{
   register Integer i;

   if (pp == NULL) return;

   SFREE(pp->pattern);
   if (pp->arg != NULL) {
      for (i = 0; i < pp->numargs; ++i) {
	 switch (pp->arg[i].type) {
	    case PMAT_TYPE_STRING :
	       if (pp->arg[i].dflt != NULL) SFREE(pp->arg[i].dflt);
	       break;
	    case PMAT_TYPE_PATTERN :
	    case PMAT_TYPE_NPATTERN :
	       if (pp->arg[i].match != NULL) SFREE(pp->arg[i].match);
	       if (pp->arg[i].dflt != NULL) SFREE(pp->arg[i].dflt);
	       break;
	    default :
	       break;
	  };
       };
      free(pp->arg);
    };

   free(pp);
};





/************************************************************************/
/*									*/
/*	PMATmatch -- match a pattern					*/
/*									*/
/************************************************************************/


Integer
PMATmatch(txt,pp,args)
   String txt;
   PMAT_PAT pp;
   Universal args[];
{
   register String p;
   Boolean set[MAX_ARGS];
   register Universal * ap;
   register Integer i;

   p = pp->pattern;

   for (i = 0; i < pp->retargs; ++i) set[i] = FALSE;

   for (p = pp->pattern; *p != 0; ++p) {
      if (*p != '%') {
	 while (*txt == LIT_STRING) ++txt;
	 if (*txt++ != *p) return -1;
       }
      else {
	 ++p;
	 switch (*p) {
	    case '%' :
	       if (*txt++ != '%') return -1;
	       break;
	    case ' ' :
	       if (!isspace(*txt++)) return -1;
	       while (isspace(*txt)) ++txt;
	       break;
	    case 0 :
	       if (*txt != 0) return -1;
	       else --p;
	       break;
	    case 'A' :
	       i = (*++p) - 1;
	       if (args == NULL) ap = NULL;
	       else ap = &args[i];
	       if (!match_arg(&txt,&pp->arg[i],ap)) return -1;
	       if (!pp->arg[i].dummy) set[i] = TRUE;
	       break;
	    default :
	       return -1;
	  };
       };
    };

   if (args != NULL) {
      for (i = 0; i < pp->retargs; ++i) {
	 if (!set[i]) {
	    switch (pp->arg[i].type) {
	       case PMAT_TYPE_STRING :
	       case PMAT_TYPE_QUOTES :
	       case PMAT_TYPE_PATTERN :
	       case PMAT_TYPE_NPATTERN :
		  if (pp->arg[i].dflt != NULL)
		     args[i] = (Universal) SALLOC(pp->arg[i].dflt);
		  else args[i] = NULL;
		  break;
	       default :
		  args[i] = pp->arg[i].dflt;
		  break;
	     };
	  };
       };
    };

   return pp->retargs;
};





/************************************************************************/
/*									*/
/*	PMATfree_args -- free arguments from a pattern			*/
/*									*/
/************************************************************************/


void
PMATfree_args(pp,args)
   PMAT_PAT pp;
   Universal args[];
{
   Integer i;

   if (args != NULL) {
      for (i = 0; i < pp->retargs; ++i) {
	 switch (pp->arg[i].type) {
	    case PMAT_TYPE_STRING :
	    case PMAT_TYPE_QUOTES :
	    case PMAT_TYPE_PATTERN :
	    case PMAT_TYPE_NPATTERN :
	       if (args[i] != NULL) SFREE(args[i]);
	       break;
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	scan_pattern -- preprocess a pattern string			*/
/*									*/
/************************************************************************/


static Boolean
scan_pattern(pat,ct,dflts,pp)
   String pat;
   Integer ct;
   Universal dflts[];
   PMAT_PAT pp;
{
   register String t;
   register Integer i;
   Character buf[10240];
   PMAT_ARG_INFO args[MAX_TOTAL];
   Integer mxargs;

   for (i = 0; i < MAX_TOTAL; ++i) {
      args[i].type = PMAT_TYPE_NONE;
      args[i].base = 0;
      args[i].dummy = (i >= ct);
      args[i].length = 0;
      if (dflts != NULL && i < ct) args[i].dflt = dflts[i];
      else args[i].dflt = NULL;
      args[i].match = NULL;
    };
   mxargs = ct;

   t = buf;

   while (*pat != 0) {
      i = *pat++;
      *t++ = i;
      if (i == '%') {
	 switch (*pat) {
	    case '%' :
	    case ' ' :
	       *t++ = *pat++;
	       break;
	    case 0 :
	       break;
	    default :
	       i = scan_field(&pat,args,mxargs);
	       if (i < 0) return FALSE;
	       if (i >= mxargs) mxargs = i+1;
	       *t++ = 'A';
	       *t++ = i+1;
	       break;
	  };
       };
    };

   *t = 0;
   pp->pattern = SALLOC(buf);
   pp->numargs = mxargs;
   pp->retargs = ct;
   if (mxargs == 0) pp->arg = NULL;
   else pp->arg = (PMAT_ARG_INFO *) calloc(mxargs,sizeof(PMAT_ARG_INFO));

   for (i = 0; i < mxargs; ++i) {
      pp->arg[i] = args[i];
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	scan_field -- preprocess a field definition			*/
/*									*/
/************************************************************************/


static Integer
scan_field(pp,args,mx)
   String * pp;
   PMAT_ARG_INFO args[];
   Integer mx;
{
   register String p;
   register Integer base;
   Integer argno,len;
   Boolean usefg;
   Character buf[64];
   register String bufp;

   p = *pp;

   if (*p == '*') {
      usefg = FALSE;
      ++p;
    }
   else usefg = TRUE;

   if (usefg && isdigit(*p)) {
      argno = *p++ - '0';
      while (isdigit(*p)) argno = argno*10 + *p++ - '0';
      if (argno == 0) argno = mx;
      else argno -= 1;
    }
   else argno = mx;

   if (*p == '.') ++p;

   if (isdigit(*p)) {
      len = *p++ - '0';
      while (isdigit(*p)) len = len*10 + *p++ - '0';
    }
   else len = 0;
   args[argno].length = len;

   if (*p == 'l' || *p == 'h') ++p;

   switch (*p) {
      case 'D' :
      case 'd' :
	 base = 10;
	 break;
      case 'o' :
      case 'O' :
	 base = 8;
	 break;
      case 'x' :
      case 'X' :
	 base = 16;
	 break;
      case 'e' :
      case 'E' :
      case 'f' :
      case 'F' :
      case 'g' :
      case 'G' :
	 base = -1;
	 break;
      default :
	 base = 0;
	 break;
    };

   if (base > 0) {
      args[argno].type = PMAT_TYPE_INT;
      args[argno].base = base;
    }
   else if (*p == 'c') {
      args[argno].type = PMAT_TYPE_CHAR;
    }
   else if (*p == 's') {
      args[argno].type = PMAT_TYPE_STRING;
      if (*(p+1) == '%') args[argno].match = (Universal) 0;
      else args[argno].match = (Universal) *(p+1);
      if (args[argno].dflt != NULL)
	 args[argno].dflt = (Universal) SALLOC(args[argno].dflt);
    }
   else if (*p == 'r') {
      args[argno].type = PMAT_TYPE_STRING;
      args[argno].match = (Universal) -1;
      if (args[argno].dflt != NULL)
	 args[argno].dflt = (Universal) SALLOC(args[argno].dflt);
    }
   else if (*p == 'q') {
      args[argno].type = PMAT_TYPE_QUOTES;
      if (args[argno].dflt != NULL)
	 args[argno].dflt = (Universal) SALLOC(args[argno].dflt);
    }
   else if (base = -1) {
      args[argno].type = PMAT_TYPE_FLOAT;
    }
   else if (*p == '[') {
      if (args[argno].dflt != NULL)
	 args[argno].dflt = (Universal) SALLOC(args[argno].dflt);
      ++p;
      bufp = buf;
      if (*p != '^') args[argno].type = PMAT_TYPE_PATTERN;
      else {
	 args[argno].type = PMAT_TYPE_NPATTERN;
	 ++p;
       };
      while (*p != ']' && *p != 0) *bufp++ = *p++;
      if (*p == 0) return -1;
      *bufp = 0;
      args[argno].match = (Universal) SALLOC(buf);
    }
   else return -1;

   if (!usefg) args[argno].dummy = TRUE;

   ++p;

   *pp = p;

   return argno;
};





/************************************************************************/
/*									*/
/*	match_arg -- check for match of argument			*/
/*									*/
/************************************************************************/


static Boolean
match_arg(sp,pa,argp)
   String * sp;
   PMAT_ARG pa;
   Universal * argp;
{
   register String s;
   register Integer i,v,base,j;
   Integer len,mode,sign;
   Character buf[20480];
   register String bufp;
   Universal u;
   double f;
   Integer pct,qct;

   s = *sp;
   len = pa->length;

   switch (pa->type) {
      case PMAT_TYPE_INT :
	 base = pa->base;
	 sign = 1;
	 mode = 0;
	 if (len > 0) {
	    v = 0;
	    for (i = 0; i < len; ++i) {
	       if (*s == ' ') v = v*base;
	       else if (*s == '+') {
		  if (mode != 0) return FALSE;
		}
	       else if (*s == '-') {
		  if (mode != 0) return FALSE;
		  sign = -1;
		}
	       else if (isdigit(*s)) v = v*base + *s++ - '0';
	       else if (base > 10 && isalpha(*s)) {
		  if (isupper(*s)) j = *s++ - 'A' + 10;
		  else j = *s++ - 'a' + 10;
		  if (j >= base) return FALSE;
		  v = v*base + j;
		}
	       else return FALSE;
	       mode = 1;
	     };
	  }
	 else {
	    v = 0;
	    if (*s == '0' && *(s+1) == 'x') {
	       s += 2;
	       base = 16;
	     }
	    else if (*s == '0') base = 8;

	    for ( ; ; ) {
	       if (isdigit(*s)) v = v*base + *s++ - '0';
	       else if (*s == '+') {
		  ++s;
		  if (mode != 0) return FALSE;
		}
	       else if (*s == '-') {
		  ++s;
		  if (mode != 0) return FALSE;
		  sign = -1;
		}
	       else if (base > 10 && isalpha(*s)) {
		  if (isupper(*s)) j = *s++ - 'A' + 10;
		  else j = *s++ - 'a' + 10;
		  if (j >= base) break;
		  v = v*base + j;
		}
	       else break;
	       mode = 1;
	     };
	  };
	 if (mode == 0) return FALSE;

	 u = (Universal) v*sign;
	 break;

      case PMAT_TYPE_CHAR :
	 if (len < 2) len = 1;
	 v = 0;
	 for (i = 0; i < len; ++i) {
	    if (*s == 0) return FALSE;
	    v = (v << 8) | *s++;
	  };
	 u = (Universal) v;
	 break;

      case PMAT_TYPE_STRING :
	 bufp = buf;
	 if (len > 0) {
	    for (i = 0; i < len; ++i) {
	       v = *s++;
	       if (v == 0) return FALSE;
	       *bufp++ = v;
	     };
	  }
	 else if (((int) pa->match) == ')' && *s != LIT_STRING) {
	    pct = 0;
	    qct = 0;
	    while (*s != 0) {
	       if (qct == 0 && *s == ')' && --pct < 0) break;
	       if (*s == '"' || *s == '\'') qct = !qct;
	       else if (qct == 0 && *s == '(') ++pct;
	       *bufp++ = *s++;
	     };
	  }
	 else {
	    j = (int) pa->match;
	    if (j == ' ') j = 0;
	    if (*s == LIT_STRING && j >= 0) {
	       ++s;
	       while (*s != LIT_STRING && *s != 0) *bufp++ = *s++;
	       if (*s != 0) ++s;
	     }
	    else {
	       while (*s != 0 &&
			 ((j != 0 && *s != j) || (j == 0 && !isspace(*s)))) {
		  *bufp++ = *s++;
		};
	     };
	  };
	 *bufp = 0;
	 u = (Universal) SALLOC(buf);
	 break;

      case PMAT_TYPE_QUOTES :
	 if (*s != '"') return FALSE;
	 bufp = buf;
	 *bufp++ = *s++;
	 while (*s != 0 && *s != '"') {
	    *bufp++ = *s++;
	  };
	 if (*s != '"') return FALSE;
	 *bufp++ = *s++;
	 *bufp = 0;
	 u = (Universal) SALLOC(buf);
	 break;

      case PMAT_TYPE_FLOAT :
	 bufp = buf;
	 mode = 0;
	 if (len > 0) {
	    for (i = 0; i < len; ++i) {
	       if (!isdigit(*s)) {
		  if (*s == '+' || *s == '-') {
		     if (mode == 0 || mode == 3) ++mode;
		     else return FALSE;
		   }
		  else if (*s == '.') {
		     if (mode < 2) mode = 2;
		     else return FALSE;
		   }
		  else if (*s == 'e' || *s == 'E') {
		     if (mode < 3) mode = 3;
		     else return FALSE;
		   }
		  else return FALSE;
		};
	       *bufp++ = *s++;
	     };
	  }
	 else {
	    for ( ; ; ) {
	       if (!isdigit(*s)) {
		  if (*s == '+' || *s == '-') {
		     if (mode == 0 || mode == 3) ++mode;
		     else break;
		   }
		  else if (*s == '.') {
		     if (mode < 2) mode = 2;
		     else break;
		   }
		  else if (*s == 'e' || *s == 'E') {
		     if (mode < 3) mode = 3;
		     else break;
		   }
		  else break;
		};
	       *bufp++ = *s++;
	     };
	  };
	 if (mode == 1 || mode == 3) return FALSE;
	 *bufp = 0;
	 f = atof(buf);
	 bcopy(&f,&u,8);
	 break;

      case PMAT_TYPE_PATTERN :
      case PMAT_TYPE_NPATTERN :
	 bufp = buf;
	 mode = (pa->type == PMAT_TYPE_PATTERN ? 0 : 1);

	 if (len > 0) {
	    for (i = 0; i < len; ++i) {
	       if (*s == 0) return FALSE;
	       if (mode == 0) {
		  if (index((String) pa->match,*s) == NULL) return FALSE;
		}
	       else {
		  if (index((String) pa->match,*s) != NULL) return FALSE;
		};
	       *bufp++ = *s++;
	     };
	  }
	 else {
	    for ( ; ; ) {
	       if (*s == 0) break;
	       if (mode == 0) {
		  if (index((String) pa->match,*s) == NULL) break;
		}
	       else {
		  if (index((String) pa->match,*s) != NULL) break;
		};
	       *bufp++ = *s++;
	     };
	  };
	 *bufp = 0;
	 u = (Universal) SALLOC(buf);
	 break;

      default :
	 return FALSE;
    };

   if (argp != NULL && !pa->dummy) {
      *argp = u;
    };

   *sp = s;

   return TRUE;
};





/* end of pmatmatch.c */
