/************************************************************************/
/*									*/
/*		msgclient.c						*/
/*									*/
/*	Client code for message handling				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "msg_local.h"

#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <pwd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <netdb.h>

#ifdef mips
#include <unistd.h>
#endif
#ifdef vax
#include <unistd.h>
#endif




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define STARTUP_TIME	1

#define REPLY_SIG	1

#define MSG_BUF_SIZE	1024		/* amt to read from server	*/
#define PRINT_SIZE	20480		/* for printf's on senda,calla  */

#ifdef mips
#ifndef INADDR_LOOPBACK
#define INADDR_LOOPBACK 	((unsigned long) 0x0100007f)
#endif
#endif
#ifdef vax
#ifndef INADDR_LOOPBACK
#define INADDR_LOOPBACK 	((unsigned long) 0x0100007f)
#endif
#endif




/************************************************************************/
/*									*/
/*	Data Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _MSG_PAT *	MSG_PAT;
typedef struct _MSG_CALL *	MSG_CALL;


typedef struct _MSG_PAT {
   String text;
   PMAT_PAT pattern;
   Function_Ptr routine;
   MSG_PAT next;
} MSG_PAT_INFO;




typedef struct _MSG_CALL {
   Character id;
   String rslt;
   Boolean done;
   MSG_CALL next;
} MSG_CALL_INFO;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	MSG_PAT 	all_mpats;
static	Boolean 	initialized = FALSE;
static	Integer 	server_id;
static	String		server_msg;
static	Integer 	server_msg_len;
static	String		next_read;
static	Integer 	call_id;
static	MSG_CALL	call_msgs;
static	Function_Ptr	wait_fct = NULL;
static	Function_Ptr	register_fct = NULL;

static	Boolean 	sync_mode;
static	Boolean 	sync_working;

static	String		lock_file = NULL;

static	THREAD_MONITOR	call_mon;
PROT_DECL;

extern	int		errno;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		msg_init();
static	Integer 	process_message();
static	Integer 	open_channel();
static	void		create_server();
static	void		send_to_server();
static	void		receive_from_server();
static	String		read_next_message();
static	void		get_lock_name();
static	void		fix_format();





/************************************************************************/
/*									*/
/*	MSGinit -- client code initialization				*/
/*	MSGlocal_init -- init for local use only			*/
/*	msg_init -- actual initialization code				*/
/*									*/
/************************************************************************/


void
MSGinit(fg)
   Boolean fg;
{
   msg_init();

   PROTECT;
   if (server_id == -2) {
      server_id = open_channel(fg);
      server_msg[0] = 0;
      next_read[0] = 0;
      MSG_file_init();
      MSG_util_init();
    };
   UNPROTECT;
};





/*ARGSUSED*/

void
MSGlocal_init(fg)
   Boolean fg;
{
   msg_init();

   PROTECT;
   if (server_id == -2) {
      server_id = -1;
    };
   UNPROTECT;
};





static void
msg_init()
{
   PROT_INIT;

   PROTECT;
   if (!initialized) {
      all_mpats = NULL;
      initialized = TRUE;
      server_id = -2;
      server_msg_len = 1000;
      server_msg = (String) malloc(server_msg_len+2);
      server_msg[0] = 0;
      next_read = (String) malloc(MSG_BUF_SIZE+2);
      next_read[0] = 0;
      call_id = 0;
      call_msgs = NULL;
      call_mon = NULL;
      if (USE_THREADS) call_mon = THREADmonitorinit(2,NULL);
      if (wait_fct == NULL) wait_fct = (Function_Ptr) CMPXselect;
      if (register_fct == NULL) register_fct = (Function_Ptr) CMPXregister;
      sync_mode = FALSE;
      sync_working = FALSE;
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	MSGset_lock_file -- set lock file name explicitly		*/
/*	MSGinq_lock_file -- return lock file name if not default	*/
/*	MSGset_application -- get lock file from arguments		*/
/*									*/
/************************************************************************/


void
MSGset_lock_name(lock)
   String lock;
{
   Character buf[1024];

   if (lock == NULL) lock_file = NULL;
   else if (lock[0] == '/') lock_file = SALLOC(lock);
   else if (getwd(buf) == 0) lock_file = NULL;
   else {
      strcat(buf,lock);
      lock_file = SALLOC(buf);
    };
};




String
MSGinq_lock_file()
{
   return lock_file;
};





Integer
MSGset_application(argc,argv)
   Integer argc;
   String argv[];
{
   Integer i,j,k;

   lock_file = NULL;

   k = 0;
   for (i = 1; i < argc; ++i) {
      j = strlen(argv[i]);
      if (j <= strlen(MSG_ARG) && strncmp(argv[i],MSG_ARG,j) == 0 &&
	     i+1 < argc && lock_file == NULL) {
	 lock_file = argv[++i];
	 k += 2;
       }
      else if (k != 0) argv[i-k] = argv[i];
    };

   if (k != 0) argv[argc-k] = 0;
   argc -= k;

   return argc;
};





/************************************************************************/
/*									*/
/*	MSGregister -- register a message pattern and a routine 	*/
/*									*/
/************************************************************************/


void
MSGregister(txt,rtn,ct,args)
   String txt;
   Function_Ptr rtn;
   Integer ct;
   Universal args[];
{
   String buf;
   MSG_PAT mp,mpl;

   if (txt == NULL || txt[0] == 0) return;

   MSGinit(FALSE);

   mp = PALLOC(MSG_PAT_INFO);

   mp->text = SALLOC(txt);
   mp->pattern = PMATmake_pattern(txt,ct,args);
   mp->routine = rtn;
   mp->next = NULL;

   PROTECT;
   if (all_mpats == NULL) all_mpats = mp;
   else {
      for (mpl = all_mpats; mpl->next != NULL; mpl = mpl->next);
      mpl->next = mp;
    };
   UNPROTECT;

   if (server_id >= 0) {
      buf = (String) alloca(strlen(txt)+10);
      sprintf(buf,"%c%s",REGISTER_CHAR,txt);
      send_to_server(buf);
    };
};





/************************************************************************/
/*									*/
/*	MSGfree -- unregister a message pattern 			*/
/*									*/
/************************************************************************/


void
MSGfree(txt,rtn)
   String txt;
   Function_Ptr rtn;
{
   String buf;
   MSG_PAT mp,mpl;

   mpl = NULL;
   for (mp = all_mpats; mp != NULL; mp = mp->next) {
      if (STREQL(txt,mp->text) && (rtn == NULL || rtn == mp->routine)) break;
      mpl = mp;
    };

   if (mp == NULL) return;

   PROTECT;
   if (mpl == NULL) all_mpats = mp->next;
   else mpl->next = mp->next;
   UNPROTECT;

   PMATfree_pattern(mp->pattern);
   SFREE(mp->text);

   free(mp);

   if (server_id >= 0) {
      buf = (String) alloca(strlen(txt)+10);
      sprintf(buf,"%c%s",FREE_CHAR,txt);
      send_to_server(buf);
    };
};





/************************************************************************/
/*									*/
/*	MSGsenda -- send a message with arguments			*/
/*	MSGsend -- send a message					*/
/*									*/
/************************************************************************/


/*VARARGS1*/

void
MSGsenda(txt,v1,v2,v3,v4,v5,v6,v7,v8,v9,v10,v11,v12)
   String txt;
   Universal v1,v2,v3,v4,v5,v6,v7,v8,v9,v10,v11,v12;
{
   Character buf[PRINT_SIZE],fmt[PRINT_SIZE];

   fix_format(txt,fmt);
   sprintf(buf,fmt,v1,v2,v3,v4,v5,v6,v7,v8,v9,v10,v11,v12);

   MSGsend(buf);
};





void
MSGsend(txt)
   String txt;
{
   if (!initialized) MSGinit(FALSE);

   if (server_id >= 0) {
      send_to_server(txt);
    }
   else {
      process_message(txt);
    };
};





/************************************************************************/
/*									*/
/*	MSGcalla -- send a message w/ args and await reply		*/
/*	MSGcall -- send a message and await reply			*/
/*									*/
/************************************************************************/


/*VARARGS1*/

String
MSGcalla(txt,v1,v2,v3,v4,v5,v6,v7,v8,v9,v10,v11,v12)
   String txt;
   Universal v1,v2,v3,v4,v5,v6,v7,v8,v9,v10,v11,v12;
{
   Character buf[PRINT_SIZE],fmt[PRINT_SIZE];
   String s;

   fix_format(txt,fmt);
   sprintf(buf,fmt,v1,v2,v3,v4,v5,v6,v7,v8,v9,v10,v11,v12);

   s = MSGcall(buf);

   return s;
};





String
MSGcall(txt)
   String txt;
{
   String buf;
   MSG_CALL mc,mca;
   String s;
   THREAD_MANAGER_BLOCK tmb;

   if (!initialized) MSGinit(FALSE);

   mc = PALLOC(MSG_CALL_INFO);

   if (call_mon != NULL) THREADmonitorentry(call_mon,&tmb);

   for ( ; ; ) {
      call_id = (call_id +1) % 32;
      for (mca = call_msgs; mca != NULL; mca = mca->next) {
	 if (mca->id == call_id + '@') break;
       };
      if (mca == NULL) break;
    };
   mc->id = call_id + '@';
   mc->rslt = NULL;
   mc->done = FALSE;
   mc->next = call_msgs;
   call_msgs = mc;

   if (call_mon != NULL) THREADmonitorexit(call_mon);

   buf = (String) alloca(strlen(txt)+10);

   strcpy(&buf[2],txt);
   buf[0] = CALL_CHAR;
   buf[1] = mc->id;

   if (server_id >= 0) {
      send_to_server(buf);
    }
   else {
      if (process_message(buf) == 0) mc->done = TRUE;
    };

   if (wait_fct == NULL) {
      return (String) mc;
    };

   if (call_mon != NULL) {
      THREADmonitorentry(call_mon,&tmb);
      while (!mc->done) {
	 THREADmonitorwait(call_mon,REPLY_SIG);
       };
      THREADmonitorsignalandexit(call_mon,REPLY_SIG);
    }
   else {
      while (!mc->done) {
	 (*wait_fct)(0);
       };
    };

   if (call_mon != NULL) THREADmonitorentry(call_mon,&tmb);
   if (call_msgs == mc) call_msgs = mc->next;
   else {
      for (mca = call_msgs; mca != NULL; mca = mca->next) {
	 if (mca->next == mc) {
	    mca->next = mc->next;
	    break;
	  };
       };
    };
   if (call_mon != NULL) THREADmonitorexit(call_mon);

   s = mc->rslt;
   free(mc);

   return s;
};





/************************************************************************/
/*									*/
/*	MSGcheck_call -- check if any calls are complete		*/
/*									*/
/************************************************************************/


String
MSGcheck_call(result)
   String * result;
{
   MSG_CALL mc,mca;
   THREAD_MANAGER_BLOCK tmb;

   if (wait_fct != NULL) {
      *result = NULL;
      return NULL;
    };

   for (mc = call_msgs; mc != NULL; mc = mc->next) {
      if (mc->done) break;
    };

   if (mc == NULL) {
      *result = NULL;
      return NULL;
    };

   if (call_mon != NULL) THREADmonitorentry(call_mon,&tmb);
   if (call_msgs == mc) call_msgs = mc->next;
   else {
      for (mca = call_msgs; mca != NULL; mca = mca->next) {
	 if (mca->next == mc) {
	    mca->next = mc->next;
	    break;
	  };
       };
    };
   if (call_mon != NULL) THREADmonitorexit(call_mon);

   *result = mc->rslt;
   free(mc);

   return (String) mc;
};





/************************************************************************/
/*									*/
/*	MSGreply -- reply to a message					*/
/*									*/
/************************************************************************/


void
MSGreply(id,txt)
   Integer id;
   String txt;
{
   String buf;

   if (txt == NULL) txt = "";

   buf = (String) alloca(strlen(txt)+10);
   sprintf(buf,"%c%c%s",REPLY_CHAR,id,txt);

   MSGsend(buf);
};





/************************************************************************/
/*									*/
/*	MSGwait_fct -- set wait function				*/
/*	MSGreigster_fct -- set register function			*/
/*									*/
/************************************************************************/


void
MSGwait_fct(fct)
   Function_Ptr fct;
{
   wait_fct = fct;
};





void
MSGregister_fct(fct)
   Function_Ptr fct;
{
   register_fct = fct;
};





/************************************************************************/
/*									*/
/*	MSGsync_mode -- turn sync_mode on or off			*/
/*	MSGsync_allow -- allow more messages right to be processed	*/
/*									*/
/************************************************************************/


void
MSGsync_mode(fg)
   Boolean fg;
{
   sync_mode = fg;
};




void
MSGsync_allow()
{
   sync_working = FALSE;
};





/************************************************************************/
/*									*/
/*	process_message -- handle message being received		*/
/*									*/
/************************************************************************/


static Integer
process_message(txt)
   String txt;
{
   MSG_PAT mp;
   Integer ct;
   Universal args[64];
   Integer replyid;
   MSG_CALL mc;
   THREAD_MANAGER_BLOCK tmb;
   Integer sendct;

   sendct = 0;

   if (txt[0] == REPLY_CHAR) {
      replyid = txt[1];
      txt = &txt[2];
      if (call_mon != NULL) THREADmonitorentry(call_mon,&tmb);
      for (mc = call_msgs; mc != NULL; mc = mc->next) {
	 if (mc->id == replyid) break;
       };
      if (mc != NULL) {
	 mc->rslt = SALLOC(txt);
	 mc->done = TRUE;
       };
      if (call_mon != NULL) THREADmonitorsignalandexit(call_mon,REPLY_SIG);
      return 1;
    };

   PROTECT;
   for (mp = all_mpats; mp != NULL; mp = mp->next) {
      if (txt[0] == CALL_CHAR) {
	 ct = PMATmatch(&txt[2],mp->pattern,args);
	 if (ct >= 0) args[ct++] = txt[1];
       }
      else {
	 ct = PMATmatch(txt,mp->pattern,args);
	 if (ct >= 0) args[ct++] = -1;
       };
      if (ct >= 0) {
	 sendct++;
	 switch (ct) {
	    case 0 :
	       (*(mp->routine))();
	       break;
	    case 1 :
	       (*(mp->routine))(args[0]);
	       break;
	    case 2 :
	       (*(mp->routine))(args[0],args[1]);
	       break;
	    case 3 :
	       (*(mp->routine))(args[0],args[1],args[2]);
	       break;
	    case 4 :
	       (*(mp->routine))(args[0],args[1],args[2],args[3]);
	       break;
	    case 5 :
	       (*(mp->routine))(args[0],args[1],args[2],args[3],args[4]);
	       break;
	    case 6 :
	       (*(mp->routine))(args[0],args[1],args[2],args[3],args[4],args[5]);
	       break;
	    case 7 :
	    case 8 :
	    case 9 :
	    case 10 :
	       (*(mp->routine))(args[0],args[1],args[2],args[3],args[4],args[5],
				   args[6],args[7],args[8],args[9]);
	       break;
	    default :
	       (*(mp->routine))(args[0],args[1],args[2],args[3],args[4],args[5],
				   args[6],args[7],args[8],args[9],
				   args[10],args[11],args[12],args[13],
				   args[14],args[15]);
	       break;
	  };
	 PMATfree_args(mp->pattern,args);
       };
    };
   UNPROTECT;

   return sendct;
};





/************************************************************************/
/*									*/
/*	open_channel -- open channel to message server			*/
/*									*/
/************************************************************************/


static Integer
open_channel(dbg)
   Boolean dbg;
{
   Character buf[64],fbuf[128],hbuf[64];
   struct sockaddr_in name;
   int i,j,s,fd,port;
   fd_set rmask;
   struct hostent *he;

   get_lock_name(buf);

   for (j = 0; j < 5; ++j) {
      fd = open(buf,O_RDWR,0);
#ifdef LOCKF_WORKS
      if (fd >= 0 && lockf(fd,F_TEST,0) == 0) {
#else
      if (fd >= 0 && lockf(fd,F_TLOCK,0) >= 0) {
	 lockf(fd,F_ULOCK,0);
#endif
	 close(fd);
	 fd = -1;
       };

      if (fd >= 0) {
	 i = read(fd,fbuf,128);
	 close(fd);
	 if (i > 0) i = sscanf(fbuf,"%s %d",hbuf,&port);
	 if (i != 2) fd = -1;
       };

      if (fd >= 0) {
	 bzero(&name,sizeof(name));
	 name.sin_family = AF_INET;
	 name.sin_port = port;
	 gethostname(fbuf,64);
	 if (STREQL(fbuf,hbuf)) name.sin_addr.s_addr = INADDR_LOOPBACK;
	 else {
	    he = gethostbyname(hbuf);
	    if (he == NULL) fd = -1;
	    else name.sin_addr.s_addr = * ((int *) he->h_addr);
	  };
       };

      if (fd >= 0) {
	 s = socket(AF_INET,SOCK_STREAM,0);
	 if (s < 0) {
	    fprintf(stderr,"MSG/C: Can't open socket -- %d\n",errno);
	    return -1;
	  };
	 while (connect(s,&name,sizeof(name)) < 0) {
	    if (errno == EINPROGRESS) continue;
	    fd = -1;
	    close(s);
	    if (errno != EADDRNOTAVAIL && errno != ECONNREFUSED) {
	       fprintf(stderr,"MSG/C: Can't connect socket -- %d\n",errno);
	       return -1;
	     };
	    break;
	  };
       };

      if (fd >= 0) break;
      if (j == 0) create_server(dbg);
      else sleep(1);
    };

   if (fd < 0) {
      fprintf(stderr,"MSG/C:  Couldn't create and connect to message server\n");
      exit(1);
    };

   i = fcntl(fd,F_GETFL,0);
   i |= FNDELAY;
   if (fcntl(fd,F_SETFL,i) == -1) {
      fprintf(stderr,"MSG/C: Problem with fcntl -- %d\n",errno);
      close(fd);
      return -1;
    };


   FD_ZERO(&rmask);
   FD_SET(s,&rmask);
   (*register_fct)(s+1,&rmask,NULL,NULL,receive_from_server);

   return s;
};





/************************************************************************/
/*									*/
/*	create_server -- create a server process			*/
/*									*/
/************************************************************************/


static void
create_server(dbg)
   Boolean dbg;
{
   Integer i,ln;
   Character buf[256];

   MSGcommand_name(SERVER_NAME,buf);

   i = vfork();

   if (i < 0) {
      fprintf(stderr,"MSG/C: Problem with vfork -- %d\n",errno);
      exit(1);
    };

   if (i > 0) {
      sleep(STARTUP_TIME);
      return;
    };

   ln = getdtablesize();
   for (i = 3; i < ln; ++i) close(i);
   signal(SIGINT,SIG_IGN);
   signal(SIGQUIT,SIG_IGN);
   signal(SIGPIPE,SIG_IGN);

   if (lock_file == NULL) {
      execl(buf,SERVER_ID,(dbg ? "-D" : 0),0);
    }
   else {
      execl(buf,SERVER_ID,"-msg",lock_file,(dbg ? "-D" : 0),0);
    };

   fprintf(stderr,"MSG/C: Problem exec server -- %d\n",errno);
   exit(1);
};





/************************************************************************/
/*									*/
/*	send_to_server -- send a message to the server			*/
/*									*/
/************************************************************************/


static void
send_to_server(umsg)
   String umsg;
{
   Integer ln,i;
   String msgb;
   String msg;

   msgb = (String) alloca(strlen(umsg)+10);

   sprintf(msgb,"%s%c",umsg,EOM_CHAR);
   msg = msgb;

   if (msg[0] == ABORT_CHAR) msg[1] = 0;

   ln = strlen(msg);

   for ( ; ; ) {
      i = send(server_id,msg,ln,0);
      if (i == ln) break;
      else if (i < 0 && errno == EINTR) ;
      else if (i < 0 && errno == EWOULDBLOCK) {
	 sleep(1);
       }
      else if (i > 0) {
	 sleep(1);
/*	 msg[ln-1] = 0; 	why was this here?? */
	 msg = &msg[i];
	 ln = strlen(msg);
       }
      else {
	 fprintf(stderr,"MSG/C:  Problem with send to server -- %d\n",errno);
	 exit(1);
       };
    };
};





/************************************************************************/
/*									*/
/*	receive_from_server -- get a message from the server		*/
/*									*/
/************************************************************************/


static void
receive_from_server()
{
   String  msg;
   Character buf[MSG_BUF_SIZE*2];

   if (sync_working && sync_mode) return;

   sync_working = TRUE;

   while ((msg = read_next_message(MSG_BUF_SIZE*2,buf)) != NULL) {
      process_message(msg);
      if (msg != buf) free(msg);
      sync_working = TRUE;
    };

   sync_working = FALSE;
};





/************************************************************************/
/*									*/
/*	read_next_message -- get next message from server		*/
/*									*/
/************************************************************************/


static String
read_next_message(mxlen,ubuf)
   Integer mxlen;
   String ubuf;
{
   String  msgbuf;
   Character buf[MSG_BUF_SIZE+2];
   String mpt,bpt;
   Integer ch,ln;

   PROTECT;

   if (next_read[0] != 0) {
      strcpy(buf,next_read);
      next_read[0] = 0;
      ln = strlen(buf);
    }
   else {
      ln = read(server_id,buf,MSG_BUF_SIZE);
    };

   if (ln < 0) {
      if (errno != EWOULDBLOCK && errno != EINTR) {
	 fprintf(stderr,"MSG/C:  Problem with read -- %d\n",errno);
	 exit(1);
       };
      UNPROTECT;
      return NULL;
    }
   else {
      buf[ln] = 0;
      bpt = buf;
    };

   ln = strlen(server_msg);
   if (MSG_BUF_SIZE+ln+10 > mxlen) {
      msgbuf = (String) malloc(MSG_BUF_SIZE+ln+10);
    }
   else {
      msgbuf = ubuf;
    };

   for ( ; ; ) {
      mpt = msgbuf;
      if (server_msg[0] != 0) {
	 strcpy(msgbuf,server_msg);
	 server_msg[0] = 0;
	 mpt = &msgbuf[strlen(msgbuf)];
       };

      while ((ch = *bpt++) != 0) {
	 if (ch == EOM_CHAR || ch == ABORT_CHAR) break;
	 *mpt++ = ch;
       };

      if (ch == EOM_CHAR) {
	 *mpt = 0;
	 if (mpt != msgbuf) {
	    strcpy(next_read,bpt);
	    UNPROTECT;
	    return msgbuf;
	  };
       }
      else if (ch == ABORT_CHAR) ;
      else if (mpt != msgbuf) { 		/* end of buffer	*/
	 *mpt = 0;
	 ln = strlen(msgbuf)+1;
	 if (ln >= server_msg_len) {
	    while (ln >= server_msg_len) server_msg_len *= 2;
	    server_msg = (String) realloc(server_msg,server_msg_len+2);
	  };
	 strcpy(server_msg,msgbuf);
	 if (msgbuf != ubuf) free(msgbuf);
	 break;
       }
      else break;
    };

   UNPROTECT;

   return NULL;
};





/************************************************************************/
/*									*/
/*	get_lock_name -- get name for lock file 			*/
/*									*/
/************************************************************************/


static void
get_lock_name(buf)
   Character buf[];
{
   Character nbuf[32];
   struct passwd * pwd;
   String s;

   if (lock_file != NULL) {
      strcpy(buf,lock_file);
      return;
    };

   gethostname(nbuf,32);
   pwd = getpwuid(getuid());
   if (pwd == NULL) s = (String) getenv("USER");
   else s = pwd->pw_name;
   if (s == NULL) s = "XXX";

   sprintf(buf,"/tmp/msg.%s.%s.addr",nbuf,s);
};





/************************************************************************/
/*									*/
/*	fix_format -- fix format string 				*/
/*									*/
/************************************************************************/


static void
fix_format(fmt,buf)
   String fmt;
   String buf;
{
   while (*fmt != 0) {
      if (*fmt == '%') {
	 if (fmt[1] == 'S') {
	    *buf++ = LIT_STRING;
	    *buf++ = *fmt++;
	    *buf++ = 's';
	    *buf++ = LIT_STRING;
	    ++fmt;
	  }
	 else {
	    *buf++ = *fmt++;
	    *buf++ = *fmt++;
	  }
       }
      else *buf++ = *fmt++;
    };

   *buf = 0;
};





/* end of msgclient.c */




